/* GraphContext.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred;

import java.util.Collection;
import java.util.HashSet;
import org.grinvin.Edge;
import org.grinvin.Element;
import org.grinvin.gui.SelectionModel;


/**
 * Keeps track of the following extra information about graph elements:
 * <ul>
 * <li>Whether an element is currently selected.</li>
 * <li>Whether an element is currently highlighted.</li>
 * <li>What is the current 'roll over' element.</li>
 * </ul><p>
 * Other classes may be registered as observers for this class
 * and will then be notified of any changes.
 */
public class GraphContext extends SelectionModel<Element> {
    
    /** Default constructor */
    public GraphContext() {
        rollOver = null;
    }
    
    /* ============================================================
     * ROLL OVER
     * ============================================================ */
    
    // current 'roll over'-element
    private Element rollOver;
    
    /**
     * Get the current 'roll over' element.
     * @return The current roll over element, or {@code null}
     * when there is none.
     */
    public Element getRollOver() {
        return this.rollOver;
    }
    
    /**
     * Set the current 'roll over' element.
     * @param rollOver new 'roll over' element.
     */
    public void setRollOver(Element rollOver) {
        if (this.rollOver != rollOver) {
            this.rollOver = rollOver;
            notifyRollOverChanged();
        }
    }
    
    /**
     * Is the given element the current 'roll over' element?
     */
    public boolean isRollOver (Element el) {
        return el == rollOver;
    }
    
    /* ============================================================
     * SELECTION
     * ============================================================ */
    
    /**
     * Extend the current selection by adding all end points of
     * currently selected edges.
     */
    public void extendEdgeSelection () {
        Collection<Element> vertices = new HashSet<Element> ();
        for (Element el: selection)
            if (el instanceof Edge) {
                vertices.add (((Edge)el).getFirstEndpoint ());
                vertices.add (((Edge)el).getSecondEndpoint ());
            }
        addSelection (vertices);
    }
    
     
    /* ============================================================
     * OBSERVABLE
     * ============================================================ */
    
    /**
     * Inform all observers that the 'roll over' element was changed.
     */
    protected void notifyRollOverChanged() {
        setChanged();
        notifyObservers();
    }
    
    /**
     * Inform all observers that there was a major change to
     * the current selection.
     */
    protected void notifySelectionChanged() {
        setChanged();
        notifyObservers();
    }
    
    /**
     * Inform all observers that the selection status of the
     * given element was changed.
     */
    protected void notifySelectionChanged(Element element) {
        setChanged();
        notifySelectionChanged();
    }
    
    /**
     * Inform all observers that there was a major change to
     * the current highlight set.
     */
    protected void notifyHighlightChanged() {
        setChanged();
        notifyObservers();
    }
    
    /**
     * Inform all observers that the highlight status of the
     * given element was changed.
     */
    protected void notifyHighlightChanged(Element element) {
        setChanged();
        notifyHighlightChanged();
    }
    
}
