/* DefaultInvariantFactory.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants;

import java.text.MessageFormat;

import org.grinvin.factories.AbstractFactory;
import org.grinvin.factories.FactoryParameterException;
import org.grinvin.io.IOFormatException;
import org.grinvin.params.ParameterList;

import org.jdom.Element;

/**
 * Default implementation of {@link InvariantFactory}.<p>
 */
class DefaultInvariantFactory extends AbstractFactory implements InvariantFactory {
    //TODO: implement VisualFactory
    
    //
    private class InvariantPrototype extends DefaultInvariant {
        
        //
        protected String getRootElement() {
            return "invariant-factory";
        }
        
        //
        public InvariantPrototype(String id, String path, ClassLoader classLoader) throws
                UnknownInvariantException {
            super(id, path, classLoader);
        }
        
        // implements Invariant
        public String getName() {
            return iprops.getProperty("invariant.name");
        }
        
        /**
         * Overridden to also load parameter information for the factory.
         */
        protected void init(Element root, String id) throws
                IOFormatException, UnknownInvariantTypeException {
            super.init(root, id);
            fromElement(root);
            // TODO: there must be a better way to accomplish this
        }
    }
    
    //
    private static class InvariantInstance extends Invariant {
        
        InvariantInstance() {
            // avoids creation of access type
        }
        
        //
        public String id;
        
        //
        public String name;
        
        //
        public Class type;
        
        public String getId() {
            return this.id;
        }
        
        public String getName() {
            return this.name;
        }
        
        public String toString() {
            return getName();
        }
        
        public Class getType() {
            return type;
        }
    }
    
    //
    private final InvariantPrototype prototype;
    
    /**
     * Return the path which contains the definition file for this factory.
     */
    public String getPath () {
        return prototype.getPath();
    }
    
    /**
     * Create a default invariant factory with given generic identifier.
     * Information about this invariant is loaded from
     * an XML-file in the class path, using standard naming conventions
     * for a class but with extension .xml instead of .class
     *
     * @param id Identifier of this invariant
     * @param class loader to be used for loading the XML file
     */
    public DefaultInvariantFactory(String id, String path, ClassLoader classLoader) throws
            UnknownInvariantException {
        prototype = new InvariantPrototype(id, path, classLoader);
    }
    
    public Invariant createInvariant() throws FactoryParameterException {
        InvariantInstance invariant = new InvariantInstance();
        invariant.id = getInvariantId();
        String nameFormat = prototype.iprops.getProperty("invariant.name");
        invariant.name = MessageFormat.format(nameFormat, (Object[])valueStrings);
        invariant.type = prototype.getType();
        return invariant;
    }
    
    //
    public void accept(Visitor visitor) {
        visitor.visitFactory(this);
    }

    public String getInvariantId() throws FactoryParameterException {
        return getId() + getParametersAsString();
    }
    
    public String getId() {
        return prototype.id;
    }
    
    public String getName() {
        return prototype.iprops.getProperty("factory.name");
    }
    
    // implements InvariantNode
    public Iterable<InvariantNode> getChildren() {
        return null;
    }
    
    public String toString() {
        return getName();
    }
    
    /**
     * Load extra information from the XML-element for the prototype
     * which is relevant for this factory.
     */
    protected void fromElement(Element root) throws IOFormatException {
        list = new ParameterList();
        list.fromElement(root);
    }
    
}
