/* InvariantTypes.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants;

import java.util.HashMap;
import java.util.Map;

/**
 * Helper class which handles invariant types.
 */
public final class InvariantTypes {
    
    // do not instantiate
    private InvariantTypes () {
    }
    
    private static Map<String,Class<? extends InvariantValue>> TYPES;
    
    static {
        TYPES = new HashMap<String,Class<? extends InvariantValue>> ();
        TYPES.put ("integer", IntegerValue.class);
        TYPES.put ("real", RealValue.class);
        TYPES.put ("boolean", BooleanValue.class);
    }
    
    //
    private static final Class<InvariantValue> INVARIANT_VALUE = InvariantValue.class;
    
    /**
     * Return the class for the given type.
     * @throws UnknownInvariantTypeException when the type string could not be converted
     * to a class for some reason.
     */
    public static Class<? extends InvariantValue> classForType (String type) throws
            UnknownInvariantTypeException {
        
        // well-known type?
        Class<? extends InvariantValue> result = TYPES.get (type);
        if (result != null)
            return result;
        
        // use type as class name
        try {
            result = (Class<? extends InvariantValue>)Class.forName (type);
            if (INVARIANT_VALUE.isAssignableFrom (result))
                return result;
            else
                throw new UnknownInvariantTypeException (
                        "Invariant type not of type InvariantValue", type);
        } catch (ClassNotFoundException ex) {
            throw new UnknownInvariantTypeException (
                    "Unknown invariant type", type, ex);
        }
    }
    
    
}
