/* GraphListLoader.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantManager;
import org.grinvin.invariants.UnknownInvariantException;
import org.grinvin.list.GraphInvariantList;
import org.grinvin.list.GraphList;
import org.grinvin.list.GraphListElement;
import org.grinvin.list.GraphListElementManager;
import org.grinvin.list.InvariantList;

import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

/**
 * Helper methods for loading lists of graphs and lists of invariants from
 * disk.
 */
public final class GraphListLoader {
    
    // should not be instantiated
    private GraphListLoader () { }
    
    private static final Logger LOGGER = Logger.getLogger("org.grinvin.io");
    
    private static void loadInvariants(InvariantList list, Element element) {
        
        for (Object obj: element.getChildren("invariant")) {
            try {
                Element child = (Element)obj;
                Invariant inv = InvariantManager.getInstance().getInvariant(child.getAttributeValue("invariantId"));
                list.add(inv);
            } catch (UnknownInvariantException ex) {
                LOGGER.log(Level.WARNING, "Ignored unknown invariant {0}", ex.getId());
                // ignore unknown invariants
            }
            
        }
        
    }
    
    private static void loadGraphListElements(GraphList list, Element element, File directory) throws IOFormatException {
        for (Object obj : element.getChildren("graphlistelement")) {
            try {
                Element child = (Element)obj;
                URI uri = new URI(child.getAttributeValue("graphURI"));
                GraphListElement graphlistelement 
                        = GraphListElementManager.getInstance().getGraphListElement(uri, directory);
                list.add(graphlistelement);
            } catch (NullPointerException ex) {
                // new URI(null) was called
                throw new IOFormatException("invalid graphURI");
            } catch (URISyntaxException ex) {
                throw new IOFormatException("invalid graphURI");
            }
        }
        
    }
        

    /**
     * Load a list of invariants from an XML file. The invariants are added to
     * the given list, so it may be necessary to clear the list before calling
     * this method. 
     * @see GraphListSaver#save(InvariantList,File)
     */
    public static void load(InvariantList list, File file) throws IOException {
        
        try {
            InputStream input = new FileInputStream(file);
            Element element = new SAXBuilder().build(input).getRootElement();
            if (!"invariants".equals(element.getName()))
                throw new IOFormatException("Expected 'invariants' element");
            loadInvariants(list, element);
        } catch (JDOMException ex) {
            throw new IOFormatException("XML parse error", ex);
        }
    }
    
    /**
     * Load a graph list from an XML file.
     * @param list List to which the graphs are added. It may be necessary to clear the list before calling
     * this method.
     * @param ilist Optional list of invariants. If non-null, any invariants stored in the XML-file will be
     * added to this list. It may be necessary to clear the list before calling
     * this method.
     * @param file XML file from which the list(s) will be loaded.
     * @param directory directory containing sssion graphs, or null when
     * session graphs are not supported
     * @see GraphListSaver#save(GraphList,InvariantList,File)
     */
    public static void load(GraphList list, InvariantList ilist, 
            File file, File directory) throws IOException {
        try {
            InputStream input = new FileInputStream(file);
            Element element = new SAXBuilder().build(input).getRootElement();
            if (!"graphlistelements".equals(element.getName()))
                throw new IOFormatException("Expected 'invariants' element");
            list.setName (element.getAttributeValue ("graphlistName"));
            loadGraphListElements (list, element, directory);
            list.setDirty(false);
            if (ilist != null) {
                loadInvariants(ilist, element);
                ilist.setDirty(false);
            }
        } catch (JDOMException ex) {
            throw new IOFormatException("XML parse error", ex);
        }
    }
    
    /**
     * Convenience method which loads a graph list of type 
     * {@link GraphInvariantList}. Calls {@link #load(GraphList,InvariantList,File,File)}
     * with the appropriate arguments.
     */
    public static void load (GraphInvariantList list, File file, File directory) throws IOException {
        load (list.getGraphList(), list.getInvariantList(), file, directory);
    }
}
