/* GraphLoader.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io;

import java.io.IOException;
import java.io.InputStream;
import org.grinvin.Graph;
import org.grinvin.Vertex;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

/**
 * Loads a <tt>graph.xml</tt>-section from an input stream.
 */
public class GraphLoader {
    
    //
    private Graph graph;
    
    /**
     * Creates a graph loader.
     * @param graph Graph object which shall hold the resulting graph.
     */
    public GraphLoader (Graph graph) {
        this.graph = graph;
    }
    
    /**
     * Construct the abstract graph from the given JDOM element
     * and fill the translation table.
     */
    public void toGraph(Element element) throws IOFormatException {
        if (! "graph".equals(element.getName()))
            throw new IOFormatException("Expected <graph> element");
        int numberOfVertices = Integer.parseInt(element.getAttributeValue("nrofvertices"));

        // annotations
        String[] annotations = new String[numberOfVertices];
        for (Object obj: element.getChildren("vertex")) {
            Element v = (Element)obj;
            int id = Integer.parseInt(v.getAttributeValue("id"));
            annotations[id] = v.getAttributeValue("annotation");
        }
        
        // vertices
        Vertex[] vertices = new Vertex[numberOfVertices];
        for (int i=0; i < numberOfVertices; i++)
            vertices[i] = graph.addNewVertex(annotations[i]);
        
        // edges
        for (Object obj: element.getChildren("edge")) {
            Element e = (Element)obj;
            int first = Integer.parseInt(e.getAttributeValue("from"));
            int second = Integer.parseInt(e.getAttributeValue("to"));
            graph.addNewEdge(vertices[first], vertices[second],e.getAttributeValue("annotation"));
        }
    }
     
    /**
     * Load the graph from the input stream. After the call the input stream
     * should no longer be used.
     */
    public void load (InputStream input) throws IOException {
        try {
            Document document = new SAXBuilder().build(input);
            toGraph (document.getRootElement()); // probably closes the stream
        } catch (JDOMException ex) {
            throw new IOFormatException ("Invalid input format", ex);
        }
    }
    
}
