/* ZipInputStreamSectionLoader.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

/**
 * Loads sections from a {@link ZipInputStream}. Note that each section
 * on the input strean can be accessed at most once.
 */
class ZipInputStreamSectionLoader implements SectionLoader {
    
    //
    private ZipInputStream in;
    
    //
    private byte[] buffer;
    
    //
    private int offset;
    
    //
    private static final int DEFAULT_SIZE = 4096;
    
    //
    private static final double GROWTH_FACTOR = 1.5;
    
    //
    private void checkBufferSize (int pos) {
        if (pos >= buffer.length) {
            int newSize = (int)(buffer.length * GROWTH_FACTOR);
            byte[] newBuffer = new byte[newSize];
            System.arraycopy (buffer, 0, newBuffer, 0, buffer.length);
            buffer = newBuffer;
        }
    }
    
    /**
     * Maps section names to offset/length-pairs
     */
    private Map<String,int[]> map;
    
    /**
     * Default constructor.
     */
    public ZipInputStreamSectionLoader(ZipInputStream in) {
        this.in = in;
        this.buffer = new byte[DEFAULT_SIZE];
        this.offset = 0;
        this.map = new HashMap<String,int[]>();
    }
    
    /**
     * Read the next section from the inputstream, add it to the map and
     * return the name of the section read. Returns {@code null} when
     * the input stream has no more sections.
     */
    private String readNextSection () throws IOException {
        ZipEntry entry = in.getNextEntry();
        if (entry == null)
            return null;
        String name = entry.getName();
        
        // read the entire entry
        int pos = offset;
        checkBufferSize (pos);
        int size = in.read (buffer, pos, buffer.length - pos);
        while (size > 0) {
            pos += size;
            checkBufferSize (pos);
            size = in.read (buffer, pos, buffer.length - pos);
        }
        in.closeEntry ();
        
        map.put (name, new int[] { offset, pos - offset});
        offset = pos;
        
        return name;
    }
    
    
    //
    public InputStream openSection(String name) throws IOException{
        if (! map.containsKey(name)) {
            String section = readNextSection ();
            while (section != null && ! section.equals (name))
                section = readNextSection ();
            if (section == null)
                return null;
        }
        int[] info = map.get(name);
        InputStream stream =  new ByteArrayInputStream (buffer, info[0], info[1]);
        return stream;
    }
}
