/* GraphListElementManager.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list;

import java.io.File;
import java.net.URI;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.grinvin.DefaultGraphBundle;
import org.grinvin.GraphBundle;
import org.grinvin.GraphURI;
import org.grinvin.GraphURIException;
import org.grinvin.util.InternationalizedProperties;

/**
 * Manages the mapping between Graph URI's and the GraphListElement objects representing them.
 */
public class GraphListElementManager {
    
    //
    private static GraphListElementManager SINGLETON = new GraphListElementManager();

    /**
     * Maps a Graph uri to the corresponding object of type GraphListElement.
     */
    private Map<URI, GraphListElement> graphlistelements;
    
    //
    private GraphListElementManager() {
        graphlistelements = new HashMap<URI, GraphListElement>();
    }

    /**
     * Get the singleton instance of this class
     * @return the single shared instance of this class
     */
    public static GraphListElementManager getInstance() {
        return SINGLETON;
    }
    
    //
    private void registerGraphListElement(GraphListElement element) {
        graphlistelements.put(element.getURI(), element);
    }
    
    /**
     * Get the {@link GraphListElement} for the given {@link URI}.
     * @param directory Base directory to be used for (relative) session URIs, or null
     * if session URIs are not allowed
     * @return the graph list element
     */
    public GraphListElement getGraphListElement(URI uri, File directory) {
        GraphListElement element = graphlistelements.get(uri);
        if (element == null) {
            GraphBundle bundle = new DefaultGraphBundle ();
            try {
                if (directory != null)
                    GraphURI.load(GraphURI.translateSessionURI(uri, directory), bundle);
                else
                    GraphURI.load(uri, bundle);
            } catch (GraphURIException ex) {
                Logger.getLogger("org.grinvin.io", "org.grinvin.grinvin.resources").log(Level.WARNING, "Failed to load graph from " + uri, ex);
                createMissingBundle(bundle);
            }
            element = new GraphListElement(uri, bundle);
        }
        registerGraphListElement(element);
        return element;
    }
    
    /**
     * Convenience method for retrieving a graph list element with given
     * URI. Same as <tt>getGraphListElement(uri,null)</tt>.
     */
    public GraphListElement getGraphListElement(URI uri) {
        return getGraphListElement (uri,null);
    }
    
    //
    public GraphListElement createGraphListElement(GraphBundle bundle) {
        return new GraphListElement(bundle);
    }
     
    //
    private void createMissingBundle(GraphBundle bundle) {
        InternationalizedProperties properties = new InternationalizedProperties();
        properties.setProperty("graph.name", "???");
        bundle.setProperties(properties);
    }

}
