/* GrinvinPreferences.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.preferences;

import com.apple.eio.FileManager;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.prefs.Preferences;

import org.grinvin.io.IOFormatException;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

/**
 * Class to handle GrInvIn preferences management. When the preferences are not set
 * by the user, a default value is taken from the defaults.xml file.
 */
public class GrinvinPreferences {
    
    //
    public enum OsType {
        LINUX,
        MAC_OS_X,
        WINDOWS,
        DEFAULT;
    }
    
    //
    public enum Preference {
        GRAPHICON_HEIGHT("gui.graphIconHeight"),
        GRAPHICON_WIDTH("gui.graphIconWidth"),
        WORKSPACE_LOCATION("workspace.location"),
        XMLOUTPUT_FORMAT("io.xmlOutputFormat"),
        GRINVIN_BIN_DIR("config.bindir"),
        GRINVIN_CACHE_DIR("config.cachedir");
        
        private String id;
        
        Preference(String id) {
            this.id = id;
        }
        
        protected String getId() {
            return id;
        }
    }
    
    //
    private String ourNodeName = "/org/grinvin/preferences";
    
    //
    private Preferences userPreferences = Preferences.userRoot().node(ourNodeName);
    
    //
    private Preferences systemPreferences;
    
    //
    private Element defaultsElement;
    
    //
    private List<GrinvinPreferencesListener> listeners;
    
    //
    private OsType osType;
    
    //
    public static GrinvinPreferences INSTANCE = new GrinvinPreferences();
    
    //
    private GrinvinPreferences() {
        if(System.getProperty("os.name").toLowerCase().startsWith("mac os x")) {
            osType = OsType.MAC_OS_X;
        } else if (System.getProperty("os.name").startsWith("Linux")) {
            osType = OsType.LINUX;
        } else if (System.getProperty("os.name").toLowerCase().startsWith("windows")) {
            osType = OsType.WINDOWS;
            systemPreferences = Preferences.systemRoot().node(ourNodeName);
        } else {
            osType = OsType.DEFAULT;
        }
        listeners = new ArrayList<GrinvinPreferencesListener>();
        try {
            InputStream stream = GrinvinPreferences.class.getResourceAsStream("defaults.xml");
            Document document = new SAXBuilder().build(stream);
            Element element = document.getRootElement();
            if (! "defaults".equals(element.getName()))
                throw new IOFormatException("Expected <defaults> element");
            else
                defaultsElement = element;
        } catch (IOException e) {
            throw new RuntimeException("Could not create preferences manager", e);
        } catch (JDOMException e) {
            throw new RuntimeException("Could not create preferences manager", e);
        }
    }
    
    private Element getPreferenceElement(String id, String type) {
        for (Element el : (List<Element>)defaultsElement.getChildren("preference"))
            if (id.equals(el.getAttribute("id").getValue()) && type.equals(el.getAttribute("type").getValue()))
                return el;
        return null;
    }
    
    //
    public int getIntPreference(Preference key) {
        Element child = getPreferenceElement(key.getId(), "int");
        if (child != null) {
            int defaultValue = Integer.parseInt(child.getValue());
            return userPreferences.getInt(key.getId(), defaultValue);
        } else {
            throw new RuntimeException("No default value available for: " + key);
        }
    }
    
    //
    public void setIntPreference(Preference key, int value) {
        userPreferences.putInt(key.getId(), value);
        firePreferenceChanged(key);
    }
    
    //
    public String getStringPreference(Preference key) {
        Element child = getPreferenceElement(key.getId(), "string");
        if (child != null) {
            String defaultValue = child.getValue();
            return userPreferences.get(key.getId(), defaultValue);
        } else if (key == Preference.GRINVIN_BIN_DIR) {
            String defaultValue;
            if (osType == OsType.MAC_OS_X) {
                try {
                    defaultValue = FileManager.getResource("benzene");
                    defaultValue = defaultValue.substring(0,defaultValue.lastIndexOf('/'));
                } catch (FileNotFoundException ex) {
                    defaultValue = null;
                }
            } else if (osType == OsType.WINDOWS) {
                assert systemPreferences != null : "System preferences not available";
                defaultValue = systemPreferences.get(key.getId(), null);
            } else {
                defaultValue = System.getProperty("grinvin.dir") + "/bin";
            }
            return userPreferences.get(key.getId(), defaultValue);
        } else if (key == Preference.GRINVIN_CACHE_DIR) {
            String defaultValue;
            if (osType == OsType.LINUX) {
                defaultValue = System.getProperty("user.home") + "/.grinvin/cache";
            } else if (osType == OsType.MAC_OS_X) {
                defaultValue = System.getProperty("user.home") + "/Library/Caches/GrInvIn";
            } else if (osType == OsType.WINDOWS) {
                defaultValue = System.getenv("APPDATA") + "\\GrInvIn\\Cache";
            } else {
                defaultValue = System.getProperty("user.dir") + "/cache";
            }
            return userPreferences.get(key.getId(), defaultValue);
        } else {
            throw new RuntimeException("No default value available for: " + key);
        }
    }
    
    //
    public void setStringPreference(Preference key, String value) {
        userPreferences.put(key.getId(), value);
        firePreferenceChanged(key);
    }
    
    public OsType getOsType() {
        return osType;
    }
    
    //
    public void addListener(GrinvinPreferencesListener listener) {
        listeners.add(listener);
    }
    
    //
    private void firePreferenceChanged(Preference preference) {
        for(GrinvinPreferencesListener listener : listeners)
            listener.preferenceChanged(preference);
    }
    
    /**
     * Return the path of an external program which is part of the grinvin
     * distribution. Prepends the name of the bin-directory in the installation,
     * and appends '.exe' if necessary.
     * @param name Canonical name of the program.
     */
    public File getProgramPath(String name) {
        if (osType == OsType.WINDOWS)
            name += ".exe";
        File path = new File(getStringPreference(Preference.GRINVIN_BIN_DIR), name);
        if (!path.exists() || !path.isFile())
            path = new File(name);
        return path;
    }
    
}
