/* Tridiagonal.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.util;

/**
 * This class represents a symmetric tridiagonal real matrix. The main purpose
 * of this class is to provide information related to eigenvalues and
 * eigenvectors of such matrices.
 */
public class Tridiagonal {

    /** Order of this matrix. */
    protected int order;

    /**
     * Diagonal elements of this matrix. Element d[i] contains the matrix
     * element at position (i,i).
     */
    protected double[] d;

    /**
     * Subdiagonal elements of this matrix. Element e[i] contains the matrix
     * element at position (i,i-1). Element e[0] is set to zero.
     */
    protected double[] e;

    /**
     * Create a tridiagonal matrix which is equivalent to the given symmetric
     * real matrix a by some orthogonal transformation. Uses Householder
     * reduction.
     * 
     * <p>
     * <b>Note:</b> This implementation is almost a literal copy of the
     * algorithm <code>tred2</code> from <i>Numerical recipes in C</i>, ISBN
     * 0-521-43108-5.
     * </p>
     */
    public Tridiagonal (double[][] a) {

        // init variables
        order = a.length;
        d = new double[order];
        e = new double[order];

        // make copy of a
        double[][] acopy = new double[order][];
        for (int i = 0; i < order; i++)
            acopy[i] = a[i].clone ();

        // reduce to a tridiagonal matrix
        householder (acopy);
        e[0] = 0.0;
        for (int i = 0; i < order; i++)
            d[i] = acopy[i][i];
    }

    /**
     * Auxiliary routine which performs the Householder reduction. Destroys the
     * contents of array a.
     *
     * @param vectors If true, the orthogonal transformation Q such that
     *        Q<sup>T</sup>.A.Q is tridiagonal, is stored into parameter a as
     *        the result of this algorithm. Otherwise the elements of the
     *        matrix a are destroyed and contain no meaningful information.
     */
    private void householder (double[][] a) {
        for (int i = order - 1; i > 0; i--) {
            int l = i - 1;
            double h = 0.0;
            double scale = 0.0;
            if (l > 0) {
                for (int k = 0; k <= l; k++)
                    if (a[i][k] < 0)
                        scale -= a[i][k];
                    else
                        scale += a[i][k];
                if (scale == 0.0)
                    e[i] = a[i][l];
                else {
                    for (int k = 0; k <= l; k++) {
                        a[i][k] /= scale;
                        h += a[i][k] * a[i][k];
                    }
                    double f = a[i][l];
                    double g = f >= 0.0 ? -Math.sqrt (h) : Math.sqrt (h);
                    e[i] = scale * g;
                    h -= f * g;
                    a[i][l] = f - g;
                    f = 0.0;
                    for (int j = 0; j <= l; j++) {
                        a[j][i] = a[i][j] / h;  // [*]
                        g = 0.0;
                        for (int k = 0; k <= j; k++)
                            g += a[j][k] * a[i][k];
                        for (int k = j + 1; k <= l; k++)
                            g += a[k][j] * a[i][k];
                        e[j] = g / h;
                        f += e[j] * a[i][j];
                    }
                    double hh = f / (h + h);
                    for (int j = 0; j <= l; j++) {
                        f = a[i][j];
                        g = e[j] - hh * f;
                        e[j] = g;
                        for (int k = 0; k <= j; k++)
                            a[j][k] -= f * e[k] + g * a[i][k];
                    }
                }
            } else
                e[i] = a[i][l];
            d[i] = h;
        }
    }

    /**
     * Compute the eigenvalues of this symmetric tridiagonal matrix using the
     * QL algorithm with implicit shifts.
     * 
     * <p>
     * <b>Note:</b> This implementation is almost a literal copy of the
     * algorithm <code>tqli</code> from <i>Numerical recipes in C</i>, ISBN
     * 0-521-43108-5.
     * </p>
     */
    public double[] eigenvalues () {
        return eigen (null);
    }

    /**
     * Auxiliary routine which performs the QL algorithm. Returns the
     * eigenvalues of the tridiagonal matrix, and if z is not null, stores the
     * corresponding eigenvectors into z, after applying the orthogonal
     * transformation originally contained in z.
     */
    private double[] eigen (double[][] z) {
        double[] d = this.d.clone ();
        double[] e = new double[order];
        for (int i = 1; i < order; i++)
            e[i - 1] = this.e[i];
        e[order - 1] = 0.0;
        for (int l = 0; l < order; l++) {
            int iter = 0;
            int m;
            do {
                for (m = l; m < order - 1; m++) {
                    double dd = Math.abs (d[m]) + Math.abs (d[m + 1]);
                    if (Math.abs (e[m]) + dd == dd)
                        break;
                }
                if (m != l) {
                    iter++;
                    if (iter == 30)
                        throw new RuntimeException(
                            "Too many iterations in QL algorithm ");
                    double g = (d[l + 1] - d[l]) / (2.0 * e[l]);
                    double r = Math.sqrt (1.0 + g * g);  //[!] should use pythag?
                    if (g >= 0.0)  //[!] equal sign correct ?

                        g = d[m] - d[l] + e[l] / (g + r);
                    else if (g < 0.0)
                        g = d[m] - d[l] + e[l] / (g - r);
                    double s = 1.0;
                    double c = 1.0;
                    double p = 0.0;
                    int i;
                    for (i = m - 1; i >= l; i--) {
                        double f = s * e[i];
                        double b = c * e[i];
                        r = Math.sqrt (f * f + g * g);  //[!] should use pythag?
                        e[i + 1] = r;
                        if (r == 0.0) {
                            d[i + 1] -= p;
                            e[m] = 0.0;
                            break;
                        }
                        s = f / r;
                        c = g / r;
                        g = d[i + 1] - p;
                        r = (d[i] - g) * s + 2.0 * c * b;
                        p = s * r;
                        d[i + 1] = g + p;
                        g = c * r - b;
                        if (z != null)
                            for (int k = 0; k < order; k++) {
                                f = z[k][i + 1];
                                z[k][i + 1] = s * z[k][i] + c * f;
                                z[k][i] = c * z[k][i] - s * f;
                            }
                    }
                    if (r != 0.0 || i < l) {
                        d[l] -= p;
                        e[l] = g;
                        e[m] = 0.0;
                    }
                }
            } while (m != l);
        }
        return d;
    }
}
