/* ConjecturingWindowWrapper.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import org.grinvin.graphbrowser.ConjecturingWindow;
import org.grinvin.io.GraphListLoader;
import org.grinvin.io.GraphListSaver;
import org.grinvin.io.IOFormatException;
import org.grinvin.list.DefaultGraphInvariantListModel;
import org.grinvin.list.GraphInvariantListModel;
import org.grinvin.list.GraphList;
import org.grinvin.list.GraphListElement;

import org.jdom.Element;

/**
 * Wrapper for the conjecturing window.
 */
public class ConjecturingWindowWrapper 
        extends DefaultWindowWrapper<ConjecturingWindow> {
    
    // TODO: factor out common code with GraphBrowserWindow
    // This is a literal copy with GraphBrowserWindow textually replaced
    // by ConjecturingWindow. This will require both to satisfy a
    // common interface, and the common super class to be generic. Also
    // the element name is different - of course
    
    @Override
    public ConjecturingWindow create() {
        return new ConjecturingWindow(new DefaultGraphInvariantListModel());
    }
    
    public String getElementName() {
         return "conjecturingWindow";
    }

    /**
     * Load the window, its graph list and its invariant list.
     */
    @Override
    public ConjecturingWindow load(Element element, File directory) throws IOException {
        try {
            // Load the corresponding graph list
            URI uri = new URI(element.getAttributeValue("graphlistURI"));
            GraphInvariantListModel list = new DefaultGraphInvariantListModel();
            File file = new File(directory, uri.getSchemeSpecificPart());
            GraphListLoader.load(list, file, directory);
            list.setURI(uri);
            
            // create and initialize the window
            ConjecturingWindow window = new ConjecturingWindow(list);
            
            // initialize properties of this window and of its guides tool
            loadWindowProperties(element, window);
            window.setCurrentComponent(element.getAttributeValue("component"));
            
            // set divider locations
            window.setDividerLocation(Integer.parseInt(element.getAttributeValue("dividerlocation")!=null ? element.getAttributeValue("dividerlocation"):"-1"));
            window.setInvariantTableDividerLocation(Integer.parseInt(element.getAttributeValue("invarianttabledividerlocation")!=null ? element.getAttributeValue("invarianttabledividerlocation"):"-1"));
            window.setGraphListDividerLocation(Integer.parseInt(element.getAttributeValue("graphlistdividerlocation")!=null ? element.getAttributeValue("graphlistdividerlocation"):"-1"));
            window.setGraphPropertiesComponentDividerLocation(Integer.parseInt(element.getAttributeValue("graphpropertiescomponentdividerlocation")!=null ? element.getAttributeValue("graphpropertiescomponentdividerlocation"):"-1"));

            return window;
        } catch (URISyntaxException ex) {
            throw new IOFormatException("Invalid URI in meta.xml", ex);
        }
    }
    /**
     * Save the window, its graph list and its list of invariants.
     */
    @Override
    public void save(ConjecturingWindow window, Element parent, 
            File directory) throws IOException {
        // save properties of window
        Element result = baseElement ();
        saveWindowProperties (result, window);
        result.setAttribute("component", window.getCurrentComponent());
               
        // save graph list (including contents)
        GraphList list = window.getGraphList();
        for (GraphListElement gle: list)
            GraphListSaver.saveIntoWorkspace(gle, directory);
        GraphListSaver.saveIntoWorkspace(list, 
                window.getInvariantList(), directory);
        result.setAttribute("graphlistURI", list.getURI().toString());
        
        // save divider locations
        result.setAttribute("dividerlocation", window.getDividerLocation() + "");
        result.setAttribute("invarianttabledividerlocation", window.getInvariantTableDividerLocation() + "");
        result.setAttribute("graphlistdividerlocation", window.getGraphListDividerLocation() + "");
        result.setAttribute("graphpropertiescomponentdividerlocation", window.getGraphPropertiesComponentDividerLocation() + "");
        
        
        parent.addContent(result);
     
    }
}
