/* MoharIndex2.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.chemical;

import org.grinvin.GraphBundleView;
import org.grinvin.invariants.RealValue;
import org.grinvin.invariants.computers.AbstractInvariantComputer;
import org.grinvin.util.Tridiagonal;

/**
 * Invariant computer which returns the second Mohar index.
 */
public class MoharIndex2 extends AbstractInvariantComputer {
    
    public RealValue compute (GraphBundleView bundle) {
        int[][] adjList = bundle.adjacencyList();
        int n = adjList.length;
        boolean[][] adjMatrix = bundle.booleanAdjacencyMatrix();
        double[][] laplacianMatrix = new double[n][n];
        for(int i=0; i<n; i++)
            for(int j=0; j<n; j++)
                if(i==j)
                    laplacianMatrix[i][j]=adjList[i].length;
                else
                    laplacianMatrix[i][j]=adjMatrix[i][j] ? -1 : 0;
        Tridiagonal decomp = new Tridiagonal(laplacianMatrix);
        double[] eigenValues;
        try {
            eigenValues = decomp.eigenvalues();
        } catch(RuntimeException ex){
            eigenValues = new double[0];
        }
        
        //sort
        for(int i=0; i < eigenValues.length; i++)
            for(int j=eigenValues.length-1; j>i; j--)
                if(eigenValues[j] > eigenValues[j-1]){
                    double tmp = eigenValues[j];
                    eigenValues[j] = eigenValues[j-1];
                    eigenValues[j-1] = tmp;
                }

        if(eigenValues.length == 0)
            return RealValue.undefinedValue(this);
        else if (eigenValues[eigenValues.length - 2] == 0)
            return RealValue.positiveInfinity(this);
        else
            return new RealValue (4.0/(n*eigenValues[eigenValues.length - 2]), this);
    }
    
    public String getInvariantId () {
        return "org.grinvin.invariants.MoharIndex2";
    }
    
}
