/* GeneralMatchingNumber.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard.matching;

/**
 * This abstract class contains some common methods used by
 * the other matching number classes.
 */
public abstract class GeneralMatchingNumber {
    /** Adjacency list. */
    protected int[][] adjlist;
    /** Array indicating the incident vertices of the vertices
     *  of the edges belonging the matching.
     *  If a vertex is unsaturated(does not belong to the current
     *  matching) then -1 is used as incident vertex to indicate
     *  that the specific vertex is unsaturated.
     */
    protected int[] matching;
    /** The current size of the matching. */
    protected int matchingSize;
    /** Integer to indicate a vertex is unsaturated. */
    protected static final int UNSATURATED = -1;
    
    /** Creates a new instance of GeneralMatchingNumber */
    public GeneralMatchingNumber() {
        adjlist = null;
        matching = null;
        matchingSize = 0;
    }
    
    /**
     * This method creats an initial assignment using a greedy algorithm.
     * It also initializes the array saturatedVertices correctly.
     */
    protected void initializeMatching() {
        matching = new int[adjlist.length];
        for(int i = 0; i < matching.length; i++)
            matching[i] = UNSATURATED;
        matchingSize = 0;
        int vertex = 0;
        boolean endInitialisation = false;
        while(!endInitialisation) {
            vertex = nextVertex(vertex);
            if(vertex != UNSATURATED) {
                int i = 0;
                // Look for an unsaturated neighbour
                while(i < adjlist[vertex].length 
                      && matching[adjlist[vertex][i]] != UNSATURATED)
                    i++;
                if(i != adjlist[vertex].length) {
                    matching[vertex] = adjlist[vertex][i];
                    matching[adjlist[vertex][i]] = vertex;
                    matchingSize++;
                }
            } else // No valid next vertex
                endInitialisation = true;
        }
    }
    
    /**
     * @return Next unsaturated vertex with an unsaturated neighbour,
     *         -1 otherwise
     */
    private int nextVertex(int startIndex) {
        int index = startIndex;
        while(index < adjlist.length) {
            if(matching[index] != UNSATURATED || adjlist[index].length == 0){
                index++;
            }
            else {
                int i = 0;
                // Look for an unsaturated neighbour
                while(i < adjlist[index].length
                      && matching[adjlist[index][i]] != UNSATURATED)
                    i++;
                if(i == adjlist[index].length)
                    index++;
                else
                    break;
            }
        }
        return index < adjlist.length ? index : -1;
    }
}
