/*
 * Copyright (c) 2005-2008 Laf-Widget Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Laf-Widget Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.lafwidget.tabbed;

import java.awt.Component;
import java.awt.event.*;

import javax.swing.*;

import org.jvnet.lafwidget.*;

/**
 * Adds tab pager to tabbed panes.
 * 
 * @author Kirill Grouchnikov
 */
public class TabPagerWidget extends LafWidgetAdapter<JTabbedPane> implements
		Resettable {
	/**
	 * Mouse wheel listener for Ctrl paging (from version 2.1).
	 */
	protected MouseWheelListener mouseWheelListener;

	/**
	 * Mouse listener for Ctrl paging (from version 2.1).
	 */
	protected MouseListener mouseListener;

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#installUI()
	 */
	@Override
	public void installUI() {
		super.installUI();
		InputMap currMap = SwingUtilities.getUIInputMap(this.jcomp,
				JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);

		InputMap newMap = new InputMap();
		if (currMap != null) {
			KeyStroke[] kss = currMap.allKeys();
			for (int i = 0; i < kss.length; i++) {
				KeyStroke stroke = kss[i];
				Object val = currMap.get(stroke);
				newMap.put(stroke, val);
			}
		}

		newMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT,
				InputEvent.CTRL_MASK), "tabSwitcherForward");
		newMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_LEFT,
				InputEvent.CTRL_MASK), "tabSwitcherBackward");
		newMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_CONTROL, 0, true),
				"tabSwitcherClose");
		newMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
				"tabSwitcherHide");

		this.jcomp.getActionMap().put("tabSwitcherForward",
				new AbstractAction() {
					public void actionPerformed(ActionEvent e) {
						// fix for issue 323 in Substance - don't show pager on tabbed
						// panes with no preview painter
						TabPreviewPainter tpp = LafWidgetUtilities2
								.getTabPreviewPainter(TabPagerWidget.this.jcomp);
						if (tpp == null)
							return;

						TabPagerManager te = TabPagerManager.getPager();
						te.page(TabPagerWidget.this.jcomp, true);
					}
				});

		this.jcomp.getActionMap().put("tabSwitcherBackward",
				new AbstractAction() {
					public void actionPerformed(ActionEvent e) {
						// fix for issue 323 in Substance - don't show pager on
						// tabbed panes with no preview painter
						TabPreviewPainter tpp = LafWidgetUtilities2
								.getTabPreviewPainter(TabPagerWidget.this.jcomp);
						if (tpp == null)
							return;

						TabPagerManager te = TabPagerManager.getPager();
						te.page(TabPagerWidget.this.jcomp, false);
					}
				});

		this.jcomp.getActionMap().put("tabSwitcherClose", new AbstractAction() {
			public void actionPerformed(ActionEvent e) {
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						// fix for issue 323 in Substance - don't show pager on
						// tabbed panes with no preview painter
						TabPreviewPainter tpp = LafWidgetUtilities2
								.getTabPreviewPainter(TabPagerWidget.this.jcomp);
						if (tpp == null)
							return;

						TabPagerManager te = TabPagerManager.getPager();
						int index = te.hide();
						// fix for issue 177 in Substance (disallowing selection
						// of disabled tabs).
						if ((index >= 0)
								&& tpp.isSensitiveToEvents(
										TabPagerWidget.this.jcomp, index)) {
							TabPagerWidget.this.jcomp.setSelectedIndex(index);
						}
					}
				});
			}
		});

		this.jcomp.getActionMap().put("tabSwitcherHide", new AbstractAction() {
			public void actionPerformed(ActionEvent e) {
				TabPagerManager te = TabPagerManager.getPager();
				if (te.isVisible()) {
					te.hide();
				} else {
					// fix for defect 233 on Substance - the ESCAPE event
					// is not dispatched when tab pager is not showing.
					Component comp = jcomp.getParent();
					while (comp != null) {
						if (comp instanceof JComponent) {
							JComponent jc = (JComponent) comp;
							KeyStroke escapeKeyStroke = KeyStroke.getKeyStroke(
									KeyEvent.VK_ESCAPE, 0, false);
							ActionListener al = jc
									.getActionForKeyStroke(escapeKeyStroke);
							if (al != null) {
								al.actionPerformed(e);
								return;
							}
						}
						comp = comp.getParent();
					}
				}
			}
		});

		SwingUtilities.replaceUIInputMap(this.jcomp,
				JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT, newMap);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#uninstallUI()
	 */
	@Override
	public void uninstallUI() {
		super.uninstallUI();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#installListeners()
	 */
	@Override
	public void installListeners() {
		this.mouseWheelListener = new TabPagerMouseWheelListener(this.jcomp);
		this.jcomp.addMouseWheelListener(this.mouseWheelListener);
		this.mouseListener = new MouseAdapter() {
			@Override
			public void mouseClicked(MouseEvent e) {
				if ((e.getModifiers() & InputEvent.CTRL_MASK) != 0) {
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							TabPagerManager te = TabPagerManager.getPager();
							int index = te.hide();
							if (index >= 0)
								TabPagerWidget.this.jcomp
										.setSelectedIndex(index);
						}
					});
				}
			}
		};
		this.jcomp.addMouseListener(this.mouseListener);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#uninstallListeners()
	 */
	@Override
	public void uninstallListeners() {
		this.jcomp.removeMouseWheelListener(this.mouseWheelListener);
		this.mouseWheelListener = null;

		this.jcomp.removeMouseListener(this.mouseListener);
		this.mouseListener = null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidget#requiresCustomLafSupport()
	 */
	public boolean requiresCustomLafSupport() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.Resettable#reset()
	 */
	public void reset() {
		TabPagerManager.reset();
	}
}
