/*
 *   dspfixupi.c -- Label fixup code 
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "port_types.h"
#include "dspmgr.h"

#include "dsppcmem.h"
#include "dspio.h"
#include "dspfixup.h"
#include "dspbios.h"
#include "dspstruc.h"
#define  ONE           1
#define  MULT          2
RC do_segments(PRSEG,ULONG,USHORT);

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FixupTask                                               */
/*                                                                          */
/* FUNCTION: This procedure performs fixup on a single task.                */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK          - pointer to task struct to be fixed up                */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR    - No errors occured                                    */
/*       TDB         - errors                                               */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* REFERCENED FUNCTIONS                                                     */
/*                                                                          */
/*    AllocateTemp                                                          */
/*    FreeTemp                                                              */
/*    DspChgFilePtr                                                         */
/*    DspReadFile                                                           */
/*    FixupSegImage                                                         */
/*    FixupTCB                                                              */
/*    FixupStrideHold                                                       */
/*                                                                          */
/* NOTES:                                                                   */
/*    DEPENDENCIES:                                                         */
/*       AllocateTemp                                                       */
/*       FreeTemp                                                           */
/*       DspChgFilePtr                                                      */
/*       DspReadFile                                                        */
/*       FixupSegImage                                                      */
/*       FixupTCB                                                           */
/*                                                                          */
/*    RESTRICTIONS: NONE                                                    */
/*                                                                          */
/*    If an error is encountered during the process of fixing up the        */
/*    task, the entire fixup process will be aborted, and the error code    */
/*    returned.                                                             */
/*    Some of the fixup may not have been complete upon termination in      */
/*    case of a error.                                                      */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

#define  PAGE_ALIGNED  4096
#define  PAGE_SIZE     4096
#define  DATASEG_INCR  2
#define  INSTSEG_INCR  4
RC FixupTask(PRTSK prtsk)

{
   RC         ulRC = DSP_NOERROR;
   USHORT     usStep;
   USHORT     usTempAddr;   /* CH01 CH02 */
   PRGPC      prgpc;
   PRGPC      prgpcHead;
   typedef    enum {
      ALLOCATE,INSTN_SEGMENTS,FIXUP_TCB,DATA_SEGMENTS
   }
   step_type;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfixup::FixupTask entry prtsk %x\n",(int)prtsk);

   for (usStep = INSTN_SEGMENTS; usStep <= DATA_SEGMENTS && ulRC ==
      DSP_NOERROR; usStep++) {
      switch (usStep) {
         case  INSTN_SEGMENTS :

           /*****************************************************************/
           /* PROCESS ALL THE INSTRUCTION SEGMENTS IN THE TASK              */
           /*****************************************************************/

            ulRC = do_segments(prtsk->TSK_prsegITail, INSTSEG_INCR, MULT);/**/

            break;
         case  FIXUP_TCB :
            if ((prtsk->TSK_usflgType&TSK_TYPEMASK) != TSK_OS)
               ulRC = FixupTCB(prtsk);
            break;
         case  DATA_SEGMENTS :

           /*****************************************************************/
           /* PROCESS ALL THE DATA SEGMENTS IN THE TASK                     */
           /*****************************************************************/

            ulRC = do_segments(prtsk->TSK_prsegDTail, DATASEG_INCR, MULT);/**/

            break;

      }
   }

   /*************************************************************************/
   /* process GPC's for the Task if the (GPC's) exist                       */
   /*************************************************************************/

   if (prtsk->TSK_prgpcTail != NULL && ulRC == DSP_NOERROR) {
      prgpc = prgpcHead = prtsk->TSK_prgpcTail->GPC_prgpcNext;
      do {
         if (prgpc->GPC_usflgKind == GPC_OWNER) {
            /* CH01 CH02 */
            usTempAddr = (USHORT) prgpc->GPC_prmemBuffer->MEM_ulSegAddr;
            if (prgpc->GPC_usAddrMode == GPC_BYTE)
               usTempAddr += 1;  /* CH02 */
            else
               if (prgpc->GPC_usAddrMode == GPC_TABLE)
                  usTempAddr = (usTempAddr >> 1)+0x8000;         /* CH01 */

            PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspfixup::FixupTask");
            PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)\n", prtsk);
            PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
                 "WRITE35:  process GPC's for the Task if the GPC's exist",
                 EOF, (ULONG)1L, prgpc->GPC_prseg->SEG_prtsk->TSK_prdsp,
                 prgpc->GPC_prseg->SEG_prmem->MEM_ulSegAddr +
                 prgpc->GPC_ulOffset, (ULONG)1L, &usTempAddr);

            ulRC = DspBIOS_D_Write(prgpc->GPC_prseg->SEG_prtsk->TSK_prdsp,
               prgpc->GPC_prseg->SEG_prmem->MEM_ulSegAddr+prgpc->GPC_ulOffset,
               (ULONG)1L, &usTempAddr);                          /* CH01 */
         }                             /* endif                             */
         prgpc = prgpc->GPC_prgpcNext;
      }  while (prgpc != prgpcHead && ulRC == DSP_NOERROR);
   }

  /**************************************************************************/
  /* process the IPC (only one per task)                                    */
  /**************************************************************************/

   if (prtsk->TSK_pripcte != NULL && ulRC == DSP_NOERROR) {
      usTempAddr = prtsk->TSK_pripcte->IPCTE_usMask; /* CH01 */

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspfixup::FixupTask");
      PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)\n", prtsk);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "WRITE36:  process the IPC (only one per task)",
           EOF, (ULONG)1L, prtsk->TSK_prdsp,
           prtsk->TSK_prseg_IPC->SEG_prmem->MEM_ulSegAddr+
           prtsk->TSK_ulOffset_IPC, (ULONG)1L, &usTempAddr);

      ulRC = DspBIOS_D_Write(prtsk->TSK_prdsp,
         prtsk->TSK_prseg_IPC->SEG_prmem->MEM_ulSegAddr+
         prtsk->TSK_ulOffset_IPC,
         (ULONG)1L, &usTempAddr);  /* CH01 */
   }

  /**************************************************************************/
  /* Fixup stride hold index values if any dma assoicated with this task    */
  /**************************************************************************/

   if (prtsk->TSK_prldmaTail != NULL && ulRC == DSP_NOERROR)
      ulRC = FixupStrideHold(prtsk);

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfixup::FixupTask exit ulRC %lx\n",ulRC);
   return  ulRC;
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FixupSegment                                            */
/*                                                                          */
/* FUNCTION: This procedure performs fixup on a single segment              */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRSEG          - pointer to seg  struct to be fixed up                */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR    - No errors occured                                    */
/*       TDB         - errors                                               */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* REFERCENED FUNCTIONS                                                     */
/*                                                                          */
/*    AllocateTemp                                                          */
/*    FreeTemp                                                              */
/*    DspChgFilePtr                                                         */
/*    DspReadFile                                                           */
/*    FixupSegImage                                                         */
/*    FixupTCB                                                              */
/*                                                                          */
/* NOTES:                                                                   */
/*    DEPENDENCIES:                                                         */
/*       AllocateTemp                                                       */
/*       FreeTemp                                                           */
/*       DspChgFilePtr                                                      */
/*       DspReadFile                                                        */
/*       FixupSegImage                                                      */
/*       FixupTCB                                                           */
/*                                                                          */
/*    RESTRICTIONS: NONE                                                    */
/*                                                                          */
/*    If an error is encountered during the process of fixing up the        */
/*    segment, the entire fixup process will be aborted, and the error code */
/*    returned.                                                             */
/*    Some of the fixup may not have been complete upon termination in      */
/*    case of a error.                                                      */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

#define  PAGE_ALIGNED  4096
#define  PAGE_SIZE     4096
#define  DATASEG_INCR  2
#define  INSTSEG_INCR  4
RC FixupSegment(PRSEG prseg)

{
   RC         ulRC = DSP_NOERROR;
   USHORT     usStep;
   USHORT     usTempAddr; /* CH01 CH02 */
   PRGPC      prgpc;
   PRGPC      prgpcHead;

   typedef    enum {
      ALLOCATE,INSTN_SEGMENTS,FIXUP_TCB,DATA_SEGMENTS
   }
   step_type;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfixup::FixupSegment entry prseg %x\n",(int)prseg);
   for (usStep = INSTN_SEGMENTS; usStep <= DATA_SEGMENTS && ulRC ==
      DSP_NOERROR; usStep++) {
      switch (usStep) {
         case  INSTN_SEGMENTS :

           /*****************************************************************/
           /* Process the Instruction Segment.                              */
           /*****************************************************************/

            if ((prseg->SEG_usflg&SEG_TYPEMASK) == SEG_INST)
               ulRC = do_segments(prseg->SEG_prtsk->TSK_prsegITail,
                  INSTSEG_INCR, ONE);
            break;

         case  FIXUP_TCB :
            break;

         case  DATA_SEGMENTS :

           /*****************************************************************/
           /* Process the Data Segment.                                     */
           /*****************************************************************/

            if ((prseg->SEG_usflg&SEG_TYPEMASK) == SEG_DATA)
               ulRC = do_segments(prseg->SEG_prtsk->TSK_prsegDTail,
                  DATASEG_INCR, ONE);
            break;

      }                                /* end switch                        */
   }                                   /* endfor                            */

   /*************************************************************************/
   /* Process GPCs for this Segment. Note that you have to find the GPC     */
   /* from the list of GPCs on the Task.                                    */
   /*************************************************************************/

   if (prseg->SEG_prtsk->TSK_prgpcTail != NULL) {
      prgpc = prgpcHead = prseg->SEG_prtsk->TSK_prgpcTail->GPC_prgpcNext;
      do {

         if (prgpc->GPC_prseg == prseg) {

            if (prgpc->GPC_usflgKind == GPC_OWNER) {
               /* CH01 CH02 */
               usTempAddr = (USHORT) prgpc->GPC_prmemBuffer->MEM_ulSegAddr;
               if (prgpc->GPC_usAddrMode == GPC_BYTE)
                  usTempAddr += 1; /* CH02 */
               else
                  if (prgpc->GPC_usAddrMode == GPC_TABLE)
                     usTempAddr = (usTempAddr >> 1)+0x8000;

               PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a",
                    "dspfixup::FixupSegment");
               PRINT_DEBUG(EMBEDFILEPTR, "  prseg (0x%X)\n", prseg);
               PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
                    "WRITE37:  process GPC's for the Seg if the GPC's exist",
                    EOF, (ULONG)1L, prgpc->GPC_prseg->SEG_prtsk->TSK_prdsp,
                    prgpc->GPC_prseg->SEG_prmem->MEM_ulSegAddr+
                    prgpc->GPC_ulOffset, (ULONG)1L, &usTempAddr);

               ulRC = DspBIOS_D_Write(prgpc->GPC_prseg->SEG_prtsk->TSK_prdsp,
                  prgpc->GPC_prseg->SEG_prmem->MEM_ulSegAddr+
                  prgpc->GPC_ulOffset, (ULONG)1L, &usTempAddr);
            }
         }
         prgpc = prgpc->GPC_prgpcNext;     /* CH03 DEC */
      }  while (prgpc != prgpcHead && ulRC == DSP_NOERROR);
   }

  /**************************************************************************/
  /* process the IPC. Only one per task and it has to match Seg.            */
  /**************************************************************************/

   if (prseg->SEG_prtsk->TSK_pripcte != NULL && ulRC == DSP_NOERROR) {
      if (prseg->SEG_prtsk->TSK_prseg_IPC == prseg) {
         usTempAddr = prseg->SEG_prtsk->TSK_pripcte->IPCTE_usMask;

         PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspfixup::FixupSegment");
         PRINT_DEBUG(EMBEDFILEPTR, "  prseg (0x%X)\n", prseg);
         PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
              "WRITE38:  process IPC (only one per task and must match seg)",
              prseg->SEG_prtsk->TSK_prdsp, EOF, (ULONG)1L,
              prseg->SEG_prtsk->TSK_prseg_IPC->SEG_prmem->MEM_ulSegAddr +
              prseg->SEG_prtsk->TSK_ulOffset_IPC, (ULONG)1L, &usTempAddr);

         ulRC = DspBIOS_D_Write(prseg->SEG_prtsk->TSK_prdsp,
            prseg->SEG_prtsk->TSK_prseg_IPC->
            SEG_prmem->MEM_ulSegAddr+prseg->SEG_prtsk->TSK_ulOffset_IPC, /* */
            (ULONG)1L, &usTempAddr);
      }
   }
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfixup::FixupSegment exit ulRC %lx\n",ulRC);
   return  ulRC;
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  DO SEGMENTS                                            */
/*                                                                          */
/* FUNCTION:  To process a segment chain and do whatever fixup is needed.   */
/*                                                                          */
/* INPUT:  prsegTail  - a pointer to the tail of the segment chain.         */
/*         usIncr - an increment value used to allocate memory.             */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR    - No errors occured                                    */
/*       TDB         - errors                                               */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* NOTES:                                                                   */
/*    DEPENDENCIES:                                                         */
/*                                                                          */
/*    RESTRICTIONS: NONE                                                    */
/*                                                                          */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC do_segments(PRSEG prsegTail,ULONG ulIncr,USHORT usNum)
{
   RC         ulRC = DSP_NOERROR;
   PRSEG      prseg = prsegTail;
   PRSEGTMP   prsegtmp;
   PRIMG      primg;
   PVOID      pMembuf;
   USHORT     usStep = 0;
   RC         ReadSegmentImage(PRIMG, PVOID, HFILE, ULONG);

   /* do the following steps while no error conditions                      */
   /* arise and there are more unprocessed segments                         */
   /* CH04 to next seg and just go thru once.                               */

   MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspfixup::do_segments entry prsegTail %x ulIncr %lx usNum %x\n",(int)prseg,ulIncr,usNum);
   do {
      if (usNum == MULT)
         prseg = prseg->SEG_prsegNext;
      prsegtmp = prseg->SEG_prsegtmp;

      /**********************************************************************/
      /* does this segment require fixup !                                  */
      /* NOTE: the IBM ModuleCompiler and Intermetric's Linker              */
      /*       generate different .dsp files when a segment doesn't         */
      /*       have an image.                                               */
      /*       ModComp produces an image tag with size == zero.             */
      /*       The Linker omits the image tag. The following 'if'           */
      /*       statement with two clauses handles both cases.               */
      /**********************************************************************/

      if ((prsegtmp->SEG_ulMemSize != 0) && (prsegtmp->SEG_primgTail != NULL))
         {
         pMembuf=NULL; /* CH01 initialize for post algorithm failure check */
         for (usStep = 0; usStep <= 5 && ulRC == DSP_NOERROR; usStep++) {
            switch (usStep) {
               case 0 :                /* allocate buffer for segment image */
                  ulRC = AllocateTemp(prsegtmp->SEG_ulMemSize *ulIncr,
                     PAGE_ALIGNED, &pMembuf);
                  break;
               case 1 :                /* read segment image into buffer    */
                  ulRC = ReadSegmentImage(prsegtmp->SEG_primgTail, pMembuf,
                     prsegtmp->SEG_hfile, ulIncr);
                  break;
               case 2 :                /* do run-time fixup on image        */
                  ulRC = FixupSegImage(prseg, pMembuf);
                  break;
               case 3 :                /* write segment image to dsp memory */
                  if (prseg->SEG_prmem != NULL)/* anything to write?        */
                     {
                     primg = prsegtmp->SEG_primgTail;/* point to tail of    */
                                       /* image list                        */
                     if (ulIncr == DATASEG_INCR)
                        do {
                           primg = primg->IMG_primgNext;

                        /* skip write if image size is zero                 */

                           if (primg->IMG_ulSize > 0)
                              {
                              PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME,
                                   "a", "dspfixup::do_segments");
                              PRINT_DEBUG(EMBEDFILEPTR,
                                   "  prsegTail (0x%X)", prsegTail);
                              PRINT_DEBUG(EMBEDFILEPTR,
                                   "  ulIncr (0x%X)", ulIncr);
                              PRINT_DEBUG(EMBEDFILEPTR,
                                   "  usNum (0x%X)\n", usNum);
                              PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
                                   "WRITE39:  data seg image to dsp memory",
                                   EOF, (ULONG) 01L,
                                   prseg->SEG_prtsk->TSK_prdsp,
                                   prseg->SEG_prmem->MEM_ulSegAddr+
                                   primg->IMG_ulOffset, primg->IMG_ulSize,
                                   (PVOID)((PCHAR)pMembuf +
                                   primg->IMG_ulOffset));

                              ulRC = DspBIOS_D_Write
                                 (prseg->SEG_prtsk->TSK_prdsp,
                                 prseg->SEG_prmem->MEM_ulSegAddr+
                                 primg->IMG_ulOffset, primg->IMG_ulSize,
                                 (PVOID)((PCHAR)pMembuf+primg->IMG_ulOffset));
                           }
                        }  while (primg != prsegtmp->SEG_primgTail && ulRC ==
                           DSP_NOERROR);
                     else
                        do {
                           primg = primg->IMG_primgNext;

                        /* skip write if image size is zero                 */

                           if (primg->IMG_ulSize > 0)
                              {
                              PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME,
                                   "a", "dspfixup::do_segments");
                              PRINT_DEBUG(EMBEDFILEPTR,
                                   "  prsegTail (0x%X)", prsegTail);
                              PRINT_DEBUG(EMBEDFILEPTR,
                                   "  ulIncr (0x%X)", ulIncr);
                              PRINT_DEBUG(EMBEDFILEPTR,
                                   "  usNum (0x%X)\n", usNum);
                              PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
                                   "WRITE40:  inst seg image to dsp memory",
                                   EOF, (ULONG) 01L,
                                   prseg->SEG_prtsk->TSK_prdsp,
                                   prseg->SEG_prmem->MEM_ulSegAddr+
                                   primg->IMG_ulOffset, primg->IMG_ulSize,
                                   (PVOID)((PCHAR)pMembuf+
                                   (primg->IMG_ulOffset * ulIncr)));

                              ulRC = DspBIOS_I_Write
                                 (prseg->SEG_prtsk->TSK_prdsp,
                                 prseg->SEG_prmem->MEM_ulSegAddr+
                                 primg->IMG_ulOffset, primg->IMG_ulSize,
                                 (PVOID)((PCHAR)pMembuf+(primg->IMG_ulOffset
                                 *ulIncr)));
                           }
                        }  while (primg != prsegtmp->SEG_primgTail && ulRC ==
                           DSP_NOERROR);
                  }
                  break;
               case 4 :                /* free storage for image buffer     */
                  ulRC = FreeTemp(pMembuf, prsegtmp->SEG_ulMemSize *ulIncr);
                  pMembuf=NULL;        /* CH01 indicate mem has been freed  */
                  break;
               case 5 :                /* free each image data structure for*/
                                       /* this segment                      */
                  while (prsegtmp->SEG_primgTail != NULL && ulRC ==
                     DSP_NOERROR)
                     ulRC = RemoveRImage(&prsegtmp->SEG_primgTail,
                        prsegtmp->SEG_primgTail->IMG_primgNext,
                        prsegtmp->SEG_primgTail);
                  break;
            }                          /* end of switch                     */
         }                             /* endfor                            */
         if (NULL!=pMembuf) {          /* CH01 force a free if step 4 did   */
                                       /* not execute because of error      */
            (void) FreeTemp(pMembuf, prsegtmp->SEG_ulMemSize *ulIncr);
         }                             /* endif */
      }                                /* end if                            */
   }  while ((prseg != prsegTail) && (ulRC == DSP_NOERROR) && (usNum == MULT))
      ;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfixup::do_segments exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS**************************/
/*                                                                          */
/* SUBROUTINE NAME:  READSEGMENTIMAGE                                       */
/*                                                                          */
/* FUNCTION:  Reads n pieces of the segment image from the .DSP file.       */
/*                                                                          */
/* INPUT:  primgTail - pointer to the tail of the segment image list.       */
/*         pMembuf - pointer to a memory buffer used in the process.        */
/*         pfile - handle to the .DSP file.                                 */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR    - No errors occured                                    */
/*       TDB         - errors                                               */
/*                                                                          */
/* NOTES:                                                                   */
/*    DEPENDENCIES:                                                         */
/*                                                                          */
/*    RESTRICTIONS: NONE                                                    */
/*                                                                          */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC ReadSegmentImage(PRIMG primgTail,PVOID pMembuf,HFILE hfile,ULONG ulIncr)
{
   RC         ulRC = DSP_NOERROR;
   PRIMG      primg;
   ULONG      ulOffset;

   MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspfixup::ReadSegmentImage entry primgTail %x pMembuf %x hfile %x ulIncr %lx\n",
	   (int)primgTail,(int)pMembuf,(int)hfile,ulIncr);
   primg = primgTail;                  /* point to tail of linked list      */
   do {
      primg = primg->IMG_primgNext;
      if (primg->IMG_ulSize > 0)
         {

         /* move file pointer to point at image                             */

         if ((ulRC = DspChgFilePtr(hfile, primg->IMG_ulFileOffset, FROMTOP))
            != DSP_NOERROR)
            return (ulRC);

         /* compensate for image size (DATA | INSTRUCTION)                  */

         if (ulIncr == DATASEG_INCR)
            ulOffset = primg->IMG_ulOffset;
         else
            ulOffset = primg->IMG_ulOffset *ulIncr;

         /* read image from file and write to buffer                        */

         if ((ulRC = DspReadFile(hfile, primg->IMG_ulSize *ulIncr, (PCHAR)
            pMembuf+ulOffset)) != DSP_NOERROR)
            return (ulRC);
      }
   }  while (primg != primgTail);
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfixup::ReadSegmentImage exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FixupModule                                             */
/*                                                                          */
/* FUNCTION: This procedure performs fixup on an entire module              */
/*                                                                          */
/* INPUT:                                                                   */
/*    PRTSK          - pointer to module struct to be fixed up              */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR    - No errors occured                                    */
/*       TDB         - errors                                               */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* NOTES:                                                                   */
/*    DEPENDENCIES:                                                         */
/*       FixupTask                                                          */
/*                                                                          */
/*    RESTRICTIONS: NONE                                                    */
/*                                                                          */
/*    If an error is encountered during the process of fix up,              */
/*    the entire function will be aborted, and the error code returned.     */
/*    Some of the function may not have been completed upon termination in  */
/*    case of a error.                                                      */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FixupModule(PRMOD prmod)
{
   RC         ulRC = DSP_NOERROR;
   PRTSK      prtsk;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfixup::FixupModule entry prmod %x\n",(int)prmod);
   if ((prtsk = prmod->MOD_prtskTail) != NULL)/* empty module?              */
      {
      do                               /* for each task in the module       */
         {
         prtsk = prtsk->TSK_prtskNext; /* 1st time, move to head of list    */
         if ((prtsk->TSK_usflgType&TSK_TYPEMASK) != TSK_OS)/* skip if OS    */
            ulRC = FixupTask(prtsk);
      }  while (prtsk != prmod->MOD_prtskTail && ulRC == DSP_NOERROR);
   }
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfixup::FixupModule exit ulRC %lx\n",ulRC);
   return (ulRC);
}
