/*
 *   mwmamp.c -- Mwave Modem AT Command Parser
 *
 *  Written By: Paul Schroeder IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
/*****************************************************************************/
/* These routines handle the parsing of Ampersand AT Commands.               */
/* mwmAmpCommand is called from mwmCmdParseATcmd when an '&' is encountered  */
/* in the AT command string.                                                 */
/* If the following character is recognized as a valid Ampersand Command,    */
/* the appropriate routine is called.  If not, MWM_ATCMD_ERROR is returned.  */
/*****************************************************************************/

#include <mwmparse.h>
#include <mwmparsi.h>

USHORT  mwmAmpKCommand(STATEINFO *psi);



USHORT mwmAmpCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpCommand entry\n");
  /***************************************************************************/
  /* Look at this character in AT Command.                                   */
  /* Then advance index to point to next character.                          */
  /***************************************************************************/
  switch(psi->achCommandBuffer[psi->usNextATIndex++])
  {
    case 'C' :
      /***********************************************************************/
      /* This is an &C command.                                              */
      /***********************************************************************/
      usParserStatus = mwmAmpCCommand(psi);
      break;

    case 'D' :
      /***********************************************************************/
      /* This is an &D command.                                              */
      /***********************************************************************/
      usParserStatus = mwmAmpDCommand(psi);
      break;

    case 'F' :
      /***********************************************************************/
      /* This is an &F command.                                              */
      /***********************************************************************/
      psi->usSavedCommandParameter = 2;
      psi->usCurrentCommand = MWM_AMP_F_CMD;                        /* PTR 2356*/
      usParserStatus = mwmParseInitializeModem( psi, 2 );
      break;

    case 'G' :
      /***********************************************************************/
      /* This is an &G command.                                              */
      /***********************************************************************/
      usParserStatus = mwmAmpGCommand(psi);
      break;

    case 'J' :
      /***********************************************************************/
      /* This is an &J command.                                              */
      /* Allow the &J0 and &J1 parameters, but don't send any PP commands.   */
      /* All other parameters return ERROR.                                  */
      /***********************************************************************/
      usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                          &psi->usNextATIndex);
      if (usParm > 1)
        usParserStatus = MWM_ATCMD_ERROR;
      break;

    case 'K' :
      /***********************************************************************/
      /* This is an &K command.                                              */
      /***********************************************************************/
      usParserStatus = mwmAmpKCommand(psi);
      break;

    case 'L' :
      /***********************************************************************/
      /* This is an &L command.                                              */
      /* Allow the &L0 parameter, but don't send any PP commands.            */
      /* All other parameters return ERROR.                                  */
      /***********************************************************************/
      usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                          &psi->usNextATIndex);
      if (usParm)
        usParserStatus = MWM_ATCMD_ERROR;
      break;

    case 'M' :
      /***********************************************************************/
      /* This is an &M command.                                              */
      /* Allow the &M0 parameter, but don't send any PP commands.            */
      /* All other parameters return ERROR.                                  */
      /***********************************************************************/
      usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                          &psi->usNextATIndex);
      if (usParm)
        usParserStatus = MWM_ATCMD_ERROR;
      break;

    case 'N' :          /* PTR 592*/
      /***********************************************************************/
      /* PTR 592                                                             */
      /* This is a &N command...fill the PP command.                         */
      /***********************************************************************/
      /* If the PP results of this command will fit into the buffer.         */
      /***********************************************************************/
      if ( (psi->usNextPPIndex + PP_AMP_N_CMD_SPACE) < PP_BUFFER_THRESHOLD)  /* PTR 592*/
      {                                                              /* PTR 592*/
        /*********************************************************************/
        /* Get the parameter from the command string                         */
        /*********************************************************************/
        usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,   /* PTR 592*/
                                       &psi->usNextATIndex);         /* PTR 592*/
                                                                     /* PTR 592*/
        usParserStatus = mwmAmpNCommand(psi,usParm);                 /* PTR 592*/
                                                                     /* PTR 592*/
      }                                                              /* PTR 592*/
      else                                                           /* PTR 592*/
      {                                                              /* PTR 592*/
        /*********************************************************************/
        /* There is not enough room left in this PP Command buffer to        */
        /* process this command....                                          */
        /*********************************************************************/
        usParserStatus = MWM_GET_MORE_BUFFERS;                       /* PTR 592*/
        psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;              /* PTR 592*/
      }                                                              /* PTR 592*/
      break;                                                         /* PTR 592*/
    case 'P' :
      /***********************************************************************/
      /* This is an &P command.                                              */
      /***********************************************************************/
      usParserStatus = mwmAmpPCommand(psi);
      break;

    case 'Q' :
      /***********************************************************************/
      /* This is an &Q command.                                              */
      /* Allow the &Q0 parameter, but don't send any PP commands.            */
      /* All other parameters return ERROR.                                  */
      /***********************************************************************/
      usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                          &psi->usNextATIndex);

      /***********************************************************************/
      /* MTS 5096                                                            */
      /* 05/11/95                                                            */
      /* Allow 5 and 6 as well as 0.  For each one, simply return OK.        */
      /***********************************************************************/
      switch (usParm)
      {
        case 0:
        case 5:
        case 6:
          break;
        default:
          usParserStatus = MWM_ATCMD_ERROR;
          break;
      }
      break;

    case 'S' :
      /***********************************************************************/
      /* This is an &S command.                                              */
      /***********************************************************************/
      usParserStatus = mwmAmpSCommand(psi);
      break;

    case 'T' :
      /***********************************************************************/
      /* This is an &T command.                                              */
      /***********************************************************************/
      usParserStatus = mwmAmpTCommand(psi);
      break;

    case 'U' :
      /***********************************************************************/
      /* This is an &U command.                                              */
      /***********************************************************************/
      usParserStatus = mwmAmpUCommand(psi);
      break;

    case 'V' :
      /***********************************************************************/
      /* This is an &V command.                                              */
      /***********************************************************************/
      psi->usSubCommandInfo = 0;
      psi->usSavedCommandParameter = 1;
      usParserStatus = mwmAmpVCommand(psi);
      break;

    case 'W' :
      /***********************************************************************/
      /* This is an &W command.                                              */
      /***********************************************************************/
      usParserStatus = mwmAmpWCommand(psi);
      break;

    case 'Y' :
      /***********************************************************************/
      /* This is an &Y command.                                              */
      /***********************************************************************/
      usParserStatus = mwmAmpYCommand(psi);
      break;

    case 'Z' :
      /***********************************************************************/
      /* This is an &Z command.                                              */
      /***********************************************************************/
      usParserStatus = mwmAmpZCommand(psi);
      break;

    default:
      usParserStatus = MWM_ATCMD_ERROR;
  }


  if (usParserStatus & MWM_GET_MORE_BUFFERS)
    /*************************************************************************/
    /* Move the AT Command Index back to this command to be parsed           */
    /* on the next interrupt.                                                */
    /*************************************************************************/
    psi->usNextATIndex--;
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}

USHORT mwmAmpCCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpCCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_AMP_C_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     switch (usParm)
     {
       case 0:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              21,0xffdf,0);
         break;
       case 1:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              21,0xffff,0x20);
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpCCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmAmpDCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpDCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_AMP_D_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                         &psi->usNextATIndex);

     if (usParm < 4)
     {

       USHORT usAndMask = 0xffe7|usParm;
       usParm = usParm<<3;
       mwmParseSetSRegister(psi->ausPPcmdBuffer,
                            &psi->usNextPPIndex,
                            21,usAndMask,usParm);
     }
     else
       usParserStatus = MWM_ATCMD_ERROR;
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpDCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmAmpGCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpGCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_AMP_G_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {

    /*************************************************************************/
    /* MTS 4227 Remove All WT Queries for &G.                                */
    /* Let MCTL handle the WT processing for this command.                   */
    /*************************************************************************/
    usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     switch (usParm)
     {
       case 0:
         /********************************************************************/
         /* Guard Tone is not required, so allow the &G0                     */
         /********************************************************************/
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              23,0xff3f,0);
         break;
       case 1:
         usParserStatus = MWM_ATCMD_ERROR;
         break;
       case 2:
         /********************************************************************/
         /* Gaurd Tone is allowed, so set the &G2 as the user requested.     */
         /********************************************************************/
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              23,0xffbf,0x80);
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpGCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




/*****************************************************************************/
/* MTS 5095                                                                  */
/* mwmAmpKCommand                                                            */
/*****************************************************************************/
/* Called by...                                                              */
/*   mwmAmpCommand when...                                                   */
/*   Amp   K command is encountered in the AT command string.                */
/*                                                                           */
/* Treat like a \Q command.....only map parameters somewhat differently.     */
/*****************************************************************************/
USHORT  mwmAmpKCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm = 0;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpKCommand entry\n");

  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_SLASH_Q_CMD_SPACE + PP_XON_XOFF_CMD_SPACE)
        < PP_BUFFER_THRESHOLD)
  {

    /*************************************************************************/
    /* MTS 4227 Remove All WT Queries for &G.                                */
    /* Let MCTL handle the WT processing for this command.                   */
    /*************************************************************************/
    usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

    switch (usParm)
    {
      case 0:
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x2B | psi->usParserMode);
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 0;
        break;
      case 1:
      case 3:
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x2B | psi->usParserMode);
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 3;
        break;
      /***********************************************************
      ** &K4 - Enables XON/XOFF flow control
      ** &K5 - Enables transparent XON/XOFF flow control
      **
      ** MTS 6555 - Added 5/20/96
      ************************************************************/
      case 4:
      case 5:
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x2B | psi->usParserMode);
        psi->ausPPcmdBuffer[psi->usNextPPIndex++] = usParm;
        break;
      default:
        usParserStatus = MWM_ATCMD_ERROR;
    }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpKCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}





/*****************************************************************************/
/* mwmAmpNCommand                                              // PTR 592    */
/*****************************************************************************/
/* Called by...                                                              */
/*   mwmAmpCommand when...                                                   */
/*   Amp N command is encountered in the AT command string.                  */
/*                                                                           */
/*   mwmParseInitializeModem when...                                         */
/*   The modem is initialized, either at startup, or when a profile is       */
/*   being loaded into the modem.                                            */
/*                                                                           */
/* The check for buffer full should take place before calling this routine.  */
/*****************************************************************************/
USHORT  mwmAmpNCommand(STATEINFO *psi, USHORT usParm)                /* PTR 592*/
{                                                                    /* PTR 592*/
  USHORT usParserStatus = 0;                                         /* PTR 592*/

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpNCommand entry\n");
  if (usParm > 1)                                                    /* PTR 592*/
    usParserStatus = MWM_ATCMD_ERROR;                                /* PTR 592*/

  if (!usParserStatus)                                               /* PTR 592*/
  {                                                                  /* PTR 592*/
    /*************************************************************************/
    /* add the PP command for Amp N to the PP command string.                */
    /*************************************************************************/
    psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x28 | psi->usParserMode);  /* PTR 592*/

    /*************************************************************************/
    /* Add the parameter to the end of the PP command string.                */
    /*************************************************************************/
    psi->ausPPcmdBuffer[psi->usNextPPIndex++] = usParm;              /* PTR 592*/
  }                                                                  /* PTR 592*/
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpNCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;                                             /* PTR 592*/
}                                                                    /* PTR 592*/


USHORT mwmAmpPCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpPCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_AMP_P_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     switch (usParm)
     {
       case 0:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              22,0xff7f,0);
         break;
       case 1:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              22,0xffff,0x80);
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpPCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmAmpSCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpSCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_AMP_S_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     switch (usParm)
     {
       case 0:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              21,0xffbf,0);
         break;
       case 1:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              21,0xffff,0x40);
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpPCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmAmpTCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpTCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_AMP_T_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                    &psi->usNextATIndex);

     switch (usParm)
     {
       case 0:
          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x10 | psi->usParserMode);
          usParserStatus = MWM_NO_MORE_COMMANDS;
          break;
       case 1:
          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x11 | psi->usParserMode);
          usParserStatus = MWM_NO_MORE_COMMANDS;
         break;
       case 3:
          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x13 | psi->usParserMode);
          break;
       case 4:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              23,0xffff,1);
         break;
       case 5:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              23,0xfffe,0);
         break;
       case 6:
          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x16 | psi->usParserMode);
          usParserStatus = MWM_NO_MORE_COMMANDS;
          break;
       case 7:
          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x17 | psi->usParserMode);
          usParserStatus = MWM_NO_MORE_COMMANDS;
         break;
       case 8:
          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x18 | psi->usParserMode);
          usParserStatus = MWM_NO_MORE_COMMANDS;
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpPCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmAmpUCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpUCommand entry\n");
  /***************************************************************************/
  /* If the PP results of this command will fit into the buffer.             */
  /***************************************************************************/
  if ( (psi->usNextPPIndex + PP_AMP_U_CMD_SPACE) < PP_BUFFER_THRESHOLD)
  {
     usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                         &psi->usNextATIndex);

     switch (usParm)
     {
       case 0:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              27,0xff7f,0);
         break;
       case 1:
         mwmParseSetSRegister(psi->ausPPcmdBuffer,
                              &psi->usNextPPIndex,
                              27,0xffff,0x80);
         break;
       default:
         usParserStatus = MWM_ATCMD_ERROR;
     }
  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpUCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}



USHORT mwmAmpVPhoneNumberLine(STATEINFO *psi, USHORT usLine, PSZ pszOutputLine)
{
  char   *aachTemplateString[]
       = {"TELEPHONE NUMBERS:",
          "PHONE0=%s",
          "PHONE1=%s",
          "PHONE2=%s",
          "PHONE3=%s",
          " ",
         };

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpVPhoneNumberLine entry\n");
  switch(usLine)
  {
    case 0:
      strncpy(pszOutputLine,aachTemplateString[usLine],PP_CMD_BUFFER_SIZE);
      break;

    case 1:
      sprintf(pszOutputLine,aachTemplateString[usLine],psi->ModemMemory.aachStoredNumbers[0]);
      break;

    case 2:
      sprintf(pszOutputLine,aachTemplateString[usLine],psi->ModemMemory.aachStoredNumbers[1]);
      break;

    case 3:
      sprintf(pszOutputLine,aachTemplateString[usLine],psi->ModemMemory.aachStoredNumbers[2]);
      break;

    case 4:
      sprintf(pszOutputLine,aachTemplateString[usLine],psi->ModemMemory.aachStoredNumbers[3]);
      break;

/*    case 5:*/
/*      strcpy(pszOutputLine,aachTemplateString[usLine]);*/
/*      break;*/

  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpVPhoneNumberLine exit ulRC %x\n",0);
  return 0;
}





USHORT mwmAmpVProfileLine(STATEINFO *psi, USHORT usProfile, USHORT usLine, PSZ pszOutputLine)
{
  char    achTempString[20];
  char    chDialingMode;
  USHORT  usCCommandValue = 0;
  //static  USHORT  usS32Value, usS33Value;                              /*MTS6555*/

  char   *aachTemplateString[]
       = {"%sE%1d Q%1d W%1d V%1d %c &D%1d &C%1d &N%1d &S%1d Y%1d L%1d M%1d X%1d &G%1d B%1d &U%1d -SSE%1d *TH%1d",
          "\\A%1d \\K%1d \\L%1d \\N%1d &K%1d \\T%1d \\V%1d %%A%1d %%C%1d %%E%1d \"H%1d \"N%1d \"O%d -J%1d",
          "S0=%02d S2=%02d S3=%02d S4=%02d S5=%02d S6=%02d S7=%02d S8=%02d S9=%02d S10=%02d",
          "S11=%02d S12=%02d S24=%02d S25=%02d S28=%02d S30=%02d S32=%02d S33=%02d S49=%02d",
          "S50=%02d S51=%02d S52=%02d S53=%02d S60=%02d",
         };

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpVProfileLine entry\n");
  switch(usLine)
  {
    case 0:
      switch (usProfile)
      {
        case 3:
          strncpy(pszOutputLine,"ACTIVE PROFILE:",PP_CMD_BUFFER_SIZE);
          //usS32Value = psi->usS32Value;                                /*MTS6555*/
          //usS33Value = psi->usS33Value;                                /*MTS6555*/
          break;
        case 0:
          strncpy(pszOutputLine,"STORED PROFILE 0:",PP_CMD_BUFFER_SIZE);
          //usS32Value = psi->ModemMemory.aStoredRegs[usProfile].S32;    /*MTS6555*/
          //usS33Value = psi->ModemMemory.aStoredRegs[usProfile].S33;    /*MTS6555*/
          break;
        case 1:
          strncpy(pszOutputLine,"STORED PROFILE 1:",PP_CMD_BUFFER_SIZE);
          //usS32Value = psi->ModemMemory.aStoredRegs[usProfile].S32;    /*MTS6555*/
          //usS33Value = psi->ModemMemory.aStoredRegs[usProfile].S33;    /*MTS6555*/
          break;
      }
      break;
    case 1:
      /***********************************************************************/
      /* Build the potential output line...                                  */
      /***********************************************************************/
      memset(achTempString,0,20);
      if (usProfile == 3)
      {
        sprintf(achTempString,"&Y%1d ",psi->ModemMemory.usStartupProfile);
      }



      /***********************************************************************/
      /* PTR 2238  11/11/93                                                  */
      /* One command doesn't map the same as the rest of the commands        */
      /* The C command doesn't use the bit values directly as do the other   */
      /* commands, so we will use a switch statement to map the values to    */
      /* the C command.                                                      */
      /***********************************************************************/
      switch ((psi->ModemMemory.aStoredRegs[usProfile].S[22] & 0x70)>>4)
      {
        case 0:
          usCCommandValue = 0;
          break;
        case 4:
          usCCommandValue = 1;
          break;
        case 5:
          usCCommandValue = 2;
          break;
        case 6:
          usCCommandValue = 3;
          break;
        case 7:
          usCCommandValue = 4;
          break;
      }


      /***********************************************************************/
      /* 12/06/93                                                            */
      /* PTR 2347                                                            */
      /* Determine Dialing Mode (P or T)...                                  */
      /***********************************************************************/
      if ((psi->ModemMemory.aStoredRegs[usProfile].S[14] & 0x20)>>4)
      {
        chDialingMode = 'P';
      }
      else
      {
        chDialingMode = 'T';
      }
      sprintf(pszOutputLine,aachTemplateString[0],
              achTempString,
              (psi->ModemMemory.aStoredRegs[usProfile].S[14] & 0x02)>>1,
              (psi->ModemMemory.aStoredRegs[usProfile].S[14] & 0x04)>>2,
              psi->ModemMemory.aStoredRegs[usProfile].WCommand,
              (psi->ModemMemory.aStoredRegs[usProfile].S[14] & 0x08)>>3,
              chDialingMode,                                        /* PTR 2347*/
              (psi->ModemMemory.aStoredRegs[usProfile].S[21] & 0x18)>>3,
              (psi->ModemMemory.aStoredRegs[usProfile].S[21] & 0x20)>>5,
              psi->ModemMemory.aStoredRegs[usProfile].AmpN,         /* PTR  592*/
              (psi->ModemMemory.aStoredRegs[usProfile].S[21] & 0x40)>>6,
              (psi->ModemMemory.aStoredRegs[usProfile].S[21] & 0x80)>>7,
              (psi->ModemMemory.aStoredRegs[usProfile].S[22] & 0x03),
              (psi->ModemMemory.aStoredRegs[usProfile].S[22] & 0x0C)>>2,
              usCCommandValue,
              (psi->ModemMemory.aStoredRegs[usProfile].S[23] & 0xC0)>>6,
              (psi->ModemMemory.aStoredRegs[usProfile].S[27] & 0x40)>>6,
              (psi->ModemMemory.aStoredRegs[usProfile].S[27] & 0x80)>>7,
              psi->ModemMemory.aStoredRegs[usProfile].DashSSE,
              psi->ModemMemory.aStoredRegs[usProfile].StarTH );

      break;

    case 2:
      sprintf(pszOutputLine,aachTemplateString[1],
              psi->ModemMemory.aStoredRegs[usProfile].SlashA,
              psi->ModemMemory.aStoredRegs[usProfile].SlashK,
              psi->ModemMemory.aStoredRegs[usProfile].SlashL,
              psi->ModemMemory.aStoredRegs[usProfile].SlashN,
              psi->ModemMemory.aStoredRegs[usProfile].SlashQ,   /* MTS 3017*/
              (psi->ModemMemory.aStoredRegs[usProfile].SlashT+30)/60, /*MTS6384*/
              !((psi->ModemMemory.aStoredRegs[usProfile].S[14] & 0x10)>>4), /* PTR 2347*/
              psi->ModemMemory.aStoredRegs[usProfile].PercentA,
              psi->ModemMemory.aStoredRegs[usProfile].PercentC,
              psi->ModemMemory.aStoredRegs[usProfile].PercentE, /* DCR 2534*/
              psi->ModemMemory.aStoredRegs[usProfile].QuoteH,
              psi->ModemMemory.aStoredRegs[usProfile].QuoteN,
              psi->ModemMemory.aStoredRegs[usProfile].QuoteO,
      /***********************************************************************/
      /* Remove -J MTS 3125                                                  */
      /* Readd  -J MTS 2918                                                  */
      /***********************************************************************/
              psi->ModemMemory.aStoredRegs[usProfile].DashJ     /* DCR 2823*/
              );
      break;
    case 3:
      sprintf(pszOutputLine,aachTemplateString[2],
              psi->ModemMemory.aStoredRegs[usProfile].S[0],
              psi->ModemMemory.aStoredRegs[usProfile].S[2],
              psi->ModemMemory.aStoredRegs[usProfile].S[3],
              psi->ModemMemory.aStoredRegs[usProfile].S[4],
              psi->ModemMemory.aStoredRegs[usProfile].S[5],
              psi->ModemMemory.aStoredRegs[usProfile].S[6],
              psi->ModemMemory.aStoredRegs[usProfile].S[7],
              psi->ModemMemory.aStoredRegs[usProfile].S[8],
              psi->ModemMemory.aStoredRegs[usProfile].S[9],
              psi->ModemMemory.aStoredRegs[usProfile].S[10]);
      break;
    case 4:
      sprintf(pszOutputLine,aachTemplateString[3],
              psi->ModemMemory.aStoredRegs[usProfile].S[11],
              psi->ModemMemory.aStoredRegs[usProfile].S[12],
              psi->ModemMemory.aStoredRegs[usProfile].S[24],
              psi->ModemMemory.aStoredRegs[usProfile].S[25],
              psi->ModemMemory.aStoredRegs[usProfile].S[28],
              psi->ModemMemory.aStoredRegs[usProfile].SlashT/10, /*MTS6384*/
              psi->ModemMemory.aStoredRegs[usProfile].S32,
              psi->ModemMemory.aStoredRegs[usProfile].S33,
              //usS32Value,                                        /*MTS6555*/
              //usS33Value,                                        /*MTS6555*/
              psi->ModemMemory.aStoredRegs[usProfile].SV34[0] ); /*S49*/
              /*    Note: \T=90=5400 sec, so S30 could print > 255*/
              /*    I can block that here*/
      break;
    case 5:
      sprintf(pszOutputLine,aachTemplateString[4],
              psi->ModemMemory.aStoredRegs[usProfile].SV34[1],   /*S50*/
              psi->ModemMemory.aStoredRegs[usProfile].SV34[2],   /*S51*/
              psi->ModemMemory.aStoredRegs[usProfile].SV34[3],   /*S52*/
              psi->ModemMemory.aStoredRegs[usProfile].SV34[4],   /*S53*/
              psi->usS60Value );                                 /*S60*/
      break;
    case 6:
      strncpy(pszOutputLine," ",PP_CMD_BUFFER_SIZE);
      break;
  }

 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpVProfileLine exit ulRC %x\n",0);
  return 0;
}




USHORT mwmAmpVCommand(STATEINFO *psi)
{
  BOOL   bBufferFull = FALSE;
  USHORT usParserStatus = 0;
  char   achOutputLine[PP_CMD_BUFFER_SIZE];
  USHORT usProfileNumber = 0;
  USHORT usNumberOfParagraphs = 4;
  USHORT usLinesInParagraph[] = {0,7,7,7,5};

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpVCommand entry\n");
  psi->usCurrentCommand = MWM_AMP_V_CMD;

  while ((psi->usSavedCommandParameter <= usNumberOfParagraphs) && (!bBufferFull))
  {
    /*************************************************************************/
    /* Loop for each of the profiles.                                        */
    /*************************************************************************/
    memset(achOutputLine,0,PP_CMD_BUFFER_SIZE);

    /*************************************************************************/
    /* Switch on the 4 different paragraphs of data to be output             */
    /*************************************************************************/
    switch (psi->usSavedCommandParameter)
    {
      case 1:
      case 2:
      case 3:

        /*********************************************************************/
        /* There are 3 iterations for this loop.  Each iteration dumps the   */
        /* register settings for the current setup, or a saved profile.      */
        /* The following if statements determine which element of the        */
        /* array to dump.                                                    */
        /* if it is the first iteration, we should query the registers into  */
        /* the 3rd element of the array before proceeding.                   */
        /* I realize that these if statements are redundant since I have the */
        /* 3 cases executing the same code above.  However, I didn't want to */
        /* repeat the code for each case....                                 */
        /*********************************************************************/
        if (psi->usSavedCommandParameter == 1)
        {
          usParserStatus = (USHORT)mwmParseQueryModemRegisters(&psi->ModemMemory.aStoredRegs[3]);
          /*******************************************************************/
          /* A bad RC here doesn't actually mean the buffer is full.         */
          /* Instead, both of the loops below depend on this boolean variable*/
          /* I just decided to use it to break out of the command.           */
          /*******************************************************************/
          if (usParserStatus)
            bBufferFull = TRUE;
          usProfileNumber = 3;
        }
        else if (psi->usSavedCommandParameter == 2)
          usProfileNumber = 0;
        else if (psi->usSavedCommandParameter == 3)
          usProfileNumber = 1;

        while ((!bBufferFull) &&
           (psi->usSubCommandInfo < usLinesInParagraph[psi->usSavedCommandParameter]))
        {
          /*******************************************************************/
          /* Break the PP commands into small groups.  This allows us to use */
          /* nearly all of the PP command buffer before requesting another   */
          /* interrupt with a new buffer.                                    */
          /*******************************************************************/
          mwmAmpVProfileLine(psi,usProfileNumber, psi->usSubCommandInfo, achOutputLine);
          /*******************************************************************/
          /* Attempt to echo string to PP Command buffer.                    */
          /*******************************************************************/
          usParserStatus = mwmParseEchoString(psi,achOutputLine);
          if (usParserStatus & MWM_GET_MORE_BUFFERS)
          {
            /*****************************************************************/
            /* There is not enough room left in this PP Command buffer to    */
            /* process this command....                                      */
            /*****************************************************************/
            bBufferFull = TRUE;
          }
          else
            psi->usSubCommandInfo++;
        }
        break;
      case 4:
        while ((!bBufferFull) &&
           (psi->usSubCommandInfo < usLinesInParagraph[psi->usSavedCommandParameter]))
        {
          /*******************************************************************/
          /* Break the PP commands into small groups.  This allows us to use */
          /* nearly all of the PP command buffer before requesting another   */
          /* interrupt with a new buffer.                                    */
          /*******************************************************************/
          mwmAmpVPhoneNumberLine(psi, psi->usSubCommandInfo, achOutputLine);
          /*******************************************************************/
          /* Attempt to echo string to PP Command buffer.                    */
          /*******************************************************************/
          usParserStatus = mwmParseEchoString(psi,achOutputLine);
          if (usParserStatus & MWM_GET_MORE_BUFFERS)
          {
            /*****************************************************************/
            /* There is not enough room left in this PP Command buffer to    */
            /* process this command....                                      */
            /*****************************************************************/
            bBufferFull = TRUE;
          }
          else
            psi->usSubCommandInfo++;
        }
        break;
    }
    /*************************************************************************/
    /* Reset the line number if we are at the end of a paragraph             */
    /*************************************************************************/
    if (psi->usSubCommandInfo == usLinesInParagraph[psi->usSavedCommandParameter])
    {
      psi->usSubCommandInfo = 0;
      psi->usSavedCommandParameter++;
    }
  }

  if (!bBufferFull)
  {
    psi->usSavedCommandParameter = 0;
    psi->usSubCommandInfo = 0;
    psi->usCurrentCommand = MWM_CONTINUE_AT_BUFFER;
    mwmParseSetDRVRSTAT(2);
  }
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpVCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}





USHORT mwmAmpWCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpWCommand entry\n");
  usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                      &psi->usNextATIndex);

  /***************************************************************************/
  /* 9/30/93 New code to delay &W until we get an interrupt 3                */
  /***************************************************************************/
  switch (usParm)
  {
    case 0:
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x24 | psi->usParserMode); /* Ask for INT */
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x1100);
      break;
    case 1:
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x24 | psi->usParserMode); /* Ask for INT 3 */
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x1101);
      break;
    default:
      usParserStatus = MWM_ATCMD_ERROR;
      break;
  }
  /***************************************************************************/
  /* End of new code 9/30/93                                                 */
  /***************************************************************************/
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpWCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmAmpYCommand(STATEINFO *psi)
{
  USHORT usParserStatus = 0;
  USHORT usParm;
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmamp::mwAmpYCommand entry\n");
  /***************************************************************************/
  /* Get argument from AT Command...                                         */
  /***************************************************************************/
  usParm = mwmParseGetArgFromATString(psi->achCommandBuffer,
                                      &psi->usNextATIndex);

  if (usParm < 3)
  {
      psi->ModemMemory.usStartupProfile = usParm;
      /***********************************************************************/
      /* MTS 3538 Ask for IPC 3 to do profile writing rather than posting    */
      /* the message here.                                                   */
      /***********************************************************************/
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x24 | psi->usParserMode); /* Ask for INT 3 */
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x1102);
  }
  else
    usParserStatus = MWM_ATCMD_ERROR;
 MW_SYSLOG_2(TRACE_MWMPW32,"mwmamp::mwAmpYCommand exit usParserStatus %x\n",usParserStatus);
  return usParserStatus;
}




USHORT mwmAmpZCommand(STATEINFO *psi)
{
  char    achDialString[AT_CMD_BUFFER_SIZE];
  USHORT  usParm;
  USHORT  usCharNum = 0;
  USHORT  usNumToStore = 0;
  USHORT  usEndFlag = 0;
  USHORT  usParserStatus = 0;
  USHORT  usCurrentChar = 0;                                        /* PTR 2378*/


  memset(achDialString,0,AT_CMD_BUFFER_SIZE);
  strncpy(achDialString,&psi->achCommandBuffer[psi->usNextATIndex],AT_CMD_BUFFER_SIZE);

  /***************************************************************************/
  /* No matter what happens in this routine, this is the last command that   */
  /* we will accept.                                                         */
  /***************************************************************************/
  usParserStatus = MWM_NO_MORE_COMMANDS;



  /***************************************************************************/
  /* If the Dial String has at least 2 characters in it...                   */
  /***************************************************************************/
  if (strlen(achDialString)>1)                                      /* PTR 2378*/
  {                                                                 /* PTR 2378*/
    /*************************************************************************/
    /* If the second character is an =                                       */
    /*************************************************************************/
    if ((achDialString[1] == '='))                                  /* PTR 2378*/
    {
      /***********************************************************************/
      /* Then the first character is assumed to be the id of the profile     */
      /* to store the number in.                                             */
      /***********************************************************************/
      usNumToStore = achDialString[0]-'0';                          /* PTR 2378*/
      usCurrentChar = 2;                                            /* PTR 2378*/

    }                                                               /* PTR 2378*/
    else                                                            /* PTR 2378*/
    {                                                               /* PTR 2378*/
      /*************************************************************************/
      /* The default phone storage location is 0.                              */
      /*************************************************************************/
      usNumToStore = 0;                                             /* PTR 2378*/
      usCurrentChar = 0;                                            /* PTR 2378*/
      /*************************************************************************/
      /* If the first character is an '=' then skip it...                      */
      /*************************************************************************/
      if ((achDialString[0] == '='))                                /* PTR 2378*/
        usCurrentChar++;                                            /* PTR 2378*/
    }                                                               /* PTR 2378*/
  }                                                                 /* PTR 2378*/

  /***************************************************************************/
  /* MTS 3538 &Z Sometimes failing...                                        */
  /* 7/19/94 New code to delay &Z until we get an interrupt 3                */
  /***************************************************************************/
  switch (usNumToStore)
  {
    case 0:
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x24 | psi->usParserMode); /* Ask for INT 3 */
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x1103);
      break;
    case 1:
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x24 | psi->usParserMode); /* Ask for INT 3 */
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x1104);
      break;
    case 2:
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x24 | psi->usParserMode); /* Ask for INT 3 */
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x1105);
      break;
    case 3:
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x24 | psi->usParserMode); /* Ask for INT 3 */
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x1106);
      break;

    default:
      return (usParserStatus | MWM_ATCMD_ERROR);
  }

  memset(psi->ModemMemory.aachStoredNumbers[usNumToStore],0,AT_CMD_BUFFER_SIZE);




  do
  {
    usParm = (USHORT) achDialString[usCurrentChar++];               /* PTR 2378*/

    if (isdigit(usParm))
    {
      psi->ModemMemory.aachStoredNumbers[usNumToStore][usCharNum] = (char)usParm;
      usCharNum++;
    }
    else
    {
      switch (usParm)
      {
        case 'A':
        case 'B':
        case 'C':
        case 'D':
        case '*':
        case '#':
        case 'P':
        case 'R':
        case 'T':
        case 'W':
        case ',':
        case '!':
        case '@':
        case ';':
          psi->ModemMemory.aachStoredNumbers[usNumToStore][usCharNum] = (char)usParm;
          usCharNum++;
          break;
        case 0  :
          usEndFlag=1; /* end of at buffer */
          break;
        default :  /* ignore invalid characters */
          break;
      } /* endswitch */
    }
  } while (!usEndFlag);


  return usParserStatus;
}




