/* ============================================================
 * File  : playerencap.cpp
 * Author: Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date  : 2003-12-22
 * Description : external player control, player configured in
 *               player.xml
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include <iostream>
#include <qdict.h>
#include <qregexp.h>
#include <qdom.h>
#include <qfile.h>


#include "playerencap.h"

static const QString defaultRegExp = "^(.*)$"; // default regexp string
static QString emptyStr = "";         // unknown property display value

using namespace std;

// ----------------------------------------------------------------------

    class StreamParameter
    {
      public:

        StreamParameter();
        ~StreamParameter();

        int setValue(const QString& val);
        void reset();
        QString getValueByRegExp(QRegExp expr, bool clear);
        void setReportStatus(int status);
        void setAppend(bool append); // store multiple values
        void setClear(bool clear); // allow clear after write if requested

      private:
        QString value;
        QString idleValue;
        int reportStatus;
        bool doAppend;
        bool doClear;
    };


    StreamParameter::StreamParameter()
    {
        idleValue = "";
        doAppend = false;
        doClear  = false;
        value = idleValue;
        reportStatus = 0;
    }

    StreamParameter::~StreamParameter() {}

    void StreamParameter::setReportStatus(int status)
    {
        reportStatus = status;
    }

    void StreamParameter::setAppend(bool append) // store multiple values
    {
       doAppend = append;
    }
    
    void StreamParameter::setClear(bool clear) // store multiple values
    {
       doClear = clear;
    }
    
    int StreamParameter::setValue(const QString& val)
    {
        if (doAppend)
        {
          if (value != idleValue) 
            value += "\n";
          value += val;
        }
        else
          value = val;
        return reportStatus;
    }

    void StreamParameter::reset()
    {
        value = idleValue;
    }

    QString StreamParameter::getValueByRegExp(QRegExp expr, bool clear)
    {
        if ( expr.search( value ) > -1 )
        {
          if (clear && doClear) reset();
          return expr.cap(1);
        }
          else
        {
          if (clear && doClear) reset();
          return QString::null;
        }
    }


/* ----------------------------------------------------------------------
    StreamParameters and StreamProperties are configured in player.xml
    ( <filter><item> and <filter><item><properties> )

    A StreamParameter represents text that can be found in the text the
    player writes to stdout. The text must be located between start of line
    and a regexp separator given in player.xml ( <separator> ).

    A StreamProperty represents a value that can be grepped in the text
    after the separator. The regexp for this grep is given in player.xml
    ( <filter><item><properties><value> )

    If the StreamProperty grep is not succesful, the StreamProperty returns
    a null string. If the grep was succesful the result is returned by getValue
    unless this result is not an empty string and a message value is given in
    player.xml ( <filter><item><properties><message> ). In that case message is
    returned.
   ----------------------------------------------------------------------
*/
    class StreamProperty
    {
      public:

        StreamProperty( QString& regExp,
                        QString& defmsg,
                        StreamParameter* streamParameter,
                        QString& area,
                        QString& title
                        );
        ~StreamProperty();

        QString& getValue(bool clear);
        QString& getArea();
        QString& getTitle();
        const QString getRegExp();  // debug output

      private:
        QRegExp myRegExp;
        QString lastValue;
        QString message;
        QString gui, title;
        bool gotmessage;
        StreamParameter* myStreamParameter;
    };



    StreamProperty::StreamProperty(
                    QString& regExp,
                    QString& defmsg,
                    StreamParameter* streamParameter,
                    QString& area,
                    QString& title
                    )
    {
        myRegExp = QRegExp(regExp);
        //myRegExp.setMinimal(true);
        message = defmsg;
        gui = area;
        this->title = title;
        
        if ( message != "" )
          gotmessage = true;
        else
          gotmessage = false;

        myStreamParameter = streamParameter;
    }

    StreamProperty::~StreamProperty() {}

    const QString StreamProperty::getRegExp()
    {
        return myRegExp.pattern();
    }

    // get regexp result or message if message tag is set in player.xml
    QString& StreamProperty::getValue(bool clear)
    {
        lastValue = myStreamParameter->getValueByRegExp(myRegExp, clear);
        if ( gotmessage && lastValue != QString::null && lastValue != "" )
          lastValue = message;
        
        return lastValue;
    }

    QString& StreamProperty::getArea()
    {
        return gui;
    }
    
    QString& StreamProperty::getTitle()
    {
        return title;
    }
// ----------------------------------------------------------------------

PlayerEncap::PlayerEncap()
{
    parseMap.setAutoDelete(true);
    propMap.setAutoDelete(true);
    loadPlayerRegExp();
}

PlayerEncap::~PlayerEncap() {}

void PlayerEncap::reset()
{
    QDictIterator<StreamParameter>i(parseMap);
    for (; i.current(); ++i ) i.current()->reset();
}

QString& PlayerEncap::getStreamProperty(const QString& name, bool clear)
{
    StreamProperty* sprop = propMap[name];
    if ( sprop )
      return sprop->getValue(clear);
    else
    {
      cerr << "error: filter property \""
          << name
          << "\" not loaded from player.xml"
          << endl;
      return emptyStr;
    }
}

QString& PlayerEncap::getStreamPropertyArea(const QString& name)
{
    StreamProperty* sprop = propMap[name];
    if ( sprop )
      return sprop->getArea();
    else
      return emptyStr;
}

QString& PlayerEncap::getStreamPropertyTitle(const QString& name)
{
    StreamProperty* sprop = propMap[name];
    if ( sprop )
      return sprop->getTitle();
    else
      return emptyStr;
}

bool PlayerEncap::checkStreamPropertyConfigured(const QString& name)
{
    StreamProperty* sprop = propMap[name];
    if ( sprop )
      return true;
    else
      return false;
}


QString& PlayerEncap::getPlayerSys(const QString& name)
{
    QString& value = playerSys[name];
    if ( value == "" )
      cerr << "error: player system \""
          << name
          << "\" not loaded from player.xml"
          << endl;
    return value;
}

QString& PlayerEncap::getPlayerCmd(const QString& name)
{
    QString& value = playerCmd[name];
    if ( value == "" )
      cerr << "warning: player command \""
          << name
          << "\" not loaded from player.xml"
          << endl;
    return value;
}

QString& PlayerEncap::getCurrentCacheUsage()
{
    if ( streamCacheProperty )
      return streamCacheProperty->getValue(false);
    else
      return emptyStr;
}

int PlayerEncap::pushParameter(QString& label, QString& value)
{
    int result = 0;
    StreamParameter* parameter = parseMap[label];
    if ( parameter )
    {
      result = parameter->setValue(value);
    }
    return result;
}


QDomElement PlayerEncap::getFirstElement(QDomElement base, const QString& tagname)
{
  QDomNodeList list = base.elementsByTagName(QString::fromLatin1(tagname));
  if ( list.count() > 0 )
    return list.item(0).toElement();
  else
    return QDomElement();
}

int PlayerEncap::getStatusFromStr(QString& status) // dirty
{
  int stat = 0;
  if ( status == "playing" ) stat = 5;
  if ( status == "paused" ) stat = 6;
  if ( status == "buffering" ) stat = 4;
  if ( status == "videoinit" ) stat = 3;
  return stat;
}

void PlayerEncap::fillFilterMap(QDomElement& parentNode)
{
    int stat;
    QDomNode item;
    QString name, value;
    QString label, status, message, options, area, title;
    bool noclear, noappend;
    QDomElement props;
    QDomNodeList proplist;

    QDomNodeList list = parentNode.childNodes();
    for (unsigned int i = 0; i < list.count(); i++)
    {
        item    = list.item(i);
        label   = item.namedItem(QString("label")).toElement().text();
        status  = item.namedItem(QString("statuschange")).toElement().text();
        props   = item.namedItem(QString("properties")).toElement();
        options = item.namedItem(QString("options")).toElement().text(); // noclear noappend

        if ( !parseMap.find(label) )
          parseMap.insert( label, new StreamParameter() );

        if ( status != QString::null )
        {
          stat = getStatusFromStr( status );
          if ( stat != 0 )
            parseMap[label]->setReportStatus( stat );
          else
            cerr << "invalid statuschange tag value: "
                 << status << " in player.xml" << endl;
        }

		noclear  = false; // default = clear: value is cleared after first gui read
		noappend = false; // default = append: events between gui reads will append to multiple lines

		if ( options != QString::null )
		{
			if ( options.find("noclear") != -1 )
			noclear = true;

			if ( options.find("noappend") != -1 )
			noappend = true;
		}
        
        if ( !props.isNull() )
        {
           proplist = props.childNodes();
           for (unsigned int j = 0; j < proplist.count(); j++)
           {
              item    = proplist.item(j);
              name    = item.namedItem(QString("name")).toElement().text();
              value   = item.namedItem(QString("value")).toElement().text();
              message = item.namedItem(QString("message")).toElement().text();
              area    = item.namedItem(QString("area")).toElement().text();
              title   = item.namedItem(QString("title")).toElement().text();

              if ( message == QString::null )
                message = "";

              if ( !propMap.find(name) )
              {
                if (name.left(17) == "StreamCustomEvent")
                  parseMap[label]->setAppend(!noappend);
                  parseMap[label]->setClear(!noclear);
                propMap.insert(name, new StreamProperty( value, message, parseMap[label], area, title));
              }
           }
           if ( proplist.count() == 0 )
              cerr << "warning: player.xml filter label "
                   << label
                   << "has no property assigned" << endl;
        }
           else
              cerr << "warning: player.xml filter label "
                   << label
                   << "has no property assigned" << endl;
    }
}

void PlayerEncap::fillMap(QMap<QString,QString>& parseMap, QDomElement& parentNode)
{
    QDomNode item;
    QString name, value;

    QDomNodeList list = parentNode.childNodes();
    for (unsigned int i = 0; i < list.count(); i++)
    {
      item  = list.item(i);
      name  = item.namedItem(QString("name")).toElement().text();
      value = item.namedItem(QString("value")).toElement().text();
      if ( value == QString::null )
        value = "";
      if ( name != QString::null && name != "" )
        parseMap[name] = value;
      else
        cerr << "missing name tag in item " << i
             << " of player." << parentNode.tagName() << "-block" << endl;
    }
}

void PlayerEncap::loadPlayerRegExp()
{
    QDomDocument domDoc;
    QDomElement parent, child;

    QString filename = QString(PREFIX"/share/"SUBPATH"/player.xml");
    QFile xmlFile(filename);

    if (!xmlFile.exists() || !xmlFile.open(IO_ReadOnly))
    {
        cerr << "Cannot open player.xml" << endl;
        return;
    }

    if (!domDoc.setContent(&xmlFile))
    {
        cerr << "parse error player.xml" << endl;
        return;
    }

  // get player output filters
    parent = getFirstElement(domDoc.documentElement(), "filters");
    if ( !parent.isNull() )
      fillFilterMap(parent);
    else
      cerr << "missing filter section in player.xml" << endl;

  // get player and command line parameters
    parent = getFirstElement(domDoc.documentElement(), "player");
    if ( !parent.isNull() )
    {
      child = getFirstElement(parent, "system");
      if ( !child.isNull() )
        fillMap( playerSys, child );

      child = getFirstElement(parent, "custom");
      if ( !child.isNull() )
        fillMap( playerParam, child );

      child = getFirstElement(parent, "command");
      if ( !child.isNull() )
        fillMap( playerCmd, child );
    }
      else
        cerr << "missing player section in player.xml" << endl;

    xmlFile.close();

/* debug

    StringMap::Iterator i;
    for ( i = playerSys.begin(); i != playerSys.end(); ++i )
    {
       cout << "playerSys: " << i.key() << "=" << i.data() << endl;
    }

    for ( i = playerParam.begin(); i != playerParam.end(); ++i )
    {
       cout << "playerParam: " << i.key() << "=" << i.data() << endl;
    }

    for ( i = playerCmd.begin(); i != playerCmd.end(); ++i )
    {
       cout << "playerCmd: " << i.key() << "=" << i.data() << endl;
    }

    QDictIterator<StreamParameter>it(parseMap);
    for (; it.current(); ++it )
      cout << "parseMap: " << it.currentKey() << " status: "
           << it.current()->setValue("") << endl;

    QDictIterator<StreamProperty>iter(propMap);
    for (; iter.current(); ++iter )
      cout << "propMap: " << iter.currentKey() << " regexp: "
           << iter.current()->getRegExp() << endl;

*/ // end debug

    // defaults
    if ( playerSys["separator"] == "" ) playerSys["separator"] = "[:|=]";
    if ( playerSys["player"]    == "" ) playerSys["player"]    = "mplayer";
    if ( playerSys["window"]    == "" ) playerSys["window"]    = "-wid";
    if ( playerSys["scale"]     == "" ) playerSys["scale"]     = "-xy";

    streamCacheProperty = propMap["StreamPlayCache"];

}

/*

player.xml info

mplayer output:

Playing /mnt/media/Video/Videos/misc/08829.avi
AVI file format detected.
VIDEO:  [DIVX]  320x240  24bpp  29.970 fps  366.0 kbps (44.7 kbyte/s)
==========================================================================
Opening audio decoder: [mp3lib] MPEG layer-2, layer-3
MP3lib: init layer2&3 finished, tables done
AUDIO: 22050 Hz, 2 ch, 16 bit (0x10), ratio: 7000->88200 (56.0 kbit)
Selected audio codec: [mp3] afm:mp3lib (mp3lib MPEG layer-2, layer-3)
==========================================================================
ID_FILENAME=/mnt/media/Video/Videos/misc/08829.avi
ID_VIDEO_FORMAT=DIVX
ID_VIDEO_BITRATE=366032
ID_VIDEO_WIDTH=320
ID_VIDEO_HEIGHT=240
ID_VIDEO_FPS=29.970
ID_VIDEO_ASPECT=0.0000
ID_AUDIO_CODEC=mp3
ID_AUDIO_FORMAT=85
ID_AUDIO_BITRATE=56000
ID_AUDIO_RATE=22050
ID_AUDIO_NCH=2
ID_LENGTH=1413
vo: X11 running at 1280x1024 with depth 16 and 16 bpp (":0.0" => local display)
==========================================================================
Opening video decoder: [ffmpeg] FFmpeg's libavcodec codec family
Selected video codec: [ffodivx] vfm:ffmpeg (FFmpeg MPEG-4)
==========================================================================


*/


