/* ============================================================
 * File        : storage.cpp
 * Author      : Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date        : 2004-01-21
 * Description : repository access db, file or web
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include <iostream>

#include <qnetwork.h>
#include <qsqlquery.h>
#include <qdatetime.h>
#include <qdir.h>
#include <qstringlist.h>
#include <qdict.h>

#include <qapplication.h>
#include <qurl.h>
#include <qptrlist.h>

#include "storage.h"

using namespace std;


//----------------------- Storage ---------------------------

Storage::Storage(QString table, QString name,
                 int key_pri, int key_mul1, int key_mul2) : QObject()
{
    currentStorage = 0;
    oldStorage = 0;

    defaultTable = table;
    defaultName  = name;

    this->key_pri  = key_pri;
    this->key_mul1 = key_mul1;
    this->key_mul2 = key_mul2;
}

Storage::~Storage()
{
  if (currentStorage)
    delete currentStorage;
  if (oldStorage)
    delete oldStorage;
}


bool Storage::storageReadOnly()
{
  if (currentStorage)
    return currentStorage->storageReadOnly();
  else
    return false;
}

QString Storage::getStorageDescription()
{
  if (currentStorage)
    return currentStorage->getStorageDescription();
  else
    return TARGET" storage warning: no storage selected";
}

Storage::AccessType Storage::getAccessType()
{
  if (currentStorage)
    return (AccessType)currentStorage->getAccessType();
  else
    return none;
}

QString Storage::getStorageName()
{
  if (currentStorage)
    return currentStorage->getStorageName();
  else
    return "warning: no storage selected";
}

bool Storage::getSynchronized()
{
  if (currentStorage)
    return currentStorage->getSynchronized();
  else
    return false;
}

QString Storage::getLastError()
{
  if (currentStorage)
    return currentStorage->getLastError();
  else
    return "";
}

void Storage::resetRecordList()
{
  if (currentStorage)
    currentStorage->resetRecordList();
}

bool Storage::getNextRecord(ValueList& values)
{
  if (currentStorage)
    return currentStorage->getNextRecord(currentStorage->getListHandler(), values);
  else
    return false;
}

void Storage::linkNewStorage(GenStorage *storage)
{
    currentStorage = storage;

    connect(
             currentStorage,
             SIGNAL(storageEvent(int, int, bool)),
             this,
             SIGNAL(storageEvent(int, int, bool))
           );

    connect(
             currentStorage,
             SIGNAL(recordInserted(ChangedRecord*)),
             this,
             SIGNAL(recordInserted(ChangedRecord*))
           );

    connect(
             currentStorage,
             SIGNAL(recordUpdated(ChangedRecord*)),
             this,
             SIGNAL(recordUpdated(ChangedRecord*))
           );

    connect(
             currentStorage,
             SIGNAL(recordRemoved(ChangedRecord*)),
             this,
             SIGNAL(recordRemoved(ChangedRecord*))
           );

}

bool Storage::selectDbStorage(int ident, QString name, QString dbDriver,
                              QString hostName, unsigned int port, QString dbName, QString table,
                              QString login, QString password)
{
    if (oldStorage) delete oldStorage;
    oldStorage = currentStorage;

    DatabaseStorage* databaseStorage =
      new DatabaseStorage("", name, database, key_pri, key_mul1, key_mul2);

    linkNewStorage(databaseStorage);

    return databaseStorage->openDatabaseStorage(ident, dbDriver, hostName, port, dbName, table,
                                            login, password);
}

bool Storage::selectDefaultDb(int ident)
{
    if (oldStorage) delete oldStorage;
    oldStorage = currentStorage;

    DatabaseStorage* databaseStorage =
      new DatabaseStorage(defaultTable, defaultName, database, key_pri, key_mul1, key_mul2);

    linkNewStorage(databaseStorage);

    return databaseStorage->openDefaultDb(ident);
}

bool Storage::selectFileStorage(int ident, QString name, QString fileName)
{
    if (oldStorage) delete oldStorage;
    oldStorage = currentStorage;


    FileStorage* fileStorage = new FileStorage(name, file, key_pri, key_mul1, key_mul2);

    linkNewStorage(fileStorage);

    return fileStorage->openFileStorage(ident, fileName);
}

void Storage::selectWebStorage(int ident, QString name, QString url, QString login, QString password)
{

    if (oldStorage) delete oldStorage;
    oldStorage = currentStorage;

    WebStorage* webStorage =
      new WebStorage(name, web, key_pri, key_mul1, key_mul2);

    linkNewStorage(webStorage);

    return webStorage->openWebStorage(ident, url, login, password);
}


bool Storage::loadList(int ident, QString& error)
{
    if (currentStorage)
    {
       if (oldStorage)
       {
          delete oldStorage;
          oldStorage = 0;
       }
       return currentStorage->loadList(ident, error);
    }
    else
    {
       error = "no storage active";
       return false;
    }
}


// - copy from resource1 to resource2
// - update file resource after removal
bool Storage::storeList(int ident, QString& error)
{
   if (!currentStorage)
   {
     error = "no storage active";
     return false;
   }

   if (!oldStorage)
   {
      error = "no previous storage to copy from";
      return false;
   }

   return currentStorage->storeList(ident, oldStorage->getListHandler(), error);
}

bool Storage::insertRecord(int ident, ValueList& values, QString& error)
{
   if (currentStorage)
     return currentStorage->insertRecord(ident, values, error);
   else
   {
     error = "no storage active";
     return false;
   }
}

bool Storage::updateRecord(int ident, ValueList& oldValues, ValueList& values, QString& error)
{

   if (currentStorage)
     return currentStorage->updateRecord(ident, oldValues, values, error);
   else
   {
     error = "no storage active";
     return false;
   }
}

bool Storage::removeRecord(int ident, ValueList values, QString& error)
{
   if (currentStorage)
     return currentStorage->removeRecord(ident, values, error);
   else
   {
     error = "no storage active";
     return false;
   }
}


