/* ============================================================
 * File  : streamconfig.cpp
 * Author: Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date  : 2004-03-08
 * Description : mythtv compatibility parts copied from mythnews
 *               0.13 plugin source, mythnews copyright Renchi Raju
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// development snapshot, many todo's

#include <iostream>
#include <qlayout.h>
#include <qcursor.h>
#include <qregexp.h>

#include "storageconfig.h"

using namespace std;

//   Needs cleanup

// ------------------- Storage Edit group box --------------------

class StorageEditGroup : public EditGroup
{

  public:
    StorageEditGroup(char* title, StorageConfig* parent) : EditGroup(title, parent)
    {
        boxAddTitle  = "Add storage";
        boxEditTitle = "Edit storage";
        boxIdleTitle = "";

        storageConfig = parent;

        databaseLabels.append("Storage name");
        databaseLabels.append("Host");
        databaseLabels.append("Port");
        databaseLabels.append("Database name");
        databaseLabels.append("Table name");
        databaseLabels.append("Login");
        databaseLabels.append("Password");

        fileLabels.append("Storage name");
        fileLabels.append("File");

        webLabels.append("Storage name");
        webLabels.append("Url");
        webLabels.append("Login");
        webLabels.append("Password");

        buildGroup(7);
        setStatus(idle);
    }

    void setIdle()
    {
        QLineEdit *editbox;
        QLabel *label;

        for ( int i = 0; i < 7; i++ )
        {
          editbox = getLineEdit( "edit" + QString::number(i) );
          editbox->setText("");
          editbox->setEnabled(false);
          label = getLabel( "label" + QString::number(i) );
          label->setText( " " );
          setStatus(idle);
        }
    }

    void initAddRecord(const QString typeStr)
    {
       newTypeStr = typeStr;
       if ( typeStr == "database"  ) newType = ReposStorage::database;
       else if ( typeStr == "file" ) newType = ReposStorage::file;
       else if ( typeStr == "web"  ) newType = ReposStorage::web;
       initFields(0);
    }

    void initFields(QListViewItem* item)
    {
        QLineEdit *editbox;
        QLabel *label;
        ValueList *lvalues;
        RepositoryItem* repos = dynamic_cast<RepositoryItem*>(item);

        if ( repos ) newType = repos->type;

        switch ( newType )
        {
          case ReposStorage::database:
            lvalues = &databaseLabels;
            newTypeStr = "database";
            boxAddTitle  = "Add database storage";
            boxEditTitle = "Edit database storage";
          break;
          case ReposStorage::file:
            lvalues = &fileLabels;
            newTypeStr = "file";
            boxAddTitle  = "Add file storage";
            boxEditTitle = "Edit file storage";
          break;
          case ReposStorage::web:
            lvalues = &webLabels;
            newTypeStr = "web";
            boxAddTitle  = "Add web storage";
            boxEditTitle = "Edit web storage";
          break;
          default:
            cerr << "Error: storage has unknown Access type" << endl;
            exit(-1);
          break;
        }

        for ( int i = 0; i < boxCount; i++ )
        {
          editbox = getLineEdit( "edit" + QString::number(i) );

          if (repos)
            editbox->setText(repos->values[i + 2]);
          else
            editbox->setText( "" );

          label = getLabel( "label" + QString::number(i) );

          if ( i < (int)lvalues->count() )
          {
            label->setText( lvalues->at(i) );
            editbox->setEnabled(true);
          }
            else
          {
            label->setText( "" );
            editbox->setEnabled(false);
          }
        }

        linkedItem = item;

        if (repos)
          setStatus(edit);
        else
          setStatus(add);
    }

    void commitFields()
    {
      ValueList values;
      RepositoryItem *repos = dynamic_cast<RepositoryItem*>(linkedItem);

      if (status == edit && !repos)
      {
         cerr << "error: could not find item for update" << endl;
         return;
      }

      if (repos)
        values.append( repos->values[0] );
      else
        values.append( "-" );

      values.append( newTypeStr );

      for ( int i = 0; i < boxCount; i++ )
        values.append( getLineEdit("edit" + QString::number(i) )->text() );

      if ( status == add )
      {
        storageConfig->createStorage( newType, values );
      }

      else if ( status == edit && dynamic_cast<RepositoryItem*>(linkedItem))
      {
        storageConfig->updateStorage(dynamic_cast<RepositoryItem*>(linkedItem), values );
      }
    }

    void removeLinked()
    {
      if ( status == edit && dynamic_cast<RepositoryItem*>(linkedItem) )
      storageConfig->removeStorage( dynamic_cast<RepositoryItem*>(linkedItem) );
    }

    private:
      StorageConfig * storageConfig;
      ValueList databaseLabels;
      ValueList fileLabels;
      ValueList webLabels;
      ReposStorage::AccessType newType;
      QString newTypeStr;

};

//--------------------- Stream Storage tab ----------------------

ReposStorage::AccessType strToAccessType(const QString& str)
{
    if ( str == "file" )
      return ReposStorage::file;
    if ( str == "database" )
      return ReposStorage::database;
    if ( str == "web" )
      return ReposStorage::web;

    return ReposStorage::none;
}


StorageConfig::StorageConfig(BaseParent *parent, const char *name,
                            StreamStorage *streamStorage, ReposStorage *storageBin)
              : BaseWidget(parent, name)
{
    storage = storageBin;
    streams = streamStorage;
    activeItem = 0;

    connect (
             storage,
             SIGNAL(recordInserted(ChangedRecord*)),
             this,
             SLOT(slotRecordInserted(ChangedRecord*))
            );

    connect (
             storage,
             SIGNAL(recordUpdated(ChangedRecord*)),
             this,
             SLOT(slotRecordUpdated(ChangedRecord*))
            );

    connect (
             storage,
             SIGNAL(recordRemoved(ChangedRecord*)),
             this,
             SLOT(slotRecordRemoved(ChangedRecord*))
            );

    connect (
               streams,
               SIGNAL( storageEvent(int, int, bool ) ),
               this,
               SLOT  ( slotStorageEvent(int, int, bool ) )
             );

    setupView();

    // Fill storage picklist
    storage->resetRecordList();
    ValueList values(9);
    ReposStorage::AccessType type;

    while ( storage->getNextRecord(values) )
    {
       if (values.size() < 9 ) values.resize( 9, "" );

       type = strToAccessType(values[1]);

       RepositoryItem *item = new RepositoryItem(m_storageList, type);
       item->setText(0, values[2]);
       item->setText(1, values[1]);
       item->setText(2, values[0]);

       item->values = values;
    }

    setCursor(QCursor(Qt::ArrowCursor));

    // show mouse pointer
    QApplication::setOverrideCursor(QCursor(Qt::ArrowCursor));
}

StorageConfig::~StorageConfig()
{
    QApplication::restoreOverrideCursor();
}

void StorageConfig::setupView()
{
    QFontMetrics metric( font() );
    int maxHeight = metric.height() + 2;
    
    int hmult = 1;
    QVBoxLayout *vbox = new QVBoxLayout(this, (int)(hmult*10));

    errorLabel = new QLabel(this);
    errorLabel->setBackgroundOrigin(QWidget::WindowOrigin);
    errorLabel->setFrameStyle(QFrame::Box + QFrame::Raised);
    errorLabel->setMargin(int(hmult*2));
    errorLabel->setText(" ");
    errorLabel->setMaximumHeight(maxHeight);
    vbox->addWidget(errorLabel);
    // List Views

    QHBoxLayout *hbox = new QHBoxLayout(vbox);
    QVBoxLayout *leftbox = new QVBoxLayout(hbox, 5);

    m_storageList = new QListView(this);
    m_storageList->addColumn("Storage name");
    m_storageList->addColumn("type");
    m_storageList->addColumn("active");
    m_storageList->setColumnAlignment(2, AlignCenter);

    m_storageList->setRootIsDecorated(true);
    leftbox->addWidget(m_storageList);

    QHBoxLayout *butbox = new QHBoxLayout(leftbox);

    ConfigButton *but;
    but = new ConfigButton("&New Db", this);
    but->setEnabled(true);
    connect(but, SIGNAL(clicked()), this, SLOT(slotNewDbClicked()));
    butbox->addWidget(but);
    but = new ConfigButton("&New File", this);
    but->setEnabled(true);
    connect(but, SIGNAL(clicked()), this, SLOT(slotNewFileClicked()));
    butbox->addWidget(but);
    but = new ConfigButton("&New Web", this);
    but->setEnabled(true);
    connect(but, SIGNAL(clicked()), this, SLOT(slotNewWebClicked()));
    butbox->addWidget(but);

    // Edit boxes

    storageEditGroup = new StorageEditGroup("Edit Storage", this);
    hbox->addWidget(storageEditGroup);

    connect(m_storageList, SIGNAL(selectionChanged(QListViewItem*)),
            this,          SLOT(itemSelected(QListViewItem*)));
}

void StorageConfig::itemSelected(QListViewItem* item)
{
  storageEditGroup->initFields(item);
}

void StorageConfig::slotStorageEvent(int ident, int eventType, bool error)
{
  QString errorMsg;

  if ( ident != ident_storageconfig ) return;

  if ( error )
  {
    reportMessage( "storage error: " + streams->getLastError(), true );
    return;
  }

  switch (eventType)
  {
    case ReposStorage::selected:
       //streams->loadList(ident_storageconfig, errorMsg );
       break;
    case ReposStorage::loaded:
       //if ( !error ) slotListLoaded();
       break;
  }
}

void StorageConfig::slotRecordInserted(ChangedRecord *rec)
{
   RepositoryItem *item = new RepositoryItem(m_storageList, strToAccessType(rec->values[1]) );
   item->setText(0, rec->values[2]);
   item->setText(1, rec->values[1]);
   item->setText(2, rec->values[0]);

   for ( unsigned int i = 0; i < rec->values.count(); i++ )
     item->values.append(rec->values[i]);

   if (item->values.size() < 9 ) item->values.resize( 9, "" );

   m_storageList->setSelected( item, true );
}

void StorageConfig::slotRecordUpdated(ChangedRecord *rec)
{
    if (rec->error)
    {
      if ( rec->ident == ident_storageconfig )
        reportMessage( storage->getLastError(), true );
      return;
    }

    QListViewItem* item = m_storageList->firstChild();

    while ( item && item->text(0) != rec->oldValues[2] )
      item = item->nextSibling();

    RepositoryItem *repos = dynamic_cast<RepositoryItem*>(item);
    if (!repos)
    {
      cerr << "cannot find " << rec->oldValues[2] << " repository "
           << rec->oldValues[2] << endl;
      return;
    }

   repos->setText(0, rec->values[2]);
   repos->setText(1, rec->values[1]);
   repos->setText(2, rec->values[0]);

   for ( unsigned int i = 0; i < rec->values.count(); i++ )
     repos->values[i] = rec->values[i];

   m_storageList->setSelected( repos, true );
}

void StorageConfig::slotRecordRemoved(ChangedRecord *rec)
{
    if (rec->error)
    {
      if ( rec->ident == ident_storageconfig )
        reportMessage( storage->getLastError(), true );
      return;
    }

    QListViewItem* item = m_storageList->firstChild();

    while ( item && item->text(0) != rec->values[2] )
      item = item->nextSibling();

    RepositoryItem *repos = dynamic_cast<RepositoryItem*>(item);
    if (!repos)
    {
      cerr << "cannot find " << rec->values[2] << " repository "
           << rec->values[2] << endl;
      return;
    }

    delete(repos);

    storageEditGroup->setIdle();
}

void StorageConfig::slotNewDbClicked()
{
    storageEditGroup->initAddRecord("database");
}

void StorageConfig::slotNewFileClicked()
{
    storageEditGroup->initAddRecord("file");
}

void StorageConfig::slotNewWebClicked()
{
    storageEditGroup->initAddRecord("web");
}


void StorageConfig::reportMessage(const QString& error, bool bad)
{
  errorLabel->setText(error);
  if ( bad )
    errorLabel->setPaletteForegroundColor( red );
  else
    errorLabel->setPaletteForegroundColor( black );
}

void StorageConfig::createStorage( ReposStorage::AccessType type, ValueList& values )
{
   switch ( type )
   {
     case ReposStorage::database:
       values.resize(r_count_db, "");
     break;
     case ReposStorage::file:
       values.resize(r_count_file, "");
     break;
     case ReposStorage::web:
       values.resize(r_count_web, "");
     break;
     default:
       cerr << "unknown access type" << endl;
     break;
   }

   QString error;
   if ( !storage->insertRecord(ident_storageconfig, values, error) )
      reportMessage(error, true);
}

void StorageConfig::updateStorage( RepositoryItem* item, ValueList& values )
{
   switch ( item->type )
   {
     case ReposStorage::database:
       values.resize(r_count_db);
     break;
     case ReposStorage::file:
       values.resize(r_count_file);
     break;
     case ReposStorage::web:
       values.resize(r_count_web);
     break;
     default:
       cerr << "unknown access type" << endl;
     break;
   }

   QString error;
   if ( item && !storage->updateRecord(ident_storageconfig, item->values, values, error) )
      reportMessage(error, true);
}

void StorageConfig::removeStorage(RepositoryItem* item)
{
   QString error;
   if ( item && !storage->removeRecord(ident_storageconfig, item->values, error) )
      reportMessage(error, true);
}

