/* -*- Mode: c++ -*-
 * vim: set expandtab tabstop=4 shiftwidth=4:
 *
 * Original Project
 *      MythTV      http://www.mythtv.org
 *
 * Copyright (c) 2008 Daniel Kristjansson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 *
 */

// Qt headers
#include <QCoreApplication>

// MythTv headers
#include "channelscanner_cli.h"
#include "channelscan_sm.h"
#include "channelimporter.h"

#define LOC      QString("ChScanCLI: ")
#define LOC_WARN QString("ChScanCLI, Warning: ")
#define LOC_ERR  QString("ChScanCLI, Error: ")

ChannelScannerCLI::ChannelScannerCLI(bool doScanSaveOnly, bool promptsOk) :
    done(false), onlysavescan(doScanSaveOnly), interactive(promptsOk),
    status_lock(false), status_complete(0), status_snr(0),
    status_text(""), status_last_log("")
{
}

ChannelScannerCLI::~ChannelScannerCLI()
{
}

void ChannelScannerCLI::HandleEvent(const ScannerEvent *scanEvent)
{
    if ((scanEvent->type() == ScannerEvent::ScanComplete) ||
        (scanEvent->type() == ScannerEvent::ScanShutdown))
    {
        cout<<endl;

        if (scanEvent->type() == ScannerEvent::ScanShutdown)
            cerr<<"HandleEvent(void) -- scan shutdown"<<endl;
        else
            cerr<<"HandleEvent(void) -- scan complete"<<endl;

        ScanDTVTransportList transports;
        if (sigmonScanner)
        {
            sigmonScanner->StopScanner();
            transports = sigmonScanner->GetChannelList();
        }

        Teardown();

        if (!transports.empty())
            Process(transports);

        done = true;
        QCoreApplication::exit(0);
    }
    else if (scanEvent->type() == ScannerEvent::AppendTextToLog)
        status_last_log = scanEvent->strValue();
    else if (scanEvent->type() == ScannerEvent::SetStatusText)
        status_text = scanEvent->strValue();
    else if (scanEvent->type() == ScannerEvent::SetStatusTitleText)
        ;
    else if (scanEvent->type() == ScannerEvent::SetPercentComplete)
        status_complete = scanEvent->intValue();
    else if (scanEvent->type() == ScannerEvent::SetStatusRotorPosition)
        ;
    else if (scanEvent->type() == ScannerEvent::SetStatusSignalLock)
        status_lock = scanEvent->intValue();
    else if (scanEvent->type() == ScannerEvent::SetStatusSignalToNoise)
        status_snr = scanEvent->intValue() / 65535.0;
    else if (scanEvent->type() == ScannerEvent::SetStatusSignalStrength)
        ;

    //cout<<"HERE<"<<print_verbose_messages<<">"<<endl;
    QString msg;
    if (VERBOSE_LEVEL_NONE || VERBOSE_LEVEL_CHECK(VB_CHANSCAN))
    {
        msg.sprintf("%3i%% S/N %3.1f %s : %s (%s) %20s",
                    status_complete, status_snr,
                    (status_lock) ? "l" : "L",
                    status_text.toAscii().constData(),
                    status_last_log.toAscii().constData(), "");
    }
    //cout<<msg.toAscii().constData()<<endl;

    if (VERBOSE_LEVEL_CHECK(VB_CHANSCAN))
    {
        static QString old_msg;
        if (msg != old_msg)
        {
            VERBOSE(VB_CHANSCAN, LOC + msg);
            old_msg = msg;
        }
    }
    else if (VERBOSE_LEVEL_NONE)
    {
        if (msg.length() > 80)
            msg = msg.left(77) + "...";
        cout<<"\r"<<msg.toAscii().constData()<<"\r";
        cout<<flush;
    }
}

void ChannelScannerCLI::InformUser(const QString &error)
{
    if (VERBOSE_LEVEL_NONE)
    {
        cerr<<"ERROR: "<<error.toAscii().constData()<<endl;
    }
    else
    {
        VERBOSE(VB_IMPORTANT, LOC_ERR + error);
    }
    post_event(scanMonitor, ScannerEvent::ScanComplete, 0);
}

void ChannelScannerCLI::Process(const ScanDTVTransportList &_transports)
{
    ChannelImporter ci(false, interactive, !onlysavescan, !onlysavescan, true,
                       freeToAirOnly, serviceRequirements);
    ci.Process(_transports);
}

void ChannelScannerCLI::MonitorProgress(
    bool lock, bool strength, bool snr, bool rotor)
{
    if (VERBOSE_LEVEL_NONE)
        cout<<"\r0%"<<flush;
}
