/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/**
 * SECTION:ogmrip-plugin
 * @title: Plugins System
 * @short_description: Functions for manipulating the plugins
 * @include: ogmrip-plugin.h
 */

#include "ogmrip-plugin.h"
#include "ogmrip-container.h"
#include "ogmrip-audio.h"
#include "ogmrip-video.h"
#include "ogmrip-subp.h"

#include "ogmjob-log.h"

#include <string.h>

typedef struct _OGMRipPlugin OGMRipPlugin;

struct _OGMRipPlugin
{
  GModule *module;
  GType type;
  gchar *name;
  gchar *description;
};

typedef struct _OGMRipPluginCodec OGMRipPluginCodec;

struct _OGMRipPluginCodec
{
  GModule *module;
  GType type;
  gchar *name;
  gchar *description;
  gint format;
};

typedef OGMRipPlugin * (* OGMRipPluginInit) (void);

GSList *video_codecs = NULL;
GSList *audio_codecs = NULL;
GSList *subp_codecs  = NULL;
GSList *containers   = NULL;

static gint
ogmrip_plugin_compare (OGMRipPlugin *plugin1, OGMRipPlugin *plugin2)
{
  return strcmp (plugin1->name, plugin2->name);
}

static GSList *
ogmrip_plugin_load (const gchar *dirname, GType type)
{
  GSList *slist = NULL;

  GModule *module;
  GPatternSpec *pspec;
  GDir *dir;

  OGMRipPlugin *plugin;
  OGMRipPluginInit init;
  gpointer ptr;

  const gchar *filename;
  gchar *fullname;
  gint len;

  len = strlen (dirname);

  pspec = g_pattern_spec_new ("*.so");

  dir = g_dir_open (dirname, 0, NULL);
  if (dir)
  {
    while ((filename = g_dir_read_name (dir)))
    {
      init = NULL;

      if (!g_pattern_match_string (pspec, filename))
        continue;

      fullname = g_build_filename (dirname, filename, NULL);
      module = g_module_open (fullname, G_MODULE_BIND_LAZY);
      g_free (fullname);

      if (!module)
      {
        g_warning ("Cannot open module %s", filename);
        continue;
      }

      if (!g_module_symbol (module, "ogmrip_init_plugin", &ptr))
      {
        g_warning ("Cannot find initialization function in module %s", filename);
        g_module_close (module);
        continue;
      }

      init = (OGMRipPluginInit) ptr;

      if (!init)
      {
        g_warning ("Invalid initialization function for module %s", filename);
        g_module_close (module);
        continue;
      }

      plugin = (* init) ();
      if (!plugin)
      {
        g_warning ("Failed to initialize module %s", filename);
        g_module_close (module);
        continue;
      }

      if (!g_type_is_a (plugin->type, type))
      {
        g_warning ("Invalid type for module %s, %s expected, %s found", filename, g_type_name (type), g_type_name (plugin->type));
        g_module_close (module);
        continue;
      }

      plugin->module = module;
      slist = g_slist_insert_sorted (slist, plugin, (GCompareFunc) ogmrip_plugin_compare);
    }
    g_dir_close (dir);
  }

  g_pattern_spec_free (pspec);

  return slist;
}

/**
 * ogmrip_plugin_init:
 *
 * Initializes the plugin system.
 */
void
ogmrip_plugin_init (void)
{
  if (!video_codecs)
    video_codecs = ogmrip_plugin_load (OGMRIP_LIB_DIR "/ogmrip/video-codecs", OGMRIP_TYPE_VIDEO);

  if (!audio_codecs)
    audio_codecs = ogmrip_plugin_load (OGMRIP_LIB_DIR "/ogmrip/audio-codecs", OGMRIP_TYPE_AUDIO);

  if (!subp_codecs)
    subp_codecs = ogmrip_plugin_load (OGMRIP_LIB_DIR "/ogmrip/subp-codecs", OGMRIP_TYPE_SUBP);

  if (!containers)
    containers = ogmrip_plugin_load (OGMRIP_LIB_DIR "/ogmrip/containers", OGMRIP_TYPE_CONTAINER);
}

/**
 * ogmrip_plugin_uninit:
 *
 * Uninitializes the plugin system.
 */
void
ogmrip_plugin_uninit (void)
{
  g_slist_foreach (video_codecs, (GFunc) g_module_close, NULL);
  g_slist_foreach (audio_codecs, (GFunc) g_module_close, NULL);
  g_slist_foreach (subp_codecs,  (GFunc) g_module_close, NULL);
  g_slist_foreach (containers,   (GFunc) g_module_close, NULL);
}

static OGMRipPluginContainer *
ogmrip_plugin_find_container_by_type (GSList *list, GType type)
{
  OGMRipPluginContainer *plugin;

  while (list)
  {
    plugin = (OGMRipPluginContainer *) list->data;
    if (plugin->type == type)
      return plugin;
    list = list->next;
  }

  return NULL;
}

static GType
ogmrip_plugin_get_nth_codec (GSList *list, guint n)
{
  OGMRipPlugin *plugin;

  if (!list)
    return G_TYPE_NONE;

  plugin = g_slist_nth_data (list, n);
  if (!plugin)
    plugin = list->data;

  return plugin->type;
}

static gint
ogmrip_plugin_get_codec_index (GSList *list, GType type)
{
  OGMRipPlugin *plugin;
  gint index;

  if (!list)
    return -1;

  for (index = 0; list; index ++, list = list->next)
  {
    plugin = list->data;
    if (plugin->type == type)
      return index;
  }

  return -1;
}

static void
ogmrip_plugin_foreach_codec (GSList *list, OGMRipPluginFunc func, gpointer data)
{
  OGMRipPlugin *plugin;

  while (list)
  {
    plugin = list->data;
    func (plugin->type, (const gchar *) plugin->name, (const gchar *) plugin->description, data);
    list = list->next;
  }
}

static OGMRipPlugin *
ogmrip_plugin_find_codec_by_type (GSList *list, GType type)
{
  OGMRipPlugin *plugin;

  while (list)
  {
    plugin = list->data;
    if (plugin->type == type)
      return plugin;
    list = list->next;
  }

  return NULL;
}

static G_CONST_RETURN gchar *
ogmrip_plugin_get_codec_name (GSList *list, GType type)
{
  OGMRipPlugin *plugin;

  plugin = ogmrip_plugin_find_codec_by_type (list, type);
  if (plugin)
    return (gchar *) plugin->name;

  return NULL;
}

static gint
ogmrip_plugin_get_codec_format (GSList *list, GType type)
{
  OGMRipPluginCodec *plugin;

  plugin = (OGMRipPluginCodec *) ogmrip_plugin_find_codec_by_type (list, type);
  if (plugin)
    return plugin->format;

  return -1;
}

static GType
ogmrip_plugin_find_codec (GSList *list, OGMRipPluginCmpFunc func, gconstpointer data)
{
  OGMRipPlugin *plugin;

  while (list)
  {
    plugin = list->data;
    if (func (plugin->type, (gchar *) plugin->name, (gchar *) plugin->description, data) == 0)
      return plugin->type;
    list = list->next;
  }

  return G_TYPE_NONE;
}

/**
 * ogmrip_plugin_get_n_containers:
 *
 * Gets the number of container plugins.
 *
 * Returns: the number of container plugins
 */
gint
ogmrip_plugin_get_n_containers (void)
{
  return g_slist_length (containers);
}

/**
 * ogmrip_plugin_foreach_container:
 * @func: The function to call with each container plugin's data
 * @data: User data to pass to the function
 *
 * Calls a function for each container plugin.
 */
void
ogmrip_plugin_foreach_container (OGMRipPluginFunc func, gpointer data)
{
  OGMRipPluginContainer *plugin;
  GSList *link;

  g_return_if_fail (func != NULL);

  for (link = containers; link; link = link->next)
  {
    plugin = link->data;
    func (plugin->type, (const gchar *) plugin->name, (const gchar *) plugin->description, data);
  }
}

/**
 * ogmrip_plugin_find_container:
 * @func: The function to call for each container plugin. It should return 0
 * when the desired container is found
 * @data: User data passed to the function
 *
 * Finds a container using the supplied function.
 *
 * Returns: The type of the container, or %G_TYPE_NONE
 */
GType
ogmrip_plugin_find_container (OGMRipPluginCmpFunc func, gconstpointer data)
{
  OGMRipPluginContainer *plugin;
  GSList *link;

  g_return_val_if_fail (func != NULL, G_TYPE_NONE);

  for (link = containers; link; link = link->next)
  {
    plugin = link->data;
    if (func (plugin->type, (gchar *) plugin->name, (gchar *) plugin->description, data) == 0)
      return plugin->type;
  }

  return G_TYPE_NONE;
}

/**
 * ogmrip_plugin_get_nth_container:
 * @n: The index of the container
 *
 * Gets the container at the given position.
 *
 * Returns: The type of the container, or %G_TYPE_NONE
 */
GType
ogmrip_plugin_get_nth_container (guint n)
{
  OGMRipPluginContainer *plugin;

  if (!containers)
    return G_TYPE_NONE;

  plugin = g_slist_nth_data (containers, n);
  if (!plugin)
    plugin = containers->data;

  return plugin->type;
}

/**
 * ogmrip_plugin_get_container_index:
 * @container: A container type
 *
 * Gets the position of the given container.
 *
 * Returns: The index of the container, or -1
 */
gint
ogmrip_plugin_get_container_index (GType container)
{
  GSList *link;
  OGMRipPluginContainer *plugin;
  gint index;

  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), 0);

  for (index = 0, link = containers; link; index ++, link = link->next)
  {
    plugin = link->data;
    if (plugin->type == container)
      return index;
  }

  return -1;
}

/**
 * ogmrip_plugin_get_container_name:
 * @container: A container type
 *
 * Gets the name of the given container.
 *
 * Returns: The name of the container, or NULL 
 */
G_CONST_RETURN gchar *
ogmrip_plugin_get_container_name (GType container)
{
  OGMRipPluginContainer *plugin;

  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), NULL);

  plugin = ogmrip_plugin_find_container_by_type (containers, container);
  if (plugin)
    return (gchar *) plugin->name;

  return NULL;
}

/**
 * ogmrip_plugin_get_container_bframes:
 * @container: A container type
 *
 * Gets whether the given container supports B-frames
 *
 * Returns: %TRUE if the container supports B-frames
 */
gboolean
ogmrip_plugin_get_container_bframes (GType container)
{
  OGMRipPluginContainer *plugin;

  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);

  plugin = ogmrip_plugin_find_container_by_type (containers, container);
  if (plugin)
    return plugin->bframes;

  return FALSE;
}

/**
 * ogmrip_plugin_get_n_video_codecs:
 *
 * Gets the number of video codec plugins.
 *
 * Returns: the number of video codec plugins
 */
gint
ogmrip_plugin_get_n_video_codecs (void)
{
  return g_slist_length (video_codecs);
}

/**
 * ogmrip_plugin_foreach_video_codec:
 * @func: The function to call with each video codec plugin's data
 * @data: User data to pass to the function
 *
 * Calls a function for each video codec plugin.
 */
void
ogmrip_plugin_foreach_video_codec (OGMRipPluginFunc func, gpointer data)
{
  g_return_if_fail (func != NULL);

  ogmrip_plugin_foreach_codec (video_codecs, func, data);
}

/**
 * ogmrip_plugin_find_video_codec:
 * @func: The function to call for each video codec plugin. It should return 0
 * when the desired video codec is found
 * @data: User data passed to the function
 *
 * Finds a video codec using the supplied function.
 *
 * Returns: The type of the video codec, or %G_TYPE_NONE
 */
GType
ogmrip_plugin_find_video_codec (OGMRipPluginCmpFunc func, gconstpointer data)
{
  g_return_val_if_fail (func != NULL, G_TYPE_NONE);

  return ogmrip_plugin_find_codec (video_codecs, func, data);
}

/**
 * ogmrip_plugin_get_nth_video_codec:
 * @n: The index of the video codec
 *
 * Gets the video codec at the given position.
 *
 * Returns: The type of the video codec, or %G_TYPE_NONE
 */
GType
ogmrip_plugin_get_nth_video_codec (guint n)
{
  return ogmrip_plugin_get_nth_codec (video_codecs, n);
}

/**
 * ogmrip_plugin_get_video_codec_index:
 * @codec: A video codec type
 *
 * Gets the position of the given video codec.
 *
 * Returns: The index of the video codec, or -1
 */
gint
ogmrip_plugin_get_video_codec_index (GType codec)
{
  return ogmrip_plugin_get_codec_index (video_codecs, codec);
}

/**
 * ogmrip_plugin_get_video_codec_name:
 * @codec: A video codec type
 *
 * Gets the name of the given video codec.
 *
 * Returns: The name of the video codec, or NULL 
 */
G_CONST_RETURN gchar *
ogmrip_plugin_get_video_codec_name (GType codec)
{
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_VIDEO), NULL);

  return ogmrip_plugin_get_codec_name (video_codecs, codec);
}

/**
 * ogmrip_plugin_get_video_codec_format:
 * @codec: A video codec type
 *
 * Gets the format of the given video codec.
 *
 * Returns: The format of the video codec, or NULL 
 */
gint
ogmrip_plugin_get_video_codec_format (GType codec)
{
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_VIDEO), -1);

  return ogmrip_plugin_get_codec_format (video_codecs, codec);
}

/**
 * ogmrip_plugin_get_video_codec_passes:
 * @codec: A video codec type
 *
 * Gets the maximum number of passes the given video codec supports.
 *
 * Returns: The maximum number of passes, or -1 
 */
gint
ogmrip_plugin_get_video_codec_passes (GType codec)
{
  OGMRipPlugin *plugin;

  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_VIDEO), -1);

  plugin = ogmrip_plugin_find_codec_by_type (video_codecs, codec);
  if (!plugin)
    return -1;

  return ((OGMRipPluginVideoCodec *) plugin)->passes;
}

/**
 * ogmrip_plugin_get_video_codec_threads:
 * @codec: A video codec type
 *
 * Gets the maximum number of threads the given video codec supports.
 *
 * Returns: The maximum number of threads, or -1
 */
gint
ogmrip_plugin_get_video_codec_threads (GType codec)
{
  OGMRipPlugin *plugin;

  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_VIDEO), -1);

  plugin = ogmrip_plugin_find_codec_by_type (video_codecs, codec);
  if (!plugin)
    return -1;

  return ((OGMRipPluginVideoCodec *) plugin)->threads;
}

/**
 * ogmrip_plugin_get_n_audio_codecs:
 *
 * Gets the number of audio codec plugins.
 *
 * Returns: the number of audio codec plugins
 */
gint
ogmrip_plugin_get_n_audio_codecs (void)
{
  return g_slist_length (audio_codecs);
}

/**
 * ogmrip_plugin_foreach_audio_codec:
 * @func: The function to call with each plugin's data
 * @data: User data to pass to the function
 *
 * Calls a function for each plugin.
 */
void
ogmrip_plugin_foreach_audio_codec (OGMRipPluginFunc func, gpointer data)
{
  g_return_if_fail (func != NULL);

  ogmrip_plugin_foreach_codec (audio_codecs, func, data);
}

/**
 * ogmrip_plugin_find_audio_codec:
 * @func: The function to call for each audio codec plugin. It should return 0
 * when the desired audio codec is found
 * @data: User data passed to the function
 *
 * Finds a audio codec using the supplied function.
 *
 * Returns: The type of the audio codec, or %G_TYPE_NONE
 */
GType
ogmrip_plugin_find_audio_codec (OGMRipPluginCmpFunc func, gconstpointer data)
{
  g_return_val_if_fail (func != NULL, G_TYPE_NONE);

  return ogmrip_plugin_find_codec (audio_codecs, func, data);
}

/**
 * ogmrip_plugin_get_nth_audio_codec:
 * @n: The index of the audio codec
 *
 * Gets the audio codec at the given position.
 *
 * Returns: The type of the audio codec, or %G_TYPE_NONE
 */
GType
ogmrip_plugin_get_nth_audio_codec (guint n)
{
  return ogmrip_plugin_get_nth_codec (audio_codecs, n);
}

/**
 * ogmrip_plugin_get_audio_codec_index:
 * @codec: An audio codec type
 *
 * Gets the position of the given audio codec.
 *
 * Returns: The index of the audio codec, or -1
 */
gint
ogmrip_plugin_get_audio_codec_index (GType codec)
{
  return ogmrip_plugin_get_codec_index (audio_codecs, codec);
}

/**
 * ogmrip_plugin_get_audio_codec_name:
 * @codec: An audio codec type
 *
 * Gets the name of the given audio codec.
 *
 * Returns: The name of the audio codec, or NULL 
 */
G_CONST_RETURN gchar *
ogmrip_plugin_get_audio_codec_name (GType codec)
{
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_AUDIO), NULL);

  return ogmrip_plugin_get_codec_name (audio_codecs, codec);
}

/**
 * ogmrip_plugin_get_audio_codec_format:
 * @codec: A audio codec type
 *
 * Gets the format of the given audio codec.
 *
 * Returns: The format of the audio codec, or NULL 
 */
gint
ogmrip_plugin_get_audio_codec_format (GType codec)
{
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_AUDIO), -1);

  return ogmrip_plugin_get_codec_format (audio_codecs, codec);
}

/**
 * ogmrip_plugin_get_n_subp_codecs:
 *
 * Gets the number of subtitle codec plugins.
 *
 * Returns: the number of subtitle codec plugins
 */
gint
ogmrip_plugin_get_n_subp_codecs (void)
{
  return g_slist_length (subp_codecs);
}

/**
 * ogmrip_plugin_foreach_subp_codec:
 * @func: The function to call with each plugin's data
 * @data: User data to pass to the function
 *
 * Calls a function for each plugin.
 */
void
ogmrip_plugin_foreach_subp_codec (OGMRipPluginFunc func, gpointer data)
{
  g_return_if_fail (func != NULL);

  ogmrip_plugin_foreach_codec (subp_codecs, func, data);
}

/**
 * ogmrip_plugin_find_subp_codec:
 * @func: The function to call for each subtitle codec plugin. It should return 0
 * when the desired subtitle codec is found
 * @data: User data passed to the function
 *
 * Finds a subtitle codec using the supplied function.
 *
 * Returns: The type of the subtitle codec, or %G_TYPE_NONE
 */
GType
ogmrip_plugin_find_subp_codec (OGMRipPluginCmpFunc func, gconstpointer data)
{
  g_return_val_if_fail (func != NULL, G_TYPE_NONE);

  return ogmrip_plugin_find_codec (subp_codecs, func, data);
}

/**
 * ogmrip_plugin_get_nth_subp_codec:
 * @n: The index of the subtitle codec
 *
 * Gets the subtitle codec at the given position.
 *
 * Returns: The type of the subtitle codec, or %G_TYPE_NONE
 */
GType
ogmrip_plugin_get_nth_subp_codec (guint n)
{
  return ogmrip_plugin_get_nth_codec (subp_codecs, n);
}

/**
 * ogmrip_plugin_get_subp_codec_index:
 * @codec: A subtitle codec type
 *
 * Gets the position of the given subtitle codec.
 *
 * Returns: The index of the subtitle codec, or -1
 */
gint
ogmrip_plugin_get_subp_codec_index (GType codec)
{
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_SUBP), 0);

  return ogmrip_plugin_get_codec_index (subp_codecs, codec);
}

/**
 * ogmrip_plugin_get_subp_codec_name:
 * @codec: A subtitle codec type
 *
 * Gets the name of the given subtitle codec.
 *
 * Returns: The name of the subtitle codec, or NULL 
 */
G_CONST_RETURN gchar *
ogmrip_plugin_get_subp_codec_name (GType codec)
{
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_SUBP), NULL);

  return ogmrip_plugin_get_codec_name (subp_codecs, codec);
}

/**
 * ogmrip_plugin_get_subp_codec_format:
 * @codec: A subtitle codec type
 *
 * Gets the format of the given subtitle codec.
 *
 * Returns: The format of the subtitle codec, or NULL 
 */
gint
ogmrip_plugin_get_subp_codec_format (GType codec)
{
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_SUBP), -1);

  return ogmrip_plugin_get_codec_format (subp_codecs, codec);
}

/**
 * ogmrip_plugin_get_subp_codec_text:
 * @codec: A subtitle codec type
 *
 * Gets whether the given codec outputs text subtitles.
 *
 * Returns: %TRUE if the codec output text subtitles
 */
gboolean
ogmrip_plugin_get_subp_codec_text (GType codec)
{
  OGMRipPlugin *plugin;

  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_SUBP), FALSE);

  plugin = ogmrip_plugin_find_codec_by_type (subp_codecs, codec);
  if (!plugin)
    return FALSE;

  return ((OGMRipPluginSubpCodec *) plugin)->text;
}

/**
 * ogmrip_plugin_can_contain_format:
 * @container: A container type
 * @format: An #OGMRipFormatType
 *
 * Returns whether @container supports the given format.
 *
 * Returns: %TRUE if @container supports @format
 */
gboolean
ogmrip_plugin_can_contain_format (GType container, OGMRipFormatType format)
{
  OGMRipPluginContainer *plugin;
  gint i;

  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);

  plugin = ogmrip_plugin_find_container_by_type (containers, container);
  if (!plugin || !plugin->formats)
    return FALSE;

  for (i = 0; plugin->formats[i] != -1; i ++)
    if (plugin->formats[i] == format)
      return TRUE;

  return FALSE;
}

static gboolean
ogmrip_plugin_can_contain_codec (GType container, GSList *codecs, GType codec)
{
  return ogmrip_plugin_can_contain_format (container,
      ogmrip_plugin_get_codec_format (codecs, codec));
}

/**
 * ogmrip_plugin_can_contain_video:
 * @container: A container type
 * @codec: A video codec type
 *
 * Returns whether @container supports the given video codec.
 *
 * Returns: %TRUE if @container supports @type
 */
gboolean
ogmrip_plugin_can_contain_video (GType container, GType codec)
{
  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_VIDEO), FALSE);

  return ogmrip_plugin_can_contain_codec (container, video_codecs, codec);
}

/**
 * ogmrip_plugin_can_contain_audio:
 * @container: A container type
 * @codec: An audio codec type
 *
 * Returns whether @container supports the given audio codec.
 *
 * Returns: %TRUE if @container supports @type
 */
gboolean
ogmrip_plugin_can_contain_audio (GType container, GType codec)
{
  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_AUDIO), FALSE);

  return ogmrip_plugin_can_contain_codec (container, audio_codecs, codec);
}

/**
 * ogmrip_plugin_can_contain_subp:
 * @container: A container type
 * @codec: A subtitle codec type
 *
 * Returns whether @container supports the given subtitle codec.
 *
 * Returns: %TRUE if @container supports @type
 */
gboolean
ogmrip_plugin_can_contain_subp (GType container, GType codec)
{
  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_SUBP), FALSE);

  return ogmrip_plugin_can_contain_codec (container, subp_codecs, codec);
}

/**
 * ogmrip_plugin_can_contain_n_audio:
 * @container: A container type
 * @ncodec: The number of audio codecs
 *
 * Returns whether @container can contain @ncodec audio streams.
 *
 * Returns: %TRUE if @container can contain @ncodec audio streams
 */
gboolean
ogmrip_plugin_can_contain_n_audio (GType container, guint ncodec)
{
  OGMRipPluginContainer *plugin;

  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);

  plugin = ogmrip_plugin_find_container_by_type (containers, container);
  if (!plugin)
    return FALSE;

  return ncodec <= plugin->max_audio;
}

/**
 * ogmrip_plugin_can_contain_n_subp:
 * @container: A container type
 * @ncodec: The number of subtitle codecs
 *
 * Returns whether @container can contain @ncodec subtitle streams.
 *
 * Returns: %TRUE if @container can contain @ncodec subtitle streams
 */
gboolean
ogmrip_plugin_can_contain_n_subp (GType container, guint ncodec)
{
  OGMRipPluginContainer *plugin;

  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);

  plugin = ogmrip_plugin_find_container_by_type (containers, container);
  if (!plugin)
    return FALSE;

  return ncodec <= plugin->max_subp;
}

/**
 * ogmrip_plugin_get_container_max_audio:
 * @container: A container type
 *
 * Returns the number of audio streams the given container can contain.
 *
 * Returns: the number of audio streams, or -1
 */
gint
ogmrip_plugin_get_container_max_audio (GType container)
{
  OGMRipPluginContainer *plugin;

  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), -1);

  plugin = ogmrip_plugin_find_container_by_type (containers, container);
  if (!plugin)
    return FALSE;

  return plugin->max_audio;
}

/**
 * ogmrip_plugin_get_container_max_subp:
 * @container: A container type
 *
 * Returns the number of subtitle streams the given container can contain.
 *
 * Returns: the number of subtitle streams, or -1
 */
gint
ogmrip_plugin_get_container_max_subp (GType container)
{
  OGMRipPluginContainer *plugin;

  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), -1);

  plugin = ogmrip_plugin_find_container_by_type (containers, container);
  if (!plugin)
    return FALSE;

  return plugin->max_subp;
}

