/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ogmrip-vobsub.h"
#include "ogmrip-mplayer.h"
#include "ogmrip-plugin.h"

#include "ogmjob-queue.h"
#include "ogmjob-exec.h"

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <glib/gstdio.h>
#include <glib/gi18n-lib.h>

static gint ogmrip_vobsub_run      (OGMJobSpawn *spawn);
static void ogmrip_vobsub_finalize (GObject     *gobject);

G_DEFINE_TYPE (OGMRipVobSub, ogmrip_vobsub, OGMRIP_TYPE_SUBP)

static gchar **
ogmrip_vobsub_command (OGMRipSubp *subp, const gchar *input, const gchar *output)
{
  GPtrArray *argv;

  argv = ogmrip_mencoder_vobsub_command (subp, NULL, output);
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static void
ogmrip_vobsub_class_init (OGMRipVobSubClass *klass)
{
  GObjectClass *gobject_class;
  OGMJobSpawnClass *spawn_class;

  gobject_class = G_OBJECT_CLASS (klass);
  spawn_class = OGMJOB_SPAWN_CLASS (klass);

  gobject_class->finalize = ogmrip_vobsub_finalize;
  spawn_class->run = ogmrip_vobsub_run;
}

static void
ogmrip_vobsub_init (OGMRipVobSub *vobsub)
{
}

static void
ogmrip_vobsub_finalize (GObject *gobject)
{
  const gchar *output;

  output = ogmrip_codec_get_output (OGMRIP_CODEC (gobject));
  if (output)
  {
    if (ogmrip_codec_get_unlink_on_unref (OGMRIP_CODEC (gobject)))
    {
      gchar *filename;

      filename = g_strconcat (output, ".idx", NULL);
      if (g_file_test (filename, G_FILE_TEST_IS_REGULAR))
        g_unlink (filename);
      g_free (filename);

      filename = g_strconcat (output, ".sub", NULL);
      if (g_file_test (filename, G_FILE_TEST_IS_REGULAR))
        g_unlink (filename);
      g_free (filename);
    }
  }

  G_OBJECT_CLASS (ogmrip_vobsub_parent_class)->finalize (gobject);
}

static gint
ogmrip_vobsub_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *child;
  gchar **argv;
  gint result;

  argv = ogmrip_vobsub_command (OGMRIP_SUBP (spawn), NULL, NULL);
  if (!argv)
    return OGMJOB_RESULT_ERROR;

  child = ogmjob_exec_newv (argv);
  ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_mencoder_vobsub_watch, spawn, TRUE, FALSE, FALSE);
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), child);
  g_object_unref (child);

  result = OGMJOB_SPAWN_CLASS (ogmrip_vobsub_parent_class)->run (spawn);
  if (result == OGMJOB_RESULT_COMPLETED)
  {
    struct stat buf; 
    const gchar *basename;
    gchar *idxname, *subname;

    basename = ogmrip_codec_get_output (OGMRIP_CODEC (spawn));
    idxname = g_strconcat (basename, ".idx", NULL);
    subname = g_strconcat (basename, ".sub", NULL);

    if ((g_file_test (idxname, G_FILE_TEST_IS_REGULAR) && g_stat (idxname, &buf) == 0 && buf.st_size > 0) &&
        (g_file_test (subname, G_FILE_TEST_IS_REGULAR) && g_stat (subname, &buf) == 0 && buf.st_size > 0))
    {
      gint fd;

      fd = g_open (basename, O_WRONLY);
      write (fd, "foo", 3);
      close (fd);
    }

    g_free (idxname);
    g_free (subname);
  }

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), child);

  return result;
}

/**
 * ogmrip_vobsub_new:
 * @subp: An #OGMDvdSubpStream
 * @output: The output file
 *
 * Creates a new #OGMRipVobSub.
 *
 * Returns: The new #OGMRipVobSub
 */
OGMJobSpawn *
ogmrip_vobsub_new (OGMDvdSubpStream *subp, const gchar *output)
{
  g_return_val_if_fail (subp != NULL, NULL);
  g_return_val_if_fail (output && *output, NULL);

  return g_object_new (OGMRIP_TYPE_VOBSUB, "stream", subp, "output", output, NULL);
}

static OGMRipPluginContainer vobsub_plugin =
{
  NULL,
  G_TYPE_NONE,
  "vobsub",
  N_("VobSub"),
  OGMRIP_FORMAT_VOBSUB,
  FALSE
};

OGMRipPluginContainer *
ogmrip_init_plugin (void)
{
  vobsub_plugin.type = OGMRIP_TYPE_VOBSUB;

  return &vobsub_plugin;
}

