/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ogmrip-x264.h"
#include "ogmrip-mplayer.h"
#include "ogmrip-version.h"
#include "ogmrip-plugin.h"
#include "ogmrip-file.h"

#include "ogmjob-exec.h"

#include <math.h>
#include <stdio.h>
#include <unistd.h>
#include <glib/gi18n-lib.h>

#if MPLAYER_CHECK_VERSION(1,0,1,0)
#define OGMRIP_X264_VERY_HIGH_OPTIONS "subq=6:brdo:b_pyramid:weight_b:8x8dct:frameref=5:mixed_refs:me=umh"
#define OGMRIP_X264_HIGH_OPTIONS      "subq=5:b_pyramid:weight_b:8x8dct:frameref=2:mixed_refs"
#else
#define OGMRIP_X264_VERY_HIGH_OPTIONS "subq=6:b_pyramid:weight_b:frameref=5:me=3"
#define OGMRIP_X264_HIGH_OPTIONS      "subq=5:b_pyramid:weight_b:frameref=2"
#endif

#define OGMRIP_X264_FAST_OPTIONS      "subq=4:b_pyramid:weight_b"

static gint ogmrip_x264_run             (OGMJobSpawn *spawn);
static gint ogmrip_x264_get_start_delay (OGMRipVideo *video);

static gchar **
ogmrip_x264_command (OGMRipVideo *video, const gchar *input, const gchar *output, const gchar *logf)
{
  OGMDvdTitle *title;
  GPtrArray *argv;
  GString *options;

  gint quality, bitrate, vid, pass, threads, bframes;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_val_if_fail (title != NULL, NULL);

  pass = ogmrip_video_get_pass (video);
  if (pass > 0 && !logf)
    logf = ogmrip_video_get_log (video);
  g_return_val_if_fail (pass == 0 || logf != NULL, NULL);

  argv = ogmrip_mencoder_video_command (video, title, pass == 1 ? "/dev/null" : output);

  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("x264"));

  quality = ogmrip_video_get_quality (video);
  switch (quality)
  {
    case OGMRIP_QUALITY_VERY_HIGH:
      options = g_string_new (OGMRIP_X264_VERY_HIGH_OPTIONS);
      break;
    case OGMRIP_QUALITY_HIGH:
      options = g_string_new (OGMRIP_X264_HIGH_OPTIONS);
      break;
    default:
      options = g_string_new (OGMRIP_X264_FAST_OPTIONS);
      break;
  }

#if MPLAYER_CHECK_VERSION(1,0,0,8)
  if (ogmrip_video_get_turbo (video))
    g_string_append (options, ":turbo=2");
  else if (pass == 1)
    g_string_append (options, ":turbo=1");
  /*
   * TODO in n-pass mode, all the passes except the last one
   * should be encoded in at least turbo=1
   */
#endif /* MPLAYER_CHECK_VERSION(1,0,0,8) */

  if (ogmrip_video_get_4mv (video))
#if MPLAYER_CHECK_VERSION(1,0,1,0)
    g_string_append (options, ":partitions=p8x8,b8x8,i8x8,i4x4");
#else
    g_string_append (options, ":4x4mv");
#endif
  if (ogmrip_video_get_trellis (video))
  {
    if (quality == OGMRIP_QUALITY_VERY_HIGH)
      g_string_append (options, ":trellis=2");
    else
      g_string_append (options, ":trellis=1");
  }
  else
    g_string_append (options, ":trellis=0");

  bframes = ogmrip_video_get_max_b_frames (video);
  g_string_append_printf (options, ":bframes=%d", bframes);

  if (bframes > 0)
    g_string_append (options, ":bime");

  bitrate = ogmrip_video_get_bitrate (video);
  if (bitrate > 0)
    g_string_append_printf (options, ":bitrate=%u", bitrate / 1000);
  else
  {
    gdouble quantizer;
    gint crf;

    quantizer = ogmrip_video_get_quantizer (video);
    crf = 12 + (unsigned int) (6.0 * log (quantizer) / log (2.0));

    g_string_append_printf (options, ":crf=%u", CLAMP (crf, 1, 50));
  }

  if (pass)
  {
    g_string_append (options, ":direct_pred=auto");
    g_string_append_printf (options, ":pass=%u", pass);
    g_ptr_array_add (argv, g_strdup ("-passlogfile"));
    g_ptr_array_add (argv, g_strdup (logf));
  }
  
  threads = ogmrip_video_get_threads (video);
  if (threads > 1)
    g_string_append_printf (options, ":threads=%u", CLAMP (threads, 1, 4));

  if (!ogmrip_video_get_cartoon (video) && quality == OGMRIP_QUALITY_VERY_HIGH)
    g_string_append (options, ":nodct_decimate");

  g_ptr_array_add (argv, g_strdup ("-x264encopts"));
  g_ptr_array_add (argv, g_string_free (options, FALSE));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_CHECK_VERSION(1,0,0,1)
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_CHECK_VERSION(1,0,0,1) */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_CHECK_VERSION(1,0,0,1) */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

G_DEFINE_TYPE (OGMRipX264, ogmrip_x264, OGMRIP_TYPE_VIDEO)

static void
ogmrip_x264_class_init (OGMRipX264Class *klass)
{
  OGMJobSpawnClass *spawn_class;
  OGMRipVideoClass *video_class;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);
  spawn_class->run = ogmrip_x264_run;

  video_class = OGMRIP_VIDEO_CLASS (klass);
  video_class->get_start_delay = ogmrip_x264_get_start_delay;
}

static void
ogmrip_x264_init (OGMRipX264 *x264)
{
}

static gint
ogmrip_x264_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *child;
  gchar **argv;
  gint result;

  argv = ogmrip_x264_command (OGMRIP_VIDEO (spawn), NULL, NULL, NULL);
  if (!argv)
    return OGMJOB_RESULT_ERROR;

  child = ogmjob_exec_newv (argv);
  ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_mencoder_codec_watch, spawn, TRUE, FALSE, FALSE);
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), child);
  g_object_unref (child);

  result = OGMJOB_SPAWN_CLASS (ogmrip_x264_parent_class)->run (spawn);

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), child);

  return result;
}

static gint
ogmrip_x264_get_start_delay (OGMRipVideo *video)
{
  if (ogmrip_video_get_max_b_frames (video) > 0)
    return 2;
  
  return 1;
}

/**
 * ogmrip_x264_new:
 * @title: An #OGMDvdTitle
 * @output: The output file
 *
 * Creates a new #OGMRipX264.
 *
 * Returns: the new #OGMRipX264
 */
OGMJobSpawn *
ogmrip_x264_new (OGMDvdTitle *title, const gchar *output)
{
  g_return_val_if_fail (title != NULL, NULL);
  g_return_val_if_fail (output && *output, NULL);

  return g_object_new (OGMRIP_TYPE_X264, "input", title, "output", output, NULL);
}

static OGMRipPluginVideoCodec x264_plugin =
{
  NULL,
  G_TYPE_NONE,
  "x264",
  N_("X264"),
  OGMRIP_FORMAT_H264,
  G_MAXINT,
  4
};

OGMRipPluginVideoCodec *
ogmrip_init_plugin (void)
{
  x264_plugin.type = OGMRIP_TYPE_X264;

  return &x264_plugin;
}

