/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ogmrip-fs.h"
#include "ogmrip-xvid.h"
#include "ogmrip-version.h"
#include "ogmrip-mplayer.h"
#include "ogmrip-plugin.h"

#include "ogmjob-exec.h"

#include <stdio.h>
#include <unistd.h>
#include <glib/gstdio.h>
#include <glib/gi18n-lib.h>

#define OGMRIP_XVID_VERY_HIGH_OPTIONS "autoaspect:par=vga11:chroma_opt:vhq=4:bvhq=1:quant_type=mpeg"
#define OGMRIP_XVID_HIGH_OPTIONS      "autoaspect:par=vga11:chroma_opt:vhq=2:bvhq=1:quant_type=mpeg"
#define OGMRIP_XVID_FAST_OPTIONS      "autoaspect:par=vga11:vhq=0"

#define OGMRIP_XVID_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_XVID, OGMRipXvidPriv))

static gint ogmrip_xvid_run (OGMJobSpawn *spawn);

struct _OGMRipXvidPriv
{
  gboolean gmc;
};

static gchar **
ogmrip_xvid_command (OGMRipVideo *video, const gchar *input, const gchar *output, const gchar *logf)
{
  OGMDvdTitle *title;
  GPtrArray *argv;
  GString *options;

  gint bitrate, vid, pass, threads;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_val_if_fail (title != NULL, NULL);

  pass = ogmrip_video_get_pass (video);
  if (pass > 0 && !logf)
    logf = ogmrip_video_get_log (video);
  g_return_val_if_fail (pass == 0 || logf != NULL, NULL);

  argv = ogmrip_mencoder_video_command (video, title, pass == 1 ? "/dev/null" : output);

  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("xvid"));

  switch (ogmrip_video_get_quality (video))
  {
    case OGMRIP_QUALITY_VERY_HIGH:
      options = g_string_new (OGMRIP_XVID_VERY_HIGH_OPTIONS);
      break;
    case OGMRIP_QUALITY_HIGH:
      options = g_string_new (OGMRIP_XVID_HIGH_OPTIONS);
      break;
    default:
      options = g_string_new (OGMRIP_XVID_FAST_OPTIONS);
      break;
  }

#if MPLAYER_CHECK_VERSION(1,0,0,6)
  if (ogmrip_video_get_cartoon (video))
    g_string_append (options, ":cartoon");
  else
    g_string_append (options, ":nocartoon");
#endif /* MPLAYER_CHECK_VERSION(1,0,0,6) */

  if (ogmrip_video_get_qpel (video))
    g_string_append (options, ":qpel");
  else
    g_string_append (options, ":noqpel");

  if (ogmrip_video_get_turbo (video))
    g_string_append (options, ":turbo");

  if (ogmrip_video_get_trellis (video))
    g_string_append (options, ":trellis");
  else
    g_string_append (options, ":notrellis");

  if (ogmrip_video_get_grayscale (video))
    g_string_append (options, ":greyscale");
  else
    g_string_append (options, ":nogreyscale");

  if (ogmrip_xvid_get_gmc (OGMRIP_XVID (video)))
    g_string_append (options, ":gmc");
  else
    g_string_append (options, ":nogmc");

  g_string_append_printf (options, ":max_bframes=%d", ogmrip_video_get_max_b_frames (video));

  bitrate = ogmrip_video_get_bitrate (video);
  if (bitrate > 0)
  {
    if (bitrate < 16001)
      g_string_append_printf (options, ":bitrate=%u", bitrate / 1000);
    else
      g_string_append_printf (options, ":bitrate=%u", bitrate);
  }
  else
    g_string_append_printf (options, ":fixed_quant=%.0lf", ogmrip_video_get_quantizer (video));

  if (pass)
  {
    g_string_append_printf (options, ":pass=%u", pass);
    g_ptr_array_add (argv, g_strdup ("-passlogfile"));
    g_ptr_array_add (argv, g_strdup (logf));
  }

  threads = ogmrip_video_get_threads (video);
  if (threads > 1)
    g_string_append_printf (options, ":threads=%u", CLAMP (threads, 1, 4 /*height / 16*/));

  g_ptr_array_add (argv, g_strdup ("-xvidencopts"));
  g_ptr_array_add (argv, g_string_free (options, FALSE));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_CHECK_VERSION(1,0,0,1)
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_CHECK_VERSION(1,0,0,1) */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_CHECK_VERSION(1,0,0,1) */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

G_DEFINE_TYPE (OGMRipXvid, ogmrip_xvid, OGMRIP_TYPE_VIDEO)

static void
ogmrip_xvid_class_init (OGMRipXvidClass *klass)
{
  OGMJobSpawnClass *spawn_class;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);

  spawn_class->run = ogmrip_xvid_run;

  g_type_class_add_private (klass, sizeof (OGMRipXvidPriv));
}

static void
ogmrip_xvid_init (OGMRipXvid *xvid)
{
  xvid->priv = OGMRIP_XVID_GET_PRIVATE (xvid);
}

static gint
ogmrip_xvid_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *child;
  gchar **argv;
  gint result;

#if !MPLAYER_CHECK_VERSION(1,0,0,8)
  gchar *cwd;
#endif /* MPLAYER_CHECK_VERSION */

  argv = ogmrip_xvid_command (OGMRIP_VIDEO (spawn), NULL, NULL, NULL);
  if (!argv)
    return OGMJOB_RESULT_ERROR;

  child = ogmjob_exec_newv (argv);
  ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_mencoder_codec_watch, spawn, TRUE, FALSE, FALSE);
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), child);
  g_object_unref (child);

#if !MPLAYER_CHECK_VERSION(1,0,0,8)
  /*
   * Workaround against xvid pass log file
   */
  cwd = g_get_current_dir ();
  g_chdir (g_get_tmp_dir ());
#endif /* MPLAYER_CHECK_VERSION */

  result = OGMJOB_SPAWN_CLASS (ogmrip_xvid_parent_class)->run (spawn);

#if !MPLAYER_CHECK_VERSION(1,0,0,8)
  /*
   * Return in cwd
   */
  g_chdir (cwd);
  g_free (cwd);
#endif /* MPLAYER_CHECK_VERSION */

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), child);

  return result;
}

/**
 * ogmrip_xvid_new:
 * @title: An #OGMDvdTitle
 * @output: The output file
 *
 * Creates a new #OGMRipXvid.
 *
 * Returns: the new #OGMRipXvid
 */
OGMJobSpawn *
ogmrip_xvid_new (OGMDvdTitle *title, const gchar *output)
{
  g_return_val_if_fail (title != NULL, NULL);
  g_return_val_if_fail (output && *output, NULL);

  return g_object_new (OGMRIP_TYPE_XVID, "input", title, "output", output, NULL);
}

/**
 * ogmrip_xvid_set_gmc:
 * @xvid: An #OGMRipXvid
 * @gmc: %TRUE to set global motion compensation
 *
 * Enables global motion compensation.
 */
void
ogmrip_xvid_set_gmc (OGMRipXvid *xvid, gboolean gmc)
{
  g_return_if_fail (OGMRIP_IS_XVID (xvid));

  xvid->priv->gmc = gmc;
}

/**
 * ogmrip_xvid_get_gmc:
 * @xvid: An #OGMRipXvid
 *
 * Gets global motion compensation.
 *
 * Returns: %TRUE if global motion compensation has been set
 */
gboolean
ogmrip_xvid_get_gmc (OGMRipXvid *xvid)
{
  g_return_val_if_fail (OGMRIP_IS_XVID (xvid), FALSE);

  return xvid->priv->gmc;
}

static OGMRipPluginVideoCodec xvid_plugin =
{
  NULL,
  G_TYPE_NONE,
  "xvid",
  N_("XviD"),
  OGMRIP_FORMAT_MPEG4,
  2,
  G_MAXINT
};

OGMRipPluginVideoCodec *
ogmrip_init_plugin (void)
{
  xvid_plugin.type = OGMRIP_TYPE_XVID;

  return &xvid_plugin;
}

