/* OGMDvd - A wrapper library around libdvdread
 * Copyright (C) 2004-2008 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/**
 * SECTION:ogmdvd-title
 * @title: OGMDvdTitle
 * @include: ogmdvd-title.h
 * @short_description: Structure describing a DVD title
 */

#include "ogmdvd-disc.h"
#include "ogmdvd-enums.h"
#include "ogmdvd-title.h"
#include "ogmdvd-stream.h"
#include "ogmdvd-priv.h"

#include "ogmdvd-reader.h"
#include "ogmdvd-parser.h"

static glong
ogmdvd_time_to_msec (dvd_time_t *dtime)
{
  guint hour, min, sec, frames;
  gfloat fps;

  hour   = ((dtime->hour    & 0xf0) >> 4) * 10 + (dtime->hour    & 0x0f);
  min    = ((dtime->minute  & 0xf0) >> 4) * 10 + (dtime->minute  & 0x0f);
  sec    = ((dtime->second  & 0xf0) >> 4) * 10 + (dtime->second  & 0x0f);
  frames = ((dtime->frame_u & 0x30) >> 4) * 10 + (dtime->frame_u & 0x0f);

  if (((dtime->frame_u & 0xc0) >> 6) == 1)
    fps = 25.0;
  else
    fps = 30000 / 1001.0;

  return hour * 60 * 60 * 1000 + min * 60 * 1000 + sec * 1000 + (gfloat) frames * 1000.0 / fps;
}

static void
ogmdvd_msec_to_time (glong msec, OGMDvdTime *dtime)
{
  dtime->hour = msec / (60 * 60 * 1000);
  dtime->min = msec / (60 * 1000) % 60;
  dtime->sec = msec / 1000 % 60;
  dtime->frames = msec % 1000;
}

/**
 * ogmdvd_title_ref:
 * @title: An #OGMDvdTitle
 *
 * Increments the reference count of an #OGMDvdTitle.
 */
void
ogmdvd_title_ref (OGMDvdTitle *title)
{
  g_return_if_fail (title != NULL);

  title->ref ++;
}

/**
 * ogmdvd_title_unref:
 * @title: An #OGMDvdTitle
 *
 * Decrements the reference count of an #OGMDvdTitle.
 */
void
ogmdvd_title_unref (OGMDvdTitle *title)
{
  g_return_if_fail (title != NULL);

  if (title->ref > 0)
  {
    title->ref --;

    if (title->ref == 0)
    {
      title->disc->titles = g_slist_remove (title->disc->titles, title);

      if (title->vts_file)
        ifoClose (title->vts_file);
      title->vts_file = NULL;

      if (title->disc)
        ogmdvd_disc_unref (title->disc);
      title->disc = NULL;

      if (title->bitrates)
        g_free (title->bitrates);
      title->bitrates = NULL;

      g_free (title);
    }
  }
}

/**
 * ogmdvd_title_get_disc:
 * @title: An #OGMDvdTitle
 *
 * Returns the disc the #OGMDvdTitle was opened from.
 *
 * Returns: The #OGMDvdDisc, or NULL
 */
OGMDvdDisc *
ogmdvd_title_get_disc (OGMDvdTitle *title)
{
  g_return_val_if_fail (title != NULL, NULL);

  return title->disc;
}

/**
 * ogmdvd_title_get_nr:
 * @title: An #OGMDvdTitle
 *
 * Returns the title number.
 *
 * Returns: The title number, or -1
 */
gint
ogmdvd_title_get_nr (OGMDvdTitle *title)
{
  g_return_val_if_fail (title != NULL, -1);

  return title->nr;
}

/**
 * ogmdvd_title_get_vts_size:
 * @title: An #OGMDvdTitle
 *
 * Returns the size of the video title set in bytes.
 *
 * Returns: The size in bytes, or -1
 */
gint64
ogmdvd_title_get_vts_size (OGMDvdTitle *title)
{
  OGMDvdDisc *disc;
  gint64 size, fullsize;
  guint vts;

  g_return_val_if_fail (title != NULL, -1);

  disc = ogmdvd_title_get_disc (title);

  vts = disc->vmg_file ? disc->vmg_file->tt_srpt->title[title->nr].title_set_nr : 1;

  fullsize  = _ogmdvd_get_ifo_size (disc, vts);
  fullsize += _ogmdvd_get_bup_size (disc, vts);
  fullsize += _ogmdvd_get_menu_size (disc, vts);

  if (vts > 0)
  {
    if ((size = _ogmdvd_get_vob_size (disc, vts)) == 0)
      return 0;
    fullsize += size;
  }

  return fullsize;
}

/**
 * ogmdvd_title_get_length:
 * @title: An #OGMDvdTitle
 * @length: A pointer to set the #OGMDvdTime, or NULL
 *
 * Returns the title length in seconds. If @length is not NULL, the data
 * structure will be filled with the length in hours, minutes seconds and
 * frames.
 *
 * Returns: The length in seconds, or -1.0
 */
gdouble
ogmdvd_title_get_length (OGMDvdTitle *title, OGMDvdTime  *length)
{
  pgc_t *pgc;

  g_return_val_if_fail (title != NULL, -1.0);

  pgc = title->vts_file->vts_pgcit->pgci_srp[title->pgcn - 1].pgc;

  if (length)
  {
    dvd_time_t *dtime = &pgc->playback_time;

    length->hour   = ((dtime->hour    & 0xf0) >> 4) * 10 + (dtime->hour    & 0x0f);
    length->min    = ((dtime->minute  & 0xf0) >> 4) * 10 + (dtime->minute  & 0x0f);
    length->sec    = ((dtime->second  & 0xf0) >> 4) * 10 + (dtime->second  & 0x0f);
    length->frames = ((dtime->frame_u & 0x30) >> 4) * 10 + (dtime->frame_u & 0x0f);
  }

  return ogmdvd_time_to_msec (&pgc->playback_time) / 1000.0;
}

/**
 * ogmdvd_title_get_chapters_length:
 * @title: An #OGMDvdTitle
 * @start: The start chapter
 * @end: The end chapter
 * @length: A pointer to set the #OGMDvdTime, or NULL
 *
 * Returns the length in seconds between start and end chapters. If @length is
 * not NULL, the data structure will be filled with the length in hours, minutes
 * seconds and frames.
 *
 * Returns: The length in seconds, or -1.0
 */
gdouble
ogmdvd_title_get_chapters_length (OGMDvdTitle *title, guint start, gint end, OGMDvdTime *length)
{
  pgc_t *pgc;
  guint pgn, pgcn;
  glong last, cell, total;

  g_return_val_if_fail (title != NULL, -1.0);
  g_return_val_if_fail (end < 0 || start <= end, -1.0);

  pgc = title->vts_file->vts_pgcit->pgci_srp[title->pgcn - 1].pgc;

  g_return_val_if_fail (start < pgc->nr_of_programs, -1.0);

  pgcn = title->vts_file->vts_ptt_srpt->title[title->ttn - 1].ptt[start].pgcn;
  pgc = title->vts_file->vts_pgcit->pgci_srp[pgcn - 1].pgc;

  pgn = title->vts_file->vts_ptt_srpt->title[title->ttn - 1].ptt[start].pgn;
  cell = pgc->program_map[pgn - 1] - 1;

  last = pgc->nr_of_cells;
  if (end > -1 && end < pgc->nr_of_programs - 1)
  {
    pgn = title->vts_file->vts_ptt_srpt->title[title->ttn - 1].ptt[end + 1].pgn;
    last = pgc->program_map[pgn - 1] - 1;
  }

  if (start == 0 && last == pgc->nr_of_cells)
    return ogmdvd_title_get_length (title, length);

  for (total = 0; cell < last; cell ++)
    total += ogmdvd_time_to_msec (&pgc->cell_playback[cell].playback_time);

  if (length)
    ogmdvd_msec_to_time (total, length);

  return total / 1000.0;
}

/**
 * ogmdvd_title_get_framerate:
 * @title: An #OGMDvdTitle
 * @numerator: A pointer to set the framerate numerator, or NULL
 * @denominator: A pointer to set the framerate denominator, or NULL
 
 * Gets the framerate of the DVD title in the form of a fraction.
 */
void
ogmdvd_title_get_framerate (OGMDvdTitle *title, guint *numerator, guint *denominator)
{
  pgc_t *pgc;

  g_return_if_fail (title != NULL);
  g_return_if_fail (numerator != NULL);
  g_return_if_fail (denominator != NULL);

  pgc = title->vts_file->vts_pgcit->pgci_srp[title->pgcn - 1].pgc;

  switch ((pgc->playback_time.frame_u & 0xc0) >> 6)
  {
    case 1:
      *numerator = 25;
      *denominator = 1;
      break;
    case 3:
      *numerator = 30000;
      *denominator = 1001;
      break;
    default:
      g_assert_not_reached ();
      break;
  }
}

/**
 * ogmdvd_title_get_size:
 * @title: An #OGMDvdTitle
 * @width: A pointer to set the width of the picture, or NULL
 * @height: A pointer to set the height of the picture, or NULL
 
 * Gets the size of the picture.
 */
void
ogmdvd_title_get_size (OGMDvdTitle *title, guint *width, guint *height)
{
  g_return_if_fail (title != NULL);
  g_return_if_fail (width != NULL);
  g_return_if_fail (height != NULL);

  *width = 0;
  *height = 480;
  if (title->vts_file->vtsi_mat->vts_video_attr.video_format != 0)
    *height = 576;

  switch (title->vts_file->vtsi_mat->vts_video_attr.picture_size)
  {
    case 0:
      *width = 720;
      break;
    case 1:
      *width = 704;
      break;
    case 2:
      *width = 352;
      break;
    case 3:
      *width = 352;
      *width /= 2;
      break;
    default:
      g_assert_not_reached ();
      break;
  }
}

/**
 * ogmdvd_title_get_aspect_ratio:
 * @title: An #OGMDvdTitle
 * @numerator: A pointer to set the aspect ratio numerator, or NULL
 * @denominator: A pointer to set the aspect ratio denominator, or NULL
 
 * Gets the aspect ratio of the DVD title in the form of a fraction.
 */
void
ogmdvd_title_get_aspect_ratio  (OGMDvdTitle *title, guint *numerator, guint *denominator)
{
  g_return_if_fail (title != NULL);
  g_return_if_fail (numerator != NULL);
  g_return_if_fail (denominator != NULL);

  switch (title->vts_file->vtsi_mat->vts_video_attr.display_aspect_ratio)
  {
    case 0:
      *numerator = 4;
      *denominator = 3;
      break;
    case 1:
    case 3:
      *numerator = 16;
      *denominator = 9;
      break;
    default:
      g_assert_not_reached ();
      break;
  }
}

/**
 * ogmdvd_title_get_video_format:
 * @title: An #OGMDvdTitle
 *
 * Returns the video format of the movie.
 *
 * Returns: #OGMDvdVideoFormat, or -1
 */
gint
ogmdvd_title_get_video_format (OGMDvdTitle *title)
{
  g_return_val_if_fail (title != NULL, -1);

  return title->vts_file->vtsi_mat->vts_video_attr.video_format;
}

/**
 * ogmdvd_title_get_display_aspect:
 * @title: An #OGMDvdTitle
 *
 * Returns the display aspect of the movie.
 *
 * Returns: #OGMDvdDisplayAspect, or -1
 */
gint
ogmdvd_title_get_display_aspect (OGMDvdTitle *title)
{
  g_return_val_if_fail (title != NULL, -1);

  switch (title->vts_file->vtsi_mat->vts_video_attr.display_aspect_ratio)
  {
    case 0:
      return OGMDVD_DISPLAY_ASPECT_4_3;
    case 1:
    case 3:
      return OGMDVD_DISPLAY_ASPECT_16_9;
    default:
      return -1;
  }
}

/**
 * ogmdvd_title_get_display_format:
 * @title: An #OGMDvdTitle
 *
 * Returns the display format of the movie.
 *
 * Returns: #OGMDvdDisplayFormat, or -1
 */
gint
ogmdvd_title_get_display_format (OGMDvdTitle *title)
{
  g_return_val_if_fail (title != NULL, -1);

  return title->vts_file->vtsi_mat->vts_video_attr.permitted_df;
}

/**
 * ogmdvd_title_get_palette:
 * @title: An #OGMDvdTitle
 *
 * Returns the palette of the movie.
 *
 * Returns: a constant array of 16 integers, or NULL
 */
G_CONST_RETURN guint *
ogmdvd_title_get_palette (OGMDvdTitle *title)
{
  pgc_t *pgc;
  guint pgn;

  g_return_val_if_fail (title != NULL, NULL);

  pgn = title->vts_file->vts_ptt_srpt->title[title->ttn - 1].ptt[0].pgn;
  pgc = title->vts_file->vts_pgcit->pgci_srp[title->pgcn - 1].pgc;

  return pgc->palette;
}

/**
 * ogmdvd_title_get_n_angles:
 * @title: An #OGMDvdTitle
 *
 * Returns the number of angles of the video title.
 *
 * Returns: The number of angles, or -1
 */
gint
ogmdvd_title_get_n_angles (OGMDvdTitle *title)
{
  g_return_val_if_fail (title != NULL, -1);

  return title->disc->vmg_file ? title->disc->vmg_file->tt_srpt->title[title->nr].nr_of_angles : 1;
}

/**
 * ogmdvd_title_get_n_chapters:
 * @title: An #OGMDvdTitle
 *
 * Returns the number of chapters of the video title.
 *
 * Returns: The number of chapters, or -1
 */
gint
ogmdvd_title_get_n_chapters (OGMDvdTitle *title)
{
  pgc_t *pgc;

  g_return_val_if_fail (title != NULL, -1);

  pgc = title->vts_file->vts_pgcit->pgci_srp[title->pgcn - 1].pgc;

  return pgc->nr_of_programs;
}

/**
 * ogmdvd_title_get_n_audio_streams:
 * @title: An #OGMDvdTitle
 *
 * Returns the number of audio streams of the video title.
 *
 * Returns: The number of audio streams, or -1
 */
gint
ogmdvd_title_get_n_audio_streams (OGMDvdTitle *title)
{
  gint i, naudio;

  g_return_val_if_fail (title != NULL, -1);

  for (naudio = i = 0; i < title->vts_file->vtsi_mat->nr_of_vts_audio_streams; i++)
    if (title->vts_file->vts_pgcit->pgci_srp[title->ttn - 1].pgc->audio_control[i] & 0x8000)
      naudio ++;

  return naudio;
}

static gint
ogmdvd_title_remap_audio_nr (OGMDvdTitle *title, guint nr)
{
  guint i = 0, j = 0;

  for (i = j = 0; i < title->vts_file->vtsi_mat->nr_of_vts_audio_streams; i++)
  {
    if (title->vts_file->vts_pgcit->pgci_srp[title->ttn - 1].pgc->audio_control[i] & 0x8000)
    {
      if (j == nr)
        return i;
      j ++;
    }
  }

  return -1;
}

static gint
ogmdvd_stream_find_by_nr (OGMDvdStream *stream, guint nr)
{
  return stream->nr - nr;
}

/**
 * ogmdvd_title_get_nth_audio_stream:
 * @title: An #OGMDvdTitle
 * @nr: The audio stream number
 *
 * Returns the audio stream at position nr. The first nr is 0.
 *
 * Returns: The #OGMDvdAudioStream, or NULL
 */
OGMDvdAudioStream *
ogmdvd_title_get_nth_audio_stream (OGMDvdTitle *title, guint nr)
{
  OGMDvdAudioStream *audio;
  GSList *link;

  g_return_val_if_fail (title != NULL, NULL);
  g_return_val_if_fail (nr < title->vts_file->vtsi_mat->nr_of_vts_audio_streams, NULL);

  link = g_slist_find_custom (title->audio_streams, GUINT_TO_POINTER (nr), (GCompareFunc) ogmdvd_stream_find_by_nr);
  if (link)
  {
    audio = link->data;
    audio->stream.ref ++;
  }
  else
  {
    gint real_nr;

    real_nr = ogmdvd_title_remap_audio_nr (title, nr);
    if (real_nr < 0)
      return NULL;

    ogmdvd_title_ref (title);

    audio = g_new0 (OGMDvdAudioStream, 1);
    audio->attr = &title->vts_file->vtsi_mat->vts_audio_attr[real_nr];
    audio->stream.title = title;
    audio->stream.nr = nr;
    audio->stream.ref = 1;

    audio->stream.id = title->vts_file->vts_pgcit->pgci_srp[title->ttn - 1].pgc->audio_control[real_nr] >> 8 & 7;

    title->audio_streams = g_slist_append (title->audio_streams, audio);
  }

  return audio;
}

/**
 * ogmdvd_title_get_audio_streams:
 * @title: An #OGMDvdTitle
 *
 * Returns a list of audio stream.
 *
 * Returns: The #GList, or NULL
 */
GList *
ogmdvd_title_get_audio_streams (OGMDvdTitle *title)
{
  OGMDvdAudioStream *stream;
  GList *list = NULL;
  guint nr;

  g_return_val_if_fail (title != NULL, NULL);

  for (nr = 0; nr < title->vts_file->vtsi_mat->nr_of_vts_audio_streams; nr ++)
  {
    stream = ogmdvd_title_get_nth_audio_stream (title, nr);
    if (stream)
      list = g_list_append (list, stream);
  }

  return list;
}

/**
 * ogmdvd_title_get_n_subp_streams:
 * @title: An #OGMDvdTitle
 *
 * Returns the number of subtitles streams of the video title.
 *
 * Returns: The number of subtitles streams, or -1
 */
gint
ogmdvd_title_get_n_subp_streams (OGMDvdTitle *title)
{
  gint i, nsubp;

  g_return_val_if_fail (title != NULL, -1);

  for (nsubp = i = 0; i < title->vts_file->vtsi_mat->nr_of_vts_subp_streams; i++)
    if (title->vts_file->vts_pgcit->pgci_srp[title->ttn - 1].pgc->subp_control[i] & 0x80000000)
      nsubp ++;

  return nsubp;
}

static gint
ogmdvd_title_remap_subp_nr (OGMDvdTitle *title, guint nr)
{
  guint i = 0, j = 0;

  for (i = j = 0; i < title->vts_file->vtsi_mat->nr_of_vts_subp_streams; i++)
  {
    if (title->vts_file->vts_pgcit->pgci_srp[title->ttn - 1].pgc->subp_control[i] & 0x80000000)
    {
      if (j == nr)
        return i;
      j ++;
    }
  }

  return -1;
}

/**
 * ogmdvd_title_get_nth_subp_stream:
 * @title: An #OGMDvdTitle
 * @nr: The subtitles stream number
 *
 * Returns the subtitles stream at position nr. The first nr is 0.
 *
 * Returns: The #OGMDvdSubpStream, or NULL
 */
OGMDvdSubpStream *
ogmdvd_title_get_nth_subp_stream (OGMDvdTitle *title, guint nr)
{
  OGMDvdSubpStream *subp;
  GSList *link;

  g_return_val_if_fail (title != NULL, NULL);
  g_return_val_if_fail (nr < title->vts_file->vtsi_mat->nr_of_vts_subp_streams, NULL);

  link = g_slist_find_custom (title->subp_streams, GUINT_TO_POINTER (nr), (GCompareFunc) ogmdvd_stream_find_by_nr);
  if (link)
  {
    subp = link->data;
    subp->stream.ref ++;
  }
  else
  {
    gint real_nr;

    real_nr = ogmdvd_title_remap_subp_nr (title, nr);
    if (real_nr < 0)
      return NULL;

    ogmdvd_title_ref (title);

    subp = g_new0 (OGMDvdSubpStream, 1);
    subp->attr = &title->vts_file->vtsi_mat->vts_subp_attr[real_nr];
    subp->stream.title = title;
    subp->stream.ref = 1;
    subp->stream.nr = nr;

    subp->stream.id = nr;
    if (title->vts_file->vtsi_mat->vts_video_attr.display_aspect_ratio == 0) /* 4:3 */
      subp->stream.id = title->vts_file->vts_pgcit->pgci_srp[title->ttn - 1].pgc->subp_control[real_nr] >> 24 & 31;
    else if (title->vts_file->vtsi_mat->vts_video_attr.display_aspect_ratio == 3) /* 16:9 */
      subp->stream.id = title->vts_file->vts_pgcit->pgci_srp[title->ttn - 1].pgc->subp_control[real_nr] >> 8 & 31;

    title->subp_streams = g_slist_append (title->subp_streams, subp);
  }

  return subp;
}

/**
 * ogmdvd_title_get_subp_streams:
 * @title: An #OGMDvdTitle
 *
 * Returns a list of subp stream.
 *
 * Returns: The #GList, or NULL
 */
GList *
ogmdvd_title_get_subp_streams (OGMDvdTitle *title)
{
  OGMDvdSubpStream *stream;
  GList *list = NULL;
  guint nr;

  g_return_val_if_fail (title != NULL, NULL);

  for (nr = 0; nr < title->vts_file->vtsi_mat->nr_of_vts_subp_streams; nr ++)
  {
    stream = ogmdvd_title_get_nth_subp_stream (title, nr);
    if (stream)
      list = g_list_append (list, stream);
  }

  return list;
}

/**
 * ogmdvd_title_analyze:
 * @title: An #OGMDvdTitle
 *
 * Performs a depper analysis of the title to get more information about it and
 * its audio and subtitle streams. This function should be called multipled timeѕ
 * until the analysis is complete.
 *
 * Returns: FALSE if the analysis if complete, TRUE otherwise
 */
gboolean
ogmdvd_title_analyze (OGMDvdTitle *title)
{
  gint status;

  g_return_val_if_fail (title != NULL, FALSE);

  if (!title->reader)
    title->reader = ogmdvd_reader_new (title, 0, -1, 0);

  if (!title->reader)
    return FALSE;

  if (!title->parser)
  {
    title->parser = ogmdvd_parser_new (title);
    title->buffer = g_new0 (guchar, 1024 * DVD_VIDEO_LB_LEN);
    title->block_len = 0;
  }

  if (!title->parser)
    return FALSE;

  if (title->block_len > 0)
  {
    title->ptr += DVD_VIDEO_LB_LEN;
    title->block_len --;
  }

  if (!title->block_len)
  {
    title->block_len = ogmdvd_reader_get_block (title->reader, 1024, title->buffer);
    if (title->block_len <= 0)
    {
      /* ERROR */
    }
    title->ptr = title->buffer;
  }

  status = ogmdvd_parser_analyze (title->parser, title->ptr);

  if (status)
  {
    gint i, n;

    n = ogmdvd_title_get_n_audio_streams (title);
    title->bitrates = g_new0 (gint, n);
    for (i = 0; i < n; i ++)
      title->bitrates[i] = ogmdvd_parser_get_audio_bitrate (title->parser, i);

    ogmdvd_parser_unref (title->parser);
    title->parser = NULL;

    ogmdvd_reader_unref (title->reader);
    title->reader = NULL;

    g_free (title->buffer);
    title->buffer = NULL;
    title->ptr = NULL;

    return FALSE;
  }

  return TRUE;
}

