/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2008 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "ogmrip-subp-options.h"
#include "ogmrip-helper.h"
#include "ogmrip-plugin.h"
#include "ogmrip-gconf.h"

#include <glib/gi18n.h>
#include <glade/glade.h>

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-profile-editor.glade"
#define OGMRIP_GLADE_ROOT "subtitles-page"

#define OGMRIP_SUBP_OPTIONS_DIALOG_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_SUBP_OPTIONS_DIALOG, OGMRipSubpOptionsDialogPriv))

struct _OGMRipSubpOptionsDialogPriv
{
  GType container;
  GtkWidget *root;
  GtkWidget *codec_combo;
  GtkWidget *default_button;
  GtkWidget *charset_combo;
  GtkWidget *eol_combo;
  GtkWidget *spell_check;
  GtkWidget *forced_subs_check;
  GtkWidget *text_table;
  GtkListStore *codec_store;
};

static void
ogmrip_subp_options_dialog_default_toggled (OGMRipSubpOptionsDialog *dialog)
{
  gint active;

  active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->default_button));
  gtk_widget_set_sensitive (dialog->priv->root, !active);
}

static void
ogmrip_subp_options_dialog_codec_changed (OGMRipSubpOptionsDialog *dialog, GtkWidget *combo)
{
  GtkTreeIter iter;

  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (combo), &iter))
  {
    GType codec;
    gchar *name;

    gtk_tree_model_get (GTK_TREE_MODEL (dialog->priv->codec_store), &iter, 1, &name, -1);
    codec = ogmrip_gconf_get_subp_codec_type (name);
    g_free (name);

    if (codec != G_TYPE_NONE)
      gtk_widget_set_sensitive (dialog->priv->text_table, ogmrip_plugin_get_subp_codec_text (codec));
  }
}

static void
ogmrip_subp_options_dialog_append_codec (GType type, const gchar *name, const gchar *description, OGMRipSubpOptionsDialog *dialog)
{
  GtkTreeIter iter;

  if (ogmrip_plugin_can_contain_subp (dialog->priv->container, type))
  {
    gtk_list_store_append (dialog->priv->codec_store, &iter);
    gtk_list_store_set (dialog->priv->codec_store, &iter, 0, description, 1, name, -1);
  }
}

static void
ogmrip_subp_options_dialog_populate_codec_combo (OGMRipSubpOptionsDialog *dialog)
{
  dialog->priv->container = ogmrip_gconf_get_container_type (NULL);
  ogmrip_plugin_foreach_subp_codec ((OGMRipPluginFunc) ogmrip_subp_options_dialog_append_codec, dialog);
}

static void
ogmrip_subp_options_dialog_construct_codec_combo (OGMRipSubpOptionsDialog *dialog)
{
  GtkCellRenderer *cell;
  gint n_codecs;

  dialog->priv->codec_store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  gtk_combo_box_set_model (GTK_COMBO_BOX (dialog->priv->codec_combo), GTK_TREE_MODEL (dialog->priv->codec_store));
  g_object_unref (dialog->priv->codec_store);

  cell = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (dialog->priv->codec_combo), cell, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (dialog->priv->codec_combo), cell, "text", 0, NULL);

  n_codecs = ogmrip_plugin_get_n_subp_codecs ();
  gtk_widget_set_sensitive (dialog->priv->codec_combo, n_codecs > 0);
}

G_DEFINE_TYPE (OGMRipSubpOptionsDialog, ogmrip_subp_options_dialog, GTK_TYPE_DIALOG);

static void
ogmrip_subp_options_dialog_class_init (OGMRipSubpOptionsDialogClass *klass)
{
  g_type_class_add_private (klass, sizeof (OGMRipSubpOptionsDialogPriv));
}

static void
ogmrip_subp_options_dialog_init (OGMRipSubpOptionsDialog *dialog)
{
  GtkWidget *alignment;
  GladeXML *xml;

  dialog->priv = OGMRIP_SUBP_OPTIONS_DIALOG_GET_PRIVATE (dialog);

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, OGMRIP_GLADE_ROOT, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return;
  }

  gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_CLOSE);

  gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Subtitles Options"));
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_PROPERTIES);

  alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 6, 0, 6, 6);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), alignment);
  gtk_widget_show (alignment);

  dialog->priv->default_button = gtk_check_button_new_with_label (_("Use profile settings"));
  gtk_container_add (GTK_CONTAINER (alignment), dialog->priv->default_button);
  gtk_widget_show (dialog->priv->default_button);

  dialog->priv->root = glade_xml_get_widget (xml, OGMRIP_GLADE_ROOT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), dialog->priv->root, TRUE, TRUE, 0);
  gtk_widget_show (dialog->priv->root);

  g_signal_connect_swapped (dialog->priv->default_button, "toggled",
      G_CALLBACK (ogmrip_subp_options_dialog_default_toggled), dialog);

  dialog->priv->codec_combo = glade_xml_get_widget (xml, "subp-codec-combo");
  ogmrip_subp_options_dialog_construct_codec_combo (dialog);
  ogmrip_subp_options_dialog_populate_codec_combo (dialog);

  g_signal_connect_swapped (dialog->priv->codec_combo, "changed",
      G_CALLBACK (ogmrip_subp_options_dialog_codec_changed), dialog);

  dialog->priv->charset_combo = glade_xml_get_widget (xml, "charset-combo");
  gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->charset_combo), OGMRIP_DEFAULT_SUBP_CHARSET);

  dialog->priv->eol_combo = glade_xml_get_widget (xml, "eol-combo");
  gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->eol_combo), OGMRIP_DEFAULT_SUBP_EOL);

  dialog->priv->spell_check = glade_xml_get_widget (xml, "spell-check");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->spell_check), OGMRIP_DEFAULT_SPELL_CHECK);

  dialog->priv->forced_subs_check = glade_xml_get_widget (xml, "forced-subs-check");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->forced_subs_check), OGMRIP_DEFAULT_FORCED_SUBS);

  dialog->priv->text_table = glade_xml_get_widget (xml, "text-table");

  g_object_unref (xml);
}

GtkWidget *
ogmrip_subp_options_dialog_new (void)
{
  return g_object_new (OGMRIP_TYPE_SUBP_OPTIONS_DIALOG, NULL);
}

void
ogmrip_subp_options_dialog_set_use_defaults (OGMRipSubpOptionsDialog *dialog, gboolean use_defaults)
{
  g_return_if_fail (OGMRIP_IS_SUBP_OPTIONS_DIALOG (dialog));

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->default_button), use_defaults);
}

gboolean
ogmrip_subp_options_dialog_get_use_defaults (OGMRipSubpOptionsDialog *dialog)
{
  g_return_val_if_fail (OGMRIP_IS_SUBP_OPTIONS_DIALOG (dialog), FALSE);

  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->default_button));
}

void
ogmrip_subp_options_dialog_set_codec (OGMRipSubpOptionsDialog *dialog, GType codec)
{
  GtkTreeModel *model;
  GtkTreeIter iter;

  model = GTK_TREE_MODEL (dialog->priv->codec_store);
  if (gtk_tree_model_iter_children (model, &iter, NULL))
  {
    const gchar *name;
    gchar *str;

    name = ogmrip_plugin_get_subp_codec_name (codec);
    do
    {
      gtk_tree_model_get (model, &iter, 1, &str, -1);
      if (g_str_equal (str, name))
      {
        gtk_combo_box_set_active_iter (GTK_COMBO_BOX (dialog->priv->codec_combo), &iter);
        g_free (str);
        break;
      }
      g_free (str);
    }
    while (gtk_tree_model_iter_next (model, &iter));

    if (gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->priv->codec_combo)) < 0)
      gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->codec_combo), 0);
  }
}

GType
ogmrip_subp_options_dialog_get_codec (OGMRipSubpOptionsDialog *dialog)
{
  GType codec;
  GtkTreeIter iter;
  gchar *name;

  if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX (dialog->priv->codec_combo), &iter))
    return G_TYPE_NONE;

  gtk_tree_model_get (GTK_TREE_MODEL (dialog->priv->codec_store), &iter, 1, &name, -1);
  codec =  ogmrip_gconf_get_subp_codec_type (name);
  g_free (name);

  return codec;
}

void
ogmrip_subp_options_dialog_set_charset (OGMRipSubpOptionsDialog *dialog, guint charset)
{
  g_return_if_fail (OGMRIP_IS_SUBP_OPTIONS_DIALOG (dialog));

  gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->charset_combo), charset);
}

gint
ogmrip_subp_options_dialog_get_charset (OGMRipSubpOptionsDialog *dialog)
{
  g_return_val_if_fail (OGMRIP_IS_SUBP_OPTIONS_DIALOG (dialog), -1);

  return gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->priv->charset_combo));
}

void
ogmrip_subp_options_dialog_set_eol (OGMRipSubpOptionsDialog *dialog, guint eol)
{
  g_return_if_fail (OGMRIP_IS_SUBP_OPTIONS_DIALOG (dialog));

  gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->priv->eol_combo), eol);
}

gint
ogmrip_subp_options_dialog_get_eol (OGMRipSubpOptionsDialog *dialog)
{
  g_return_val_if_fail (OGMRIP_IS_SUBP_OPTIONS_DIALOG (dialog), -1);

  return gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->priv->eol_combo));
}

void
ogmrip_subp_options_dialog_set_spell (OGMRipSubpOptionsDialog *dialog, gboolean spell)
{
  g_return_if_fail (OGMRIP_IS_SUBP_OPTIONS_DIALOG (dialog));

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->spell_check), spell);
}

gint
ogmrip_subp_options_dialog_get_spell (OGMRipSubpOptionsDialog *dialog)
{
  g_return_val_if_fail (OGMRIP_IS_SUBP_OPTIONS_DIALOG (dialog), FALSE);

  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->spell_check));
}

void
ogmrip_subp_options_dialog_set_forced_only (OGMRipSubpOptionsDialog *dialog, gboolean forced_only)
{
  g_return_if_fail (OGMRIP_IS_SUBP_OPTIONS_DIALOG (dialog));

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->forced_subs_check), forced_only);
}

gint
ogmrip_subp_options_dialog_get_forced_only (OGMRipSubpOptionsDialog *dialog)
{
  g_return_val_if_fail (OGMRIP_IS_SUBP_OPTIONS_DIALOG (dialog), FALSE);

  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->forced_subs_check));
}

