/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2009 Olivier Rolland <billl@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include "ogmrip-gconf.h"
#include "ogmrip-helper.h"
#include "ogmrip-marshal.h"
#include "ogmrip-plugin.h"
#include "ogmrip-preferences.h"
#include "ogmrip-profile-editor.h"
#include "ogmrip-profiles.h"

#include <libxml/parser.h>

#include <glib/gi18n.h>
#include <glade/glade.h>

#include <string.h>

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-profiles.glade"
#define OGMRIP_GLADE_ROOT "root"

#define OGMRIP_PROFILES_DIALOG_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_PROFILES_DIALOG, OGMRipProfilesDialogPriv))

enum
{
  COL_NAME,
  COL_KEY,
  COL_DLG,
  COL_LAST
};

struct _OGMRipProfilesDialogPriv
{
  GtkWidget *list;
  GtkWidget *new_button;
  GtkWidget *copy_button;
  GtkWidget *edit_button;
  GtkWidget *remove_button;
  GtkWidget *rename_button;
  GtkWidget *export_button;
};

enum
{
  NEW,
  REMOVE,
  RENAME,
  LAST_SIGNAL
};

static int signals[LAST_SIGNAL] = { 0 };

static const gchar * const * languages;

/*
 * Dialog
 */

static gint
ogmrip_profiles_dialog_compare_profiles (GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b, OGMRipProfilesDialog *dialog)
{
  gchar *aname, *bname;
  gint ret;

  gtk_tree_model_get (model, a, COL_NAME, &aname, -1);
  gtk_tree_model_get (model, b, COL_NAME, &bname, -1);

  ret = g_utf8_collate (aname, bname);

  g_free (aname);
  g_free (bname);

  return ret;
}

static void
ogmrip_profiles_dialog_construct_profiles (OGMRipProfilesDialog *dialog)
{
  GtkListStore *store;
  GtkCellRenderer *cell;
  GtkTreeViewColumn *column;

  store = gtk_list_store_new (COL_LAST, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER);
  gtk_tree_view_set_model (GTK_TREE_VIEW (dialog->priv->list), GTK_TREE_MODEL (store));
  g_object_unref (store);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_append_column (GTK_TREE_VIEW (dialog->priv->list), column);

  cell = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, cell, TRUE);
  gtk_tree_view_column_set_attributes (column, cell, "markup", COL_NAME, NULL);

  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (store), COL_NAME,
      (GtkTreeIterCompareFunc) ogmrip_profiles_dialog_compare_profiles, dialog, NULL);
  gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (store),
      COL_NAME, GTK_SORT_ASCENDING);
}

static gboolean
ogmrip_profiles_dialog_find_profile (GtkTreeModel *model, GtkTreeIter *iter, const gchar *key)
{
  gboolean retval = FALSE;

  if (gtk_tree_model_get_iter_first (model, iter))
  {
    gchar *str;

    do
    {
      gtk_tree_model_get (model, iter, COL_KEY, &str, -1);
      retval = g_str_equal (str, key);
      g_free (str);
    }
    while (!retval && gtk_tree_model_iter_next (model, iter));
  }

  return retval;
}

gboolean
ogmrip_profiles_check_profile (const gchar *dir, GError **error)
{
  GType type;
  gchar *key, *value;
  gboolean retval = TRUE;

  key = gconf_concat_dir_and_key (dir, "container/format");
  value = ogmrip_preferences_get_string (key, NULL);
  g_free (key);

  type = ogmrip_gconf_get_container_type (value);
  if (type == G_TYPE_NONE)
  {
    gchar *name;

    key = gconf_concat_dir_and_key (dir, "name");
    name = ogmrip_preferences_get_string (key, NULL);
    g_free (key);

    g_set_error (error, 0, 0, _("The container '%s' is not available"), value);
    g_free (name);

    retval = FALSE;
  }

  if (value)
    g_free (value);

  if (retval)
  {
    key = gconf_concat_dir_and_key (dir, "video/codec");
    value = ogmrip_preferences_get_string (key, NULL);
    g_free (key);

    type = ogmrip_gconf_get_video_codec_type (value);
    if (type == G_TYPE_NONE)
    {
      gchar *name;

      key = gconf_concat_dir_and_key (dir, "name");
      name = ogmrip_preferences_get_string (key, NULL);
      g_free (key);

      g_set_error (error, 0, 0, _("The video codec '%s' is not available"), value);
      g_free (name);

      retval = FALSE;
    }

    if (value)
      g_free (value);
  }

  if (retval)
  {
    key = gconf_concat_dir_and_key (dir, "audio/codec");
    value = ogmrip_preferences_get_string (key, NULL);
    g_free (key);

    type = ogmrip_gconf_get_audio_codec_type (value);
    if (type == G_TYPE_NONE)
    {
      gchar *name;

      key = gconf_concat_dir_and_key (dir, "name");
      name = ogmrip_preferences_get_string (key, NULL);
      g_free (key);

      g_set_error (error, 0, 0, _("The audio codec '%s' is not available"), value);
      g_free (name);

      retval = FALSE;
    }

    if (value)
      g_free (value);
  }

  if (retval)
  {
    key = gconf_concat_dir_and_key (dir, "subp/codec");
    value = ogmrip_preferences_get_string (key, NULL);
    g_free (key);

    type = ogmrip_gconf_get_subp_codec_type (value);
    if (type == G_TYPE_NONE)
    {
      gchar *name;

      key = gconf_concat_dir_and_key (dir, "name");
      name = ogmrip_preferences_get_string (key, NULL);
      g_free (key);

      g_set_error (error, 0, 0, _("The subtitles codec '%s' is not available"), value);
      g_free (name);

      retval = FALSE;
    }

    if (value)
      g_free (value);
  }

  return retval;
}

static void
ogmrip_profiles_dialog_add_profiles (OGMRipProfilesDialog *dialog, gboolean reload)
{
  GtkTreeModel *model;
  GtkTreeIter iter;

  GSList *dirs, *dir;
  const gchar *name;
  gchar *key;

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->priv->list));
  dirs = ogmrip_preferences_all_dirs (OGMRIP_GCONF_PROFILES);

  for (dir = dirs; dir; dir = dir->next)
  {
    if (ogmrip_profiles_check_profile (dir->data, NULL))
    {
      key = gconf_concat_dir_and_key (dir->data, "name");
      name = ogmrip_preferences_get_string (key, "");

      if (!reload || !ogmrip_profiles_dialog_find_profile (model, &iter, key))
      {
        gtk_list_store_append (GTK_LIST_STORE (model), &iter);
        gtk_list_store_set (GTK_LIST_STORE (model), &iter, COL_NAME, name, COL_KEY, dir->data, COL_DLG, NULL, -1);

        if (reload)
          g_signal_emit (dialog, signals[NEW], 0, name, dir->data);
      }

      g_free (key);
    }
  }

  g_slist_foreach (dirs, (GFunc) g_free, NULL);
  g_slist_free (dirs);
}

static void
ogmrip_profiles_dialog_profile_name_changed (GtkTextBuffer *buffer, GtkWidget *dialog)
{
  GtkTextIter start, end;
  gchar *text;

  gtk_text_buffer_get_bounds (buffer, &start, &end);
  text = gtk_text_buffer_get_text (buffer, &start, &end, TRUE);

  gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog),
      GTK_RESPONSE_ACCEPT, text != NULL && text[0] != '\0');

  g_free (text);
}

static gchar *
ogmrip_profiles_dialog_run_profile_dialog (GtkWindow *parent, const gchar *old_name)
{
  GtkWidget *dialog, *vbox, *label, *frame, *entry;
  GtkTextBuffer *buffer;

  gchar *new_name = NULL;

  dialog = gtk_dialog_new_with_buttons (old_name ? _("Rename profile") : _("New profile"), NULL,
      GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
      GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);

  gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT, FALSE);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);

  gtk_window_set_default_size (GTK_WINDOW (dialog), 250, 125);
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_PREFERENCES);
  gtk_window_set_parent (GTK_WINDOW (dialog), parent);

  vbox = gtk_vbox_new (FALSE, 6);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 6);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), vbox);
  gtk_widget_show (vbox);

  label = gtk_label_new (_("_Profile name:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_label_set_use_underline (GTK_LABEL (label), TRUE);
  gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show (frame);

  entry = gtk_text_view_new ();
  gtk_text_view_set_accepts_tab (GTK_TEXT_VIEW (entry), FALSE);
  gtk_container_add (GTK_CONTAINER (frame), entry);
  gtk_widget_show (entry);

  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (entry));

  if (old_name)
    gtk_text_buffer_set_text (buffer, old_name, -1);

  g_signal_connect (buffer, "changed",
      G_CALLBACK (ogmrip_profiles_dialog_profile_name_changed), dialog);

  gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);

  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
  {
    GtkTextIter start, end;

    gtk_text_buffer_get_bounds (buffer, &start, &end);
    new_name = gtk_text_buffer_get_text (buffer, &start, &end, TRUE);
  }
  gtk_widget_destroy (dialog);

  return new_name;
}

static void
ogmrip_profiles_dialog_new_button_clicked (OGMRipProfilesDialog *parent)
{
  gchar *name;

  name = ogmrip_profiles_dialog_run_profile_dialog (GTK_WINDOW (parent), NULL);
  if (name)
  {
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    GtkTreeIter iter;
    gchar *key, *dir;

    key = gconf_unique_key ();
    dir = gconf_concat_dir_and_key (OGMRIP_GCONF_PROFILES, key);
    g_free (key);

    model = gtk_tree_view_get_model (GTK_TREE_VIEW (parent->priv->list));
    gtk_list_store_append (GTK_LIST_STORE (model), &iter);
    gtk_list_store_set (GTK_LIST_STORE (model), &iter, COL_NAME, name, COL_KEY, dir, -1);

    key = gconf_concat_dir_and_key (dir, "name");
    ogmrip_preferences_set_string (key, name);
    g_free (key);

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (parent->priv->list));
    gtk_tree_selection_select_iter (selection, &iter);

    g_signal_emit (parent, signals[NEW], 0, name, dir);
    g_free (dir); g_free (name);
  }
}

static void
ogmrip_profiles_dialog_profile_dialog_destroyed (GtkWidget *dialog, GtkTreeRowReference *ref)
{
  if (gtk_tree_row_reference_valid (ref))
  {
    GtkTreePath *path;

    path = gtk_tree_row_reference_get_path (ref);
    if (path)
    {
      GtkTreeModel *model;
      GtkTreeIter iter;

      model = gtk_tree_row_reference_get_model (ref);
      if (gtk_tree_model_get_iter (model, &iter, path))
        gtk_list_store_set (GTK_LIST_STORE (model), &iter, COL_DLG, NULL, -1);

      gtk_tree_path_free (path);
    }
  }

  gtk_tree_row_reference_free (ref);
}

static gboolean
ogmrip_profiles_dialog_selection_changed (OGMRipProfilesDialog *dialog, GtkTreeSelection *selection)
{
  GtkTreeIter iter;
  gboolean sensitive;

  sensitive = gtk_tree_selection_get_selected (selection, NULL, &iter);

  gtk_widget_set_sensitive (dialog->priv->copy_button, sensitive);
  gtk_widget_set_sensitive (dialog->priv->edit_button, sensitive);
  gtk_widget_set_sensitive (dialog->priv->remove_button, sensitive);
  gtk_widget_set_sensitive (dialog->priv->rename_button, sensitive);
  gtk_widget_set_sensitive (dialog->priv->export_button, sensitive);

  return TRUE;
}

static gchar *
get_profile_title (const gchar *name)
{
  gchar *str1, *str2;
  gint i;

  for (i = 0; name[i] != '\0'; i++)
    if (name[i] == '\n' || name[i] == '\r')
      break;

  if (name[i] == '\0')
    return g_strdup (name);

  str1 = g_strndup (name, i);

  if (!pango_parse_markup (str1, -1, 0, NULL, &str2, NULL, NULL))
    str2 = g_strdup (name);

  g_free (str1);

  str1 = g_strdup_printf (_("Editing profile \"%s\""), str2);
  g_free (str2);

  return str1;
}

static void
ogmrip_profiles_dialog_copy_dir (const gchar *current_dir, const gchar *new_dir, guint offset)
{
  GSList *list, *link;
  GConfEntry *entry;
  gchar *dir, *key;

  list = ogmrip_preferences_all_entries (current_dir);
  for (link = list; link; link = link->next)
  {
    entry = link->data;

    key = gconf_concat_dir_and_key (new_dir, entry->key + offset);
    ogmrip_preferences_set (key, entry->value);
    g_free (key);

    gconf_entry_free (entry);
  }
  g_slist_free (list);

  list = ogmrip_preferences_all_dirs (current_dir);
  for (link = list; link; link = link->next)
  {
    dir = (gchar *) link->data;
    ogmrip_profiles_dialog_copy_dir (dir, new_dir, offset);
    g_free (dir);
  }
  g_slist_free (list);
}

static void
ogmrip_profiles_dialog_copy_button_clicked (OGMRipProfilesDialog *parent)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (parent->priv->list));
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
  {
    gchar *name, *new_name, *dir, *new_dir, *key;

    name = gconf_unique_key ();
    new_dir = gconf_concat_dir_and_key (OGMRIP_GCONF_PROFILES, name);
    g_free (name);

    gtk_tree_model_get (model, &iter, COL_NAME, &name, COL_KEY, &dir, -1);

    new_name = g_strconcat (_("Copy of"), " ", name, NULL);
    g_free (name);

    ogmrip_profiles_dialog_copy_dir (dir, new_dir, strlen (dir));
    g_free (dir);

    gtk_list_store_append (GTK_LIST_STORE (model), &iter);
    gtk_list_store_set (GTK_LIST_STORE (model), &iter, COL_NAME, new_name, COL_KEY, new_dir, -1);

    key = gconf_concat_dir_and_key (new_dir, "name");
    ogmrip_preferences_set_string (key, new_name);
    g_free (key);

    gtk_tree_selection_select_iter (selection, &iter);
    g_signal_emit (parent, signals[NEW], 0, new_name, new_dir);

    g_free (new_dir);
    g_free (new_name);
  }
}

static void
ogmrip_profiles_dialog_edit_button_clicked (OGMRipProfilesDialog *parent)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (parent->priv->list));
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
  {
    GtkWidget *dialog;
    gchar *name, *key;

    gtk_tree_model_get (model, &iter, COL_NAME, &name, COL_KEY, &key, COL_DLG, &dialog, -1);
    if (!dialog)
    {
      GtkTreePath *path;
      GtkTreeRowReference *ref;
      gchar *title;

      dialog = ogmrip_profile_editor_dialog_new (key);

      gtk_window_set_parent (GTK_WINDOW (dialog), GTK_WINDOW (parent));
      gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), FALSE);

      title = get_profile_title (name);
      gtk_window_set_title (GTK_WINDOW (dialog), title);
      g_free (title);

      g_signal_connect (dialog, "delete-event", G_CALLBACK (gtk_true), NULL);
      g_signal_connect (dialog, "response", G_CALLBACK (gtk_widget_destroy), NULL);

      path = gtk_tree_model_get_path (model, &iter);
      ref = gtk_tree_row_reference_new (model, path);
      gtk_tree_path_free (path);

      g_signal_connect (dialog, "destroy", G_CALLBACK (ogmrip_profiles_dialog_profile_dialog_destroyed), ref);

      gtk_list_store_set (GTK_LIST_STORE (model), &iter, COL_DLG, dialog, -1);
    }
    g_free (name);
    g_free (key);

    gtk_window_present (GTK_WINDOW (dialog));
  }
}

static void
ogmrip_profiles_dialog_remove_button_clicked (OGMRipProfilesDialog *parent)
{
  GtkTreeIter iter;
  GtkTreeSelection *selection;
  GtkTreeModel *model;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (parent->priv->list));
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
  {
    GtkWidget *dialog;
    gchar *key, *name;

    gtk_tree_model_get (model, &iter, COL_NAME, &name, COL_KEY, &key, COL_DLG, &dialog, -1);

    if (dialog)
    {
      dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL,
          GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, _("Cannot remove profile"));
      gtk_window_set_title (GTK_WINDOW (dialog), _("Cannot remove profile"));

      gtk_message_dialog_format_secondary_markup (GTK_MESSAGE_DIALOG (dialog), "%s", name);

      gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
      gtk_window_set_parent (GTK_WINDOW (dialog), GTK_WINDOW (parent));

      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
    }
    else
    {
      dialog = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL,
          GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK_CANCEL, _("Delete profile ?"));
      gtk_window_set_title (GTK_WINDOW (dialog), _("Delete profile ?"));

      gtk_message_dialog_format_secondary_markup (GTK_MESSAGE_DIALOG (dialog), "%s", name);

      gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
      gtk_window_set_parent (GTK_WINDOW (dialog), GTK_WINDOW (parent));

      if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
      {
        gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

        if (!g_str_equal (key, OGMRIP_GCONF_PROFILES "/" OGMRIP_DEFAULT_PROFILE))
          ogmrip_preferences_recursive_unset (key);

        if (!gtk_tree_model_iter_n_children (model, NULL))
          ogmrip_preferences_set_string (OGMRIP_GCONF_PROFILE, OGMRIP_DEFAULT_PROFILE);

        g_signal_emit (parent, signals[REMOVE], 0, name, key);

        if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (model), &iter))
          gtk_tree_selection_select_iter (selection, &iter);
      }
      gtk_widget_destroy (dialog);
    }

    g_free (name);
    g_free (key);
  }
}

static void
ogmrip_profiles_dialog_rename_button_clicked (OGMRipProfilesDialog *parent)
{
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  GtkTreeIter iter;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (parent->priv->list));
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
  {
    gchar *old_name, *new_name, *key;

    gtk_tree_model_get (model, &iter, COL_NAME, &old_name, COL_KEY, &key, -1);
    new_name = ogmrip_profiles_dialog_run_profile_dialog (GTK_WINDOW (parent), old_name);
  
    if (new_name && !g_str_equal (new_name, old_name))
    {
      gchar *str;

      str = gconf_concat_dir_and_key (key, "name");
      ogmrip_preferences_set_string (str, new_name);
      g_free (str);

      gtk_list_store_set (GTK_LIST_STORE (model), &iter, COL_NAME, new_name, -1);

      g_signal_emit (parent, signals[RENAME], 0, new_name, key);
    }

    g_free (new_name);
    g_free (old_name);
    g_free (key);
  }
}

static void
ogmrip_profiles_dialog_export_button_clicked (OGMRipProfilesDialog *parent)
{
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  GtkTreeIter iter;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (parent->priv->list));
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
  {
    GtkWidget *dialog;

    dialog = gtk_file_chooser_dialog_new (_("Export profile as"),
        GTK_WINDOW (parent), GTK_FILE_CHOOSER_ACTION_SAVE,
        GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
        GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
        NULL);
    gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog), TRUE);

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      gchar *filename, *key;

      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      gtk_tree_model_get (model, &iter, COL_KEY, &key, -1);

      ogmrip_profiles_export (filename, key, NULL);

      g_free (filename);
      g_free (key);
    }
    gtk_widget_destroy (dialog);
  }
}

static void
ogmrip_profiles_dialog_import_button_clicked (OGMRipProfilesDialog *parent)
{
  GtkWidget *dialog;

  dialog = gtk_file_chooser_dialog_new (_("Select profile to import"),
      GTK_WINDOW (parent), GTK_FILE_CHOOSER_ACTION_OPEN,
      GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
      NULL);

  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
  {
    GError *error = NULL;
    gchar *filename;

    gtk_widget_hide (dialog);

    filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    if (ogmrip_profiles_import (filename, &error))
    {
      GtkTreeModel *model;

      model = gtk_tree_view_get_model (GTK_TREE_VIEW (parent->priv->list));
      gtk_list_store_clear (GTK_LIST_STORE (model));

      ogmrip_profiles_dialog_add_profiles (parent, TRUE);
    }
    else
    {
      gtk_widget_destroy (dialog);
      dialog = gtk_message_dialog_new (GTK_WINDOW (parent),
          GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
          GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, _("Cannot load profile"));
      gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog), 
          "%s", error ? error->message : _("Unknown error"));
      g_error_free (error);

      gtk_dialog_run (GTK_DIALOG (dialog));
    }
    g_free (filename);
  }
  gtk_widget_destroy (dialog);
}

G_DEFINE_TYPE (OGMRipProfilesDialog, ogmrip_profiles_dialog, GTK_TYPE_DIALOG);

static void
ogmrip_profiles_dialog_class_init (OGMRipProfilesDialogClass *klass)
{
  g_type_class_add_private (klass, sizeof (OGMRipProfilesDialogPriv));

  signals[NEW] = g_signal_new ("new-profile", G_TYPE_FROM_CLASS (klass), 
      G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
      G_STRUCT_OFFSET (OGMRipProfilesDialogClass, new_profile), NULL, NULL,
      ogmrip_cclosure_marshal_VOID__STRING_STRING,
      G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_STRING);

  signals[REMOVE] = g_signal_new ("remove-profile", G_TYPE_FROM_CLASS (klass), 
      G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
      G_STRUCT_OFFSET (OGMRipProfilesDialogClass, remove_profile), NULL, NULL,
      ogmrip_cclosure_marshal_VOID__STRING_STRING,
      G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_STRING);

  signals[RENAME] = g_signal_new ("rename-profile", G_TYPE_FROM_CLASS (klass), 
      G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
      G_STRUCT_OFFSET (OGMRipProfilesDialogClass, rename_profile), NULL, NULL,
      ogmrip_cclosure_marshal_VOID__STRING_STRING,
      G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_STRING);
}

static void
ogmrip_profiles_dialog_init (OGMRipProfilesDialog *dialog)
{
  GtkWidget *widget;
  GtkTreeSelection *selection;
  GladeXML *xml;

  dialog->priv = OGMRIP_PROFILES_DIALOG_GET_PRIVATE (dialog);

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, OGMRIP_GLADE_ROOT, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return;
  }

  gtk_dialog_add_buttons (GTK_DIALOG (dialog),
      GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
      NULL);
  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_window_set_default_size (GTK_WINDOW (dialog), 450, -1);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Edit Profiles"));
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_PREFERENCES);

  widget = glade_xml_get_widget (xml, OGMRIP_GLADE_ROOT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), widget, TRUE, TRUE, 0);
  gtk_widget_show (widget);

  dialog->priv->edit_button = glade_xml_get_widget (xml, "edit-button");
  gtk_widget_set_sensitive (dialog->priv->edit_button, FALSE);
  g_signal_connect_swapped (dialog->priv->edit_button, "clicked",
      G_CALLBACK (ogmrip_profiles_dialog_edit_button_clicked), dialog);

  dialog->priv->new_button = glade_xml_get_widget (xml, "new-button");
  g_signal_connect_swapped (dialog->priv->new_button, "clicked",
      G_CALLBACK (ogmrip_profiles_dialog_new_button_clicked), dialog);

  dialog->priv->remove_button = glade_xml_get_widget (xml, "delete-button");
  gtk_widget_set_sensitive (dialog->priv->remove_button, FALSE);
  g_signal_connect_swapped (dialog->priv->remove_button, "clicked",
      G_CALLBACK (ogmrip_profiles_dialog_remove_button_clicked), dialog);

  dialog->priv->copy_button = glade_xml_get_widget (xml, "copy-button");
  gtk_widget_set_sensitive (dialog->priv->copy_button, FALSE);
  g_signal_connect_swapped (dialog->priv->copy_button, "clicked",
      G_CALLBACK (ogmrip_profiles_dialog_copy_button_clicked), dialog);

  dialog->priv->rename_button = glade_xml_get_widget (xml, "rename-button");
  gtk_widget_set_sensitive (dialog->priv->rename_button, FALSE);
  g_signal_connect_swapped (dialog->priv->rename_button, "clicked",
      G_CALLBACK (ogmrip_profiles_dialog_rename_button_clicked), dialog);

  dialog->priv->export_button = glade_xml_get_widget (xml, "export-button");
  gtk_widget_set_sensitive (dialog->priv->export_button, FALSE);
  g_signal_connect_swapped (dialog->priv->export_button, "clicked",
      G_CALLBACK (ogmrip_profiles_dialog_export_button_clicked), dialog);

  widget = glade_xml_get_widget (xml, "import-button");
  g_signal_connect_swapped (widget, "clicked",
      G_CALLBACK (ogmrip_profiles_dialog_import_button_clicked), dialog);

  dialog->priv->list = glade_xml_get_widget (xml, "treeview");
  ogmrip_profiles_dialog_construct_profiles (dialog);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->priv->list));
  g_signal_connect_swapped (selection, "changed",
      G_CALLBACK (ogmrip_profiles_dialog_selection_changed), dialog);

  ogmrip_profiles_dialog_add_profiles (dialog, FALSE);

  g_object_unref (xml);
}

GtkWidget *
ogmrip_profiles_dialog_new (void)
{
  return g_object_new (OGMRIP_TYPE_PROFILES_DIALOG, NULL);
}

void
ogmrip_profiles_dialog_set_active (OGMRipProfilesDialog *dialog, const gchar *profile)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *key;

  g_return_if_fail (OGMRIP_IS_PROFILES_DIALOG (dialog));

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->priv->list));

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->priv->list));
  if (gtk_tree_model_get_iter_first (model, &iter))
  {
    if (profile)
    {
      do
      {
        gtk_tree_model_get (model, &iter, COL_KEY, &key, -1);
        if (key)
        {
          if (g_str_equal (key, profile))
          {
            g_free (key);
            break;
          }
          g_free (key);
        }
      }
      while (gtk_tree_model_iter_next (model, &iter));
    }

    gtk_tree_selection_select_iter (selection, &iter);
  }
}

static inline const gchar*
get_key_relative(const gchar* key, const gchar* dir)
{
  int i;

  for (i = 0; dir[i]; i++)
    if (dir [i] != key [i])
      return key;

  if (key [i] != '/' || key [i] == '\0')
    return key;

  return key + i + 1;
}

gboolean
ogmrip_profiles_check_language (xmlNode *node)
{
  xmlChar *lang;
  guint i;

  lang = xmlNodeGetLang (node);
  if (!lang)
    return FALSE;

  for (i = 0; languages[i]; i++)
    if (xmlStrEqual ((xmlChar *) languages[i], lang))
      break;

  xmlFree (lang);

  return languages[i] != NULL;
}

static GConfValue *
ogmrip_profiles_get_value (xmlNode *node)
{
  xmlNode *iter;
  GConfValue *value = NULL;
  GConfValueType type = GCONF_VALUE_INVALID;
  gchar *content = NULL;

  for (iter = node->xmlChildrenNode; iter; iter = iter->next)
    if (iter->type == XML_ELEMENT_NODE)
      break;

  if (!iter)
    return NULL;

  if (g_str_equal ((gchar *) iter->name, "int"))
    type = GCONF_VALUE_INT;
  else if (g_str_equal ((gchar *) iter->name, "float"))
    type = GCONF_VALUE_FLOAT;
  else if (g_str_equal ((gchar *) iter->name, "bool"))
    type = GCONF_VALUE_BOOL;
  else if (g_str_equal ((gchar *) iter->name, "string"))
  {
    xmlNode *iter2;

    for (iter2 = iter; iter2; iter2 = iter2->next)
      if (ogmrip_profiles_check_language (iter2))
        break;

    if (iter2)
      iter = iter2;

    type = GCONF_VALUE_STRING;
  }

  if (type == GCONF_VALUE_INVALID)
    return NULL;

  content = (gchar *) xmlNodeGetContent (iter);
  if (!content)
    return NULL;

  value = gconf_value_new_from_string (type, content, NULL);
  xmlFree (content);

  return value;
}

static void
ogmrip_profiles_parse_entry (xmlNode *node, const gchar *dir)
{
  xmlNode *iter;
  GConfValue *value = NULL;
  char* key = NULL;

  for (iter = node->xmlChildrenNode; iter; iter = iter->next)
  {
    if (g_str_equal ((char *) iter->name, "key"))
      key = (char *) xmlNodeGetContent (iter);
    else if (g_str_equal ((char *) iter->name, "value"))
      value = ogmrip_profiles_get_value (iter);
  }

  if (key && value)
  {
    gchar *full_key;

    full_key = gconf_concat_dir_and_key (dir, key);
    ogmrip_preferences_set (full_key, value);
    g_free (full_key);
  }

  if (value)
    gconf_value_free (value);

  if (key)
    xmlFree (key);
}

static gboolean
ogmrip_profiles_parse_profile (xmlNode *root, GError **error)
{
  gboolean retval = TRUE;
  gchar *dir;

  dir = (char *) xmlGetProp (root, (xmlChar *) "base");
  if (!ogmrip_preferences_dir_exists (dir))
  {
    GError *tmp_error = NULL;
    xmlNode *iter;

    for (iter = root->xmlChildrenNode; iter; iter = iter->next)
      if (iter->type == XML_ELEMENT_NODE && g_str_equal ((char *) iter->name, "entry"))
        ogmrip_profiles_parse_entry (iter, dir);

    if (!ogmrip_profiles_check_profile (dir, &tmp_error))
    {
      ogmrip_preferences_recursive_unset (dir);
      if (error == NULL || *error == NULL)
        g_propagate_error (error, tmp_error);
      else if (tmp_error)
        g_error_free (tmp_error);
      retval = FALSE;
    }
  }

  xmlFree (dir);

  return retval;
}

gboolean
ogmrip_profiles_import (const gchar *filename, GError **error)
{
  gboolean retval = TRUE;
  xmlDocPtr doc;
  xmlNode *iter;

  g_return_val_if_fail (filename != NULL, FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  if (!languages)
    languages = g_get_language_names ();

  doc = xmlParseFile (filename);
  if (!doc)
  {
    g_set_error (error, 0, 0, _("Failed to open '%s'"), filename);
    return FALSE;
  }

  iter = xmlDocGetRootElement (doc);

  if (!iter)
  {
    g_set_error (error, 0, 0, _("'%s' does not contain a valid profile"), filename);
    xmlFreeDoc (doc);
    return FALSE;
  }

  while (iter != NULL)
  {
    if (iter->type == XML_ELEMENT_NODE)
    {
      if (!g_str_equal((char *)iter->name, "ogmrip"))
      {
        g_set_error (error, 0, 0, _("'%s' does not contain a valid profile"), filename);
        xmlFreeDoc (doc);
        return FALSE;
      }
      else
        break;
    }

    iter = iter->next;
  }

  if (!iter)
  {
    g_set_error (error, 0, 0, _("'%s' does not contain a valid profile"), filename);
    xmlFreeDoc (doc);
    return FALSE;
  }

  iter = iter->xmlChildrenNode;

  while (iter)
  {
    if (iter->type == XML_ELEMENT_NODE)
    {
      if (!g_str_equal ((char *) iter->name, "profile"))
      {
        g_set_error (error, 0, 0, _("'%s' does not contain a valid profile"), filename);
        xmlFreeDoc (doc);
        return FALSE;
      }

      if (!ogmrip_profiles_parse_profile (iter, error))
        retval = FALSE;
    }
    iter = iter->next;
  }

  xmlFreeDoc (doc);

  return retval;
}

gboolean
ogmrip_profiles_import_all (const gchar *dirname, GError **error)
{
  GError *tmp_error = NULL;
  const gchar *filename;
  gchar *fullname;
  GDir *dir;

  dir = g_dir_open (dirname, 0, &tmp_error);
  if (!dir)
  {
    g_propagate_error (error, tmp_error);
    return FALSE;
  }

  while ((filename = g_dir_read_name (dir)))
  {
    fullname = g_build_filename (dirname, filename, NULL);
    ogmrip_profiles_import (fullname, NULL);
    g_free (fullname);
  }

  g_dir_close (dir);

  return TRUE;
}

static void
ogmrip_profiles_dump_value (FILE *f, GConfValue *value, GError **error)
{
  gchar *tmp;

  fprintf (f, "      <value>\n");

  switch (value->type)
    {
    case GCONF_VALUE_INT:
      tmp = gconf_value_to_string (value);
      fprintf (f, "        <int>%s</int>\n", tmp);
      g_free (tmp);
      break;
    case GCONF_VALUE_FLOAT:
      tmp = gconf_value_to_string(value);
      fprintf (f, "        <float>%s</float>\n", tmp);
      g_free (tmp);
      break;
    case GCONF_VALUE_STRING:
      tmp = g_markup_escape_text (gconf_value_get_string (value), -1);
      fprintf (f, "        <string>%s</string>\n", (tmp[0] == ' ' && tmp[1] == '\0') ? "" : tmp);
      g_free (tmp);
      break;
    case GCONF_VALUE_BOOL:
      tmp = gconf_value_to_string (value);
      fprintf (f, "        <bool>%s</bool>\n", tmp);
      g_free (tmp);
      break;
    case GCONF_VALUE_LIST:
    case GCONF_VALUE_PAIR:
    case GCONF_VALUE_SCHEMA:
      break;
    default:
      g_assert_not_reached();
      break;
    }

  fprintf (f, "      </value>\n");
}

static void
ogmrip_profiles_dump_dir (FILE *f, const gchar *dir, const gchar *base_dir, GError **error)
{
  GSList *list, *link;
  GConfEntry* entry;

  list = ogmrip_preferences_all_entries (dir);
  for (link = list; link; link = link->next)
  {
    entry = link->data;

    fprintf (f, "    <entry>\n");
    fprintf (f, "      <key>%s</key>\n", get_key_relative (gconf_entry_get_key (entry), base_dir));

    if (entry->value)
      ogmrip_profiles_dump_value (f, entry->value, error);

    fprintf (f, "    </entry>\n");

    gconf_entry_free (entry);
  }
  g_slist_free (list);

  list = ogmrip_preferences_all_dirs (dir);
  for (link = list; link; link = link->next)
  {
    ogmrip_profiles_dump_dir (f, link->data, base_dir, error);
    g_free (link->data);
  }
  g_slist_free (list);
}

gboolean
ogmrip_profiles_export (const gchar *filename, const gchar *dir, GError **error)
{
  FILE *f;

  g_return_val_if_fail (filename != NULL, FALSE);
  g_return_val_if_fail (dir != NULL, FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  f = fopen (filename, "w");
  if (!f)
    return FALSE;

  fprintf (f, "<ogmrip>\n");
  fprintf (f, "  <profile base=\"%s\">\n", dir);

  ogmrip_profiles_dump_dir (f, dir, dir, error);

  fprintf (f, "  </profile>\n");
  fprintf (f, "</ogmrip>\n");

  fclose (f);

  return TRUE;
}

