/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2009 Olivier Rolland <billl@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n.h>
#include <glade/glade.h>

#include "ogmrip-helper.h"
#include "ogmrip-options-plugin.h"
#include "ogmrip-plugin.h"
#include "ogmrip-settings.h"
#include "ogmrip-x264.h"

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-x264.glade"
#define OGMRIP_GLADE_ROOT "root"

#define OGMRIP_TYPE_X264_DIALOG          (ogmrip_x264_dialog_get_type ())
#define OGMRIP_X264_DIALOG(obj)          (G_TYPE_CHECK_INSTANCE_CAST ((obj), OGMRIP_TYPE_X264_DIALOG, OGMRipX264Dialog))
#define OGMRIP_X264_DIALOG_CLASS(klass)  (G_TYPE_CHECK_CLASS_CAST ((klass), OGMRIP_TYPE_X264_DIALOG, OGMRipX264DialogClass))
#define OGMRIP_IS_X264_DIALOG(obj)       (G_TYPE_CHECK_INSTANCE_TYPE ((obj), OGMRIP_TYPE_X264_DIALOG))
#define OGMRIP_IS_X264_DIALOG_CLASS(obj) (G_TYPE_CHECK_CLASS_TYPE ((klass), OGMRIP_TYPE_X264_DIALOG))

#define OGMRIP_X264_KEY_8X8DCT        OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_8X8DCT
#define OGMRIP_X264_KEY_AUD           OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_AUD
#define OGMRIP_X264_KEY_BFRAMES       OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_BFRAMES
#define OGMRIP_X264_KEY_B_ADAPT       OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_B_ADAPT
#define OGMRIP_X264_KEY_B_PYRAMID     OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_B_PYRAMID
#define OGMRIP_X264_KEY_BRDO          OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_BRDO
#define OGMRIP_X264_KEY_CABAC         OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_CABAC
#define OGMRIP_X264_KEY_DIRECT        OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_DIRECT
#define OGMRIP_X264_KEY_FRAMEREF      OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_FRAMEREF
#define OGMRIP_X264_KEY_GLOBAL_HEADER OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_GLOBAL_HEADER
#define OGMRIP_X264_KEY_LEVEL_IDC     OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_LEVEL_IDC
#define OGMRIP_X264_KEY_ME            OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_ME
#define OGMRIP_X264_KEY_MERANGE       OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_MERANGE
#define OGMRIP_X264_KEY_MIXED_REFS    OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_MIXED_REFS
#define OGMRIP_X264_KEY_PSY_RD        OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_PSY_RD
#define OGMRIP_X264_KEY_PSY_TRELLIS   OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_PSY_TRELLIS
#define OGMRIP_X264_KEY_RC_LOOKAHEAD  OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_RC_LOOKAHEAD
#define OGMRIP_X264_KEY_SUBQ          OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_SUBQ
#define OGMRIP_X264_KEY_VBV_BUFSIZE   OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_VBV_BUFSIZE
#define OGMRIP_X264_KEY_VBV_MAXRATE   OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_VBV_MAXRATE
#define OGMRIP_X264_KEY_WEIGHT_B      OGMRIP_X264_SECTION "/" OGMRIP_X264_PROP_WEIGHT_B

typedef struct _OGMRipX264Dialog      OGMRipX264Dialog;
typedef struct _OGMRipX264DialogClass OGMRipX264DialogClass;

struct _OGMRipX264Dialog
{
  OGMRipPluginDialog parent_instance;

  GtkWidget *aud_check;
  GtkWidget *b_pyramid_check;
  GtkWidget *brdo_check;
  GtkWidget *cabac_check;
  GtkWidget *global_header_check;
  GtkWidget *mixed_refs_check;
  GtkWidget *weight_b_check;
  GtkWidget *x88dct_check;

  GtkWidget *b_adapt_spin;
  GtkWidget *bframes_spin;
  GtkWidget *frameref_spin;
  GtkWidget *level_idc_spin;
  GtkWidget *merange_spin;
  GtkWidget *psy_rd_spin;
  GtkWidget *psy_trellis_spin;
  GtkWidget *rc_lookahead_spin;
  GtkWidget *subq_spin;
  GtkWidget *vbv_bufsize_spin;
  GtkWidget *vbv_maxrate_spin;

  GtkWidget *direct_combo;
  GtkWidget *me_combo;
};

struct _OGMRipX264DialogClass
{
  OGMRipPluginDialogClass parent_class;
};

GType ogmrip_x264_get_type ();

static void ogmrip_x264_dialog_set_section   (OGMRipPluginDialog *dialog,
                                              const gchar        *section);

static gboolean x264_have_brdo      = FALSE;
static gboolean x264_have_psy       = FALSE;
static gboolean x264_have_aud       = FALSE;
static gboolean x264_have_lookahead = FALSE;
static gboolean x264_have_me_tesa   = FALSE;

static void
ogmrip_x264_dialog_bframes_changed (OGMRipX264Dialog *dialog)
{
  gint bframes;

  bframes = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->bframes_spin));

  gtk_widget_set_sensitive (dialog->b_pyramid_check, bframes > 1);
  gtk_widget_set_sensitive (dialog->weight_b_check, bframes > 1);

  if (bframes <= 1)
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->b_pyramid_check), FALSE);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->weight_b_check), FALSE);
  }
}

static void
ogmrip_x264_dialog_subq_changed (OGMRipX264Dialog *dialog)
{
  gint subq;

  subq = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->subq_spin));

  gtk_widget_set_sensitive (dialog->brdo_check, x264_have_brdo && subq > 5);
  gtk_widget_set_sensitive (dialog->psy_rd_spin, x264_have_psy && subq > 5);

  if (subq <= 5)
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->brdo_check), FALSE);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog->psy_rd_spin), 0.0);
  }
}

static void
ogmrip_x264_dialog_frameref_changed (OGMRipX264Dialog *dialog)
{
  gint frameref;

  frameref = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->frameref_spin));

  gtk_widget_set_sensitive (dialog->mixed_refs_check, frameref > 1);
  if (frameref <= 1)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->mixed_refs_check), FALSE);
}

static void
ogmrip_x264_dialog_vbv_maxrate_changed (OGMRipX264Dialog *dialog)
{
  gtk_widget_set_sensitive (dialog->vbv_bufsize_spin,
      gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog->vbv_maxrate_spin)) > 0);
}

G_DEFINE_TYPE (OGMRipX264Dialog, ogmrip_x264_dialog, OGMRIP_TYPE_PLUGIN_DIALOG)

static void
ogmrip_x264_dialog_class_init (OGMRipX264DialogClass *klass)
{
  OGMRipPluginDialogClass *dialog_class;

  dialog_class = (OGMRipPluginDialogClass *) klass;
  dialog_class->set_section = ogmrip_x264_dialog_set_section;
}

static void
ogmrip_x264_dialog_init (OGMRipX264Dialog *dialog)
{
  GtkWidget *widget;
  GladeXML *xml;

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, OGMRIP_GLADE_ROOT, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return;
  }

  gtk_dialog_add_buttons (GTK_DIALOG (dialog),
      GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
      NULL);
  gtk_window_set_title (GTK_WINDOW (dialog), _("X264 Options"));
  gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_CLOSE);
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_PREFERENCES);

  widget = glade_xml_get_widget (xml, OGMRIP_GLADE_ROOT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), widget, TRUE, TRUE, 0);
  gtk_widget_show (widget);

  dialog->bframes_spin = glade_xml_get_widget (xml, "bframes-spin");
  g_signal_connect_swapped (dialog->bframes_spin, "value-changed",
      G_CALLBACK (ogmrip_x264_dialog_bframes_changed), dialog);

  dialog->cabac_check = glade_xml_get_widget (xml, "cabac-check");

  dialog->subq_spin = glade_xml_get_widget (xml, "subq-spin");
  g_signal_connect_swapped (dialog->subq_spin, "value-changed",
      G_CALLBACK (ogmrip_x264_dialog_subq_changed), dialog);
  
  dialog->global_header_check = glade_xml_get_widget (xml, "global_header-check");
  dialog->b_pyramid_check = glade_xml_get_widget (xml, "b_pyramid-check");
  dialog->weight_b_check = glade_xml_get_widget (xml, "weight_b-check");
  
  dialog->frameref_spin = glade_xml_get_widget (xml, "frameref-spin");
  g_signal_connect_swapped (dialog->frameref_spin, "value-changed",
      G_CALLBACK (ogmrip_x264_dialog_frameref_changed), dialog);
  
  dialog->me_combo = glade_xml_get_widget (xml, "me-combo");
  if (x264_have_me_tesa)
    gtk_combo_box_append_text (GTK_COMBO_BOX (dialog->me_combo),
        _("Transformed Exhaustive search (tesa - even slower)"));

  dialog->x88dct_check = glade_xml_get_widget (xml, "dct8x8-check");
  dialog->mixed_refs_check = glade_xml_get_widget (xml, "mixed_refs-check");
  
  dialog->brdo_check = glade_xml_get_widget (xml, "brdo-check");
  gtk_widget_set_sensitive (dialog->brdo_check, x264_have_brdo);

  dialog->vbv_maxrate_spin = glade_xml_get_widget (xml, "vbv_maxrate-spin");
  g_signal_connect_swapped (dialog->vbv_maxrate_spin, "value-changed",
      G_CALLBACK (ogmrip_x264_dialog_vbv_maxrate_changed), dialog);
  
  dialog->vbv_bufsize_spin = glade_xml_get_widget (xml, "vbv_bufsize-spin");
  dialog->level_idc_spin = glade_xml_get_widget (xml, "level_idc-spin");
  dialog->direct_combo = glade_xml_get_widget (xml, "direct-combo");
  dialog->b_adapt_spin = glade_xml_get_widget (xml, "b_adapt-spin");
  dialog->merange_spin = glade_xml_get_widget (xml, "merange-spin");

  dialog->psy_rd_spin = glade_xml_get_widget (xml, "psy_rd-spin");
  gtk_widget_set_sensitive (dialog->psy_rd_spin, x264_have_psy);

  dialog->psy_trellis_spin = glade_xml_get_widget (xml, "psy_trellis-spin");
  gtk_widget_set_sensitive (dialog->psy_trellis_spin, x264_have_psy);

  dialog->aud_check = glade_xml_get_widget (xml, "aud-check");
  gtk_widget_set_sensitive (dialog->aud_check, x264_have_aud);

  dialog->rc_lookahead_spin = glade_xml_get_widget (xml, "rc_lookahead-spin");
  gtk_widget_set_sensitive (dialog->rc_lookahead_spin, x264_have_lookahead);

  g_object_unref (xml);
}

static void
ogmrip_x264_get_me (GObject *object, const gchar *property, GValue *value, gpointer data)
{
  gint active;

  g_object_get (object, "active", &active, NULL);
  g_value_set_uint (value, active + 1);
}

static void
ogmrip_x264_set_me (GObject *object, const gchar *property, const GValue *value, gpointer data)
{
  gint active;

  if (G_VALUE_HOLDS (value, G_TYPE_UINT))
    active = g_value_get_uint (value);
  else
    active = g_value_get_int (value);

  g_object_set (object, "active", active - 1, NULL);
}

static void
ogmrip_x264_dialog_set_section (OGMRipPluginDialog *plugin_dialog, const gchar *section)
{
  OGMRipSettings *settings;

  settings = ogmrip_settings_get_default ();
  if (settings)
  {
    OGMRipX264Dialog *dialog;

    dialog = OGMRIP_X264_DIALOG (plugin_dialog);

    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_BFRAMES, G_OBJECT (dialog->bframes_spin), "value");
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_SUBQ, G_OBJECT (dialog->subq_spin), "value");
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_FRAMEREF, G_OBJECT (dialog->frameref_spin), "value");
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_MERANGE, G_OBJECT (dialog->merange_spin), "value");
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_VBV_MAXRATE, G_OBJECT (dialog->vbv_maxrate_spin), "value");
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_VBV_BUFSIZE, G_OBJECT (dialog->vbv_bufsize_spin), "value");
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_LEVEL_IDC, G_OBJECT (dialog->level_idc_spin), "value");
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_B_ADAPT, G_OBJECT (dialog->b_adapt_spin), "value");

    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_CABAC, G_OBJECT (dialog->cabac_check), "active");
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_B_PYRAMID, G_OBJECT (dialog->b_pyramid_check), "active");
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_WEIGHT_B, G_OBJECT (dialog->weight_b_check), "active");
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_8X8DCT, G_OBJECT (dialog->x88dct_check), "active");
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_MIXED_REFS, G_OBJECT (dialog->mixed_refs_check), "active");
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_GLOBAL_HEADER, G_OBJECT (dialog->global_header_check), "active");

    ogmrip_settings_bind_custom (settings, section,
        OGMRIP_X264_KEY_ME, G_OBJECT (dialog->me_combo), "active",
        ogmrip_x264_get_me, ogmrip_x264_set_me, NULL);
    ogmrip_settings_bind (settings, section,
        OGMRIP_X264_KEY_DIRECT, G_OBJECT (dialog->direct_combo), "active");

    if (x264_have_brdo)
      ogmrip_settings_bind (settings, section,
          OGMRIP_X264_KEY_BRDO, G_OBJECT (dialog->brdo_check), "active");

    if (x264_have_psy)
    {
      ogmrip_settings_bind (settings, section,
          OGMRIP_X264_KEY_PSY_RD, G_OBJECT (dialog->psy_rd_spin), "value");
      ogmrip_settings_bind (settings, section,
          OGMRIP_X264_KEY_PSY_TRELLIS, G_OBJECT (dialog->psy_trellis_spin), "value");
    }

    if (x264_have_aud)
      ogmrip_settings_bind (settings, section,
          OGMRIP_X264_KEY_AUD, G_OBJECT (dialog->aud_check), "active");

    if (x264_have_lookahead)
      ogmrip_settings_bind (settings, section,
          OGMRIP_X264_KEY_RC_LOOKAHEAD, G_OBJECT (dialog->rc_lookahead_spin), "value");
  }
}

static OGMRipVideoOptionsPlugin x264_options_plugin =
{
  NULL,
  G_TYPE_NONE,
  G_TYPE_NONE
};

OGMRipVideoOptionsPlugin *
ogmrip_init_options_plugin (void)
{
  GModule *module;

  x264_options_plugin.type = ogmrip_plugin_get_video_codec_by_name ("x264");
  if (x264_options_plugin.type == G_TYPE_NONE)
    return NULL;

  module = ogmrip_plugin_get_video_codec_module (x264_options_plugin.type);
  if (module)
  {
    gboolean *symbol;

    if (g_module_symbol (module, "x264_have_brdo", (gpointer *) &symbol))
      x264_have_brdo = *symbol;

    if (g_module_symbol (module, "x264_have_psy", (gpointer *) &symbol))
      x264_have_psy = *symbol;

    if (g_module_symbol (module, "x264_have_aud", (gpointer *) &symbol))
      x264_have_aud = *symbol;

    if (g_module_symbol (module, "x264_have_lookahead", (gpointer *) &symbol))
      x264_have_lookahead = *symbol;

    if (g_module_symbol (module, "x264_have_me_tesa", (gpointer *) &symbol))
      x264_have_me_tesa = *symbol;
  }

  x264_options_plugin.dialog = OGMRIP_TYPE_X264_DIALOG;

  return &x264_options_plugin;
}

