/*
 *  OpenDuke
 *  Copyright (C) 1999  Rusty Wagner
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */


//---------------------------------------------------------------------------
#pragma hdrstop

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "ConEngine.h"
#ifdef PLATFORM_UNIX
#include "linux_inc.h"
#else
//---------------------------------------------------------------------------
#pragma package(smart_init)
#endif

ConEngine::ConEngine()
{
}

ConEngine::~ConEngine()
{
}

// start engine with passed con file
void ConEngine::ParseFile(char *fname)
{
  _Parse(fname);
//  _Dump();
}

// add define to list
void ConEngine::AddDefine(char *name, int val)
{
  _defines[string(name)] = val;
}

// return define value from list
int ConEngine::GetDefine(char *name)
{
  if (_defines.count(name))
    return _defines[string(name)];
  else
    Error("define '%s' not defined.", name);

  return -1;
}

// add a quote to the list
void ConEngine::AddQuote(int num, char *val)
{
  _quotes[num] = string(val);
}

// return quote value from list
const char *ConEngine::GetQuote(int num)
{
  if (_quotes.count(num))
    return _quotes[num].c_str();
  else
    Error("quote %i not defined.", num);

  return "";
}

const char *ConEngine::GetLevelName (int episode, int map)
{
	// TODO: Range checking !!! - DDOI
	if (episode <= MAXEPS && map <- MAXLVLS)
		return _levels[episode][map].title;
	else
		return "";
}

// recursive parser for passed con file
void ConEngine::_Parse(char *fname)
{
  FILE *fp;
  char buf[256],
       buf2[256];
  int i, j;

  fp = fopen(fname, "r");
  if (!fp) {
    Error("cannot open con file (%s).", fname);
    return;
  }
  while (_GetTokF(fp, buf, true))
  {
    if (!stricmp(buf, "ACTION"))
    {
      _Action action;

      _GetTok(fp, buf2, true);
      action.startframe = 0;
      action.frames = 0;
      action.viewtype = 0;
      action.incvalue = 1;
      action.delay = 0;
      if (_IsVal(_PeekTok(fp, buf)))
        action.startframe = _ToVal(_GetTok(fp, buf, true));
      if (_IsVal(_PeekTok(fp, buf)))
        action.frames = _ToVal(_GetTok(fp, buf, true));
      if (_IsVal(_PeekTok(fp, buf)))
        action.viewtype = _ToVal(_GetTok(fp, buf, true));
      if (_IsVal(_PeekTok(fp, buf)))
        action.incvalue = _ToVal(_GetTok(fp, buf, true));
      if (_IsVal(_PeekTok(fp, buf)))
        action.delay = _ToVal(_GetTok(fp, buf, true));
      _actions[buf2] = _action.size();
      _action.push_back(action);
    }
    else if (!stricmp(buf, "ACTOR"))
    {
      _Actor actor;

      _GetTok(fp, buf2, true);
      actor.strength = 0;
      actor.speed = -1;
      actor.ai = 0;
      actor.code = _code.size();
      if (_IsVal(_PeekTok(fp, buf)))
        actor.strength = _ToVal(_GetTok(fp, buf, true));
      if (_actions.count(string(_PeekTok(fp, buf))))
        actor.action = _actions[string(_GetTok(fp, buf, true))];
      if (_velocities.count(string(_PeekTok(fp, buf))))
        actor.speed = _velocities[string(_GetTok(fp, buf, true))];
      else if (_IsVal(_PeekTok(fp, buf)))
        _GetTok(fp, buf, true);
      while (_IsVal(_PeekTok(fp, buf)))
        actor.ai += _ToVal(_GetTok(fp, buf, true));
      _actors[string(buf2)] = _actor.size();
      _actor.push_back(actor);
      _GenerateCode(fp, "", "enda", true);
    }
    else if (!stricmp(buf, "DEFINE"))
    {
      _GetTok(fp, buf, true);
      AddDefine(buf, _GetVal(fp));
    }
    else if (!stricmp(buf, "DEFINELEVELNAME"))
    {
      // TODO: range checking
      _Level level;

      i = _GetVal(fp);
      j = _GetVal(fp);
      strcpy(level.fname, _GetTok(fp, buf, true));
      strcpy(level.par, _GetTok(fp, buf, true));
      strcpy(level.par3dr, _GetTok(fp, buf, true));
      strcpy(level.title, _GetTok(fp, buf, false));
      _levels[i][j] = level;
    }
    else if (!stricmp(buf, "DEFINEQUOTE"))
    {
      i = _GetVal(fp);
      AddQuote(i, _GetTok(fp, buf, false));
    }
    else if (!stricmp(buf, "DEFINESOUND"))
    {
      _Sound sound;

      _GetTok(fp, buf, true);
      strcpy(sound.fname, _GetTok(fp, buf2, true));
      sound.pitch1 = _GetVal(fp);
      sound.pitch2 = _GetVal(fp);
      sound.priority = _GetVal(fp);
      sound.type = _GetVal(fp);
      sound.volume = _GetVal(fp);
      _sounds[string(buf)] = _sound.size();
      _sound.push_back(sound);
    }
    else if (!stricmp(buf, "GAMESTARTUP"))
    {
      for(i = 0; i < 26; i++)
        _GetVal(fp);
    }
    else if (!stricmp(buf, "INCLUDE"))
    {
      _Parse(strlwr(_GetTok(fp, buf, true)));
    }
    else if (!stricmp(buf, "MOVE"))
    {
      _Velocity vel;

      _GetTok(fp, buf2, true);
      vel.horz = 0;
      vel.vert = 0;
      if (_IsVal(_PeekTok(fp, buf)))
        vel.horz = _ToVal(_GetTok(fp, buf, true));
      if (_IsVal(_PeekTok(fp, buf)))
        vel.vert = _ToVal(_GetTok(fp, buf, true));
      _velocities[buf2] = _velocity.size();
      _velocity.push_back(vel);
    }
    else if (!stricmp(buf, "MUSIC"))
    {
      i = _GetVal(fp);
      if (i == 0)
      {
        strcpy(_musictitle, _GetTok(fp, buf, true));
        strcpy(_musicend, _GetTok(fp, buf, true));
      }
      else
        for(j = 0; j < MAXLVLS; j++)
          strcpy(_music[i - 1][j], _GetTok(fp, buf, true));
    }
    else if (!stricmp(buf, "STATE"))
    {
      _GetTok(fp, buf, true);
      _states[buf] = _code.size();
      _GenerateCode(fp, "", "ends", true);
    }
    else if (!stricmp(buf, ""))
    {
    }
    else
    {
      //Dump();
      Error("primitive '%s' not implemented.", buf);
    }
  }
  fclose(fp);
}

// get token, cannot fail (will cause error message)
char *ConEngine::_GetTok(FILE *fp, char *buf, bool word)
{
  if (!_GetTokF(fp, buf, word))
    Error("failed to get a token.");

  return buf;
}


// get token from file (returns true/false based on success)
// if word is true, then a single delimited word is read, else
// a string is read to the end of the line (or until a comment is hit)
bool ConEngine::_GetTokF(FILE *fp, char *buf, bool word)
{
  int c;

  *buf = 0;
  c = fgetc(fp);
  for(;;)
  {
    if (ferror(fp) || feof(fp))
      return false;
    while ((iscntrl(c)) || (isspace(c)))
      c = fgetc(fp);
    if (ferror(fp) || feof(fp))
      return false;
	  if (c == '/')
	  {
	    c = fgetc(fp);
	    if (c == '/')
	    {
	      c = fgetc(fp);
	      if ((!isspace(c)) && (!iscntrl(c)))
	        Error("(compatibility) a space must follow the // comment.");
	      while (!iscntrl(c))
	        c = fgetc(fp);
	    }
	    else if (c == '*')
	    {
	      c = fgetc(fp);
        for(;;)
        {
          while (c != '*')
            c = fgetc(fp);
          c = fgetc(fp);
          if (c == '/')
            break;
        }
        c = fgetc(fp);
	    }
      continue;
	  }
    break;
  }
  while ((!iscntrl(c)))
  {
    if ((word) && (isspace(c)))
      break;
    if ((strlen(buf)) && ((c == '{') || (c == '}')))
      break;
    strccat(buf, c);
    c = fgetc(fp);
    if (ferror(fp) || feof(fp))
      return false;
  }

  return true;
}

// get the next token without removing it from the stream
char *ConEngine::_PeekTok(FILE *fp, char *buf)
{
  int i;

  i = ftell(fp);
  _GetTok(fp, buf, true);
  fseek(fp, i, SEEK_SET);

  return buf;
}

// get the numeric value of the string
int ConEngine::_ToVal(char *str)
{
  if ((isdigit(str[0])) || (str[0] == '-'))
    return atoi(str);
  else
    return GetDefine(str);
}

// get a value, either a number or defined
int ConEngine::_GetVal(FILE *fp)
{
  char buf[256];

  _GetTok(fp, buf, true);
  return _ToVal(buf);
}

// check if name is a number
bool ConEngine::_IsNum(char *name)
{
  if ((isdigit(name[0])) || (name[0] == '-'))
    return true;

  return false;
}

// check if name is defined or a number
bool ConEngine::_IsVal(char *name)
{
  return (_IsNum(name)) || (_defines.count(name));
}

// if block is true, then the code will be parsed until the ending token
// is hit, else it will return after parsing a single primitive
//
// Code format for the virtual machine is fairly simplistic.
// All values are ints.  An instruction starts with a command token.
// For a non 'if' type instruction, the parameters following it
// are dependent on that command.  For an 'if', the first parameter
// is the 'address' to branch to if it evaluates to false.
void ConEngine::_GenerateCode(FILE *fp, char *stok, char *end, bool block)
{
  char buf[256];
  int i,
      vfunc,
      retaddr;

  strcpy(buf, stok);
  if (!strlen(buf))
    _GetTok(fp, buf, true);
  while (stricmp(buf, end))
  {
    printf("%s\n", buf);
    vfunc = _MakeVFunc(buf);
    if (vfunc == STATE)
    {
      _code.push_back(CALL);
      _GetTok(fp, buf, true);
      if (!_states.count(string(buf)))
        Error("state '%s' not previously defined.", buf);
      _code.push_back(_states[string(buf)]);
    }
    else if (vfunc != -1)
      _code.push_back(vfunc);
    if (vfunc == STATE) ;
    else if (!stricmp(buf, "BREAK"))
      _code.push_back(RETURN);
    else if ((vfunc >= IFACTION) && (vfunc <= IFWASWEAPON))
    {
      retaddr = _code.size();
      _code.push_back(-1);
      if (vfunc == IFACTION)
      {
        if (!_actions.count(string(_GetTok(fp, buf, true))))
          Error("action '%s' not defined.", buf);
        _code.push_back(_actions[string(buf)]);
      }
      else
        for(i = 0; i < _VFuncParams(vfunc); i++)
          _code.push_back(_GetVal(fp));
      if (!stricmp(_GetTok(fp, buf, true), "{"))
        _GenerateCode(fp, "", "}", true);
      else
        _GenerateCode(fp, buf, "", false);
      if (!stricmp(_PeekTok(fp, buf), "ELSE"))
      {
        _GetTok(fp, buf, true);
        _code[retaddr] = _code.size() + 2;
        _code.push_back(JUMP);
        retaddr = _code.size();
        _code.push_back(-1);
        if (!stricmp(_GetTok(fp, buf, true), "{"))
          _GenerateCode(fp, "", "}", true);
        else
          _GenerateCode(fp, buf, "", false);
        _code[retaddr] = _code.size();
      }
      else
        _code[retaddr] = _code.size();
    }
    else if (vfunc == ACTION)
    {
      _Action action;

      if (!_actions.count(string(_GetTok(fp, buf, true))))
      {
        if (!_IsVal(buf))
          Error("action '%s' not defined.", buf);

        action.startframe = 0;
        action.frames = 0;
        action.viewtype = 0;
        action.incvalue = 1;
        action.delay = GetDefine(buf);
        _actions[buf] = _action.size();
        _action.push_back(action);
      }
      else
        _code.push_back(_actions[string(buf)]);
    }
    else if (vfunc == MOVE)
    {
      if (!strcmp(_GetTok(fp, buf, true), "0"))
      {
        _code.push_back(-1);
        _code.push_back(0);
      }
      else
      {
        if (!_velocities.count(string(buf)))
          Error("move velocity '%s' not defined.", buf);
        _code.push_back(_velocities[string(buf)]);
        i = 0;
        while (_IsVal(_PeekTok(fp, buf)))
          i += _GetVal(fp);
        _code.push_back(i);
      }
    }
    else if (vfunc == PALFROM)
    {
      _code.push_back(_GetVal(fp));
      for(i = 0; i < 3; i++)
        if (_IsVal(_PeekTok(fp, buf)))
          _code.push_back(_GetVal(fp));
        else
          _code.push_back(0);
    }
    else if (vfunc >= 0)
      for(i = 0; i < _VFuncParams(vfunc); i++)
        _code.push_back(_GetVal(fp));
    else
      Error("code primitive '%s' not implemented.", buf);
    if (!block)
      break;
    _GetTok(fp, buf, true);
  }
  if ((!stricmp(buf, "ends")) || (!stricmp(buf, "enda")))
    _code.push_back(RETURN);
}

// get number of params for virtual function
int ConEngine::_VFuncParams(int tok)
{
  const int PARAMS[] = { 1, 2, 2, 1, 1,
                1, 2, 1, 1, 1, 1, 1,
                1, 2, 1, 1, 0, 0,
                1, 2, 5, 1, 1,
                1, 1, 1, 1, 1,
                0, 0, 0, 0,
                0, 1, 1, 0,
                1, 1, 1, 0,
                0, 0, 0, 0, 1,
                1, 0, 0, 0,
                0, 1, 1, 1, 1, 1,
                2, 0, 1, 1, 1,
                0, 1, 1, 0, 1,
                1, 0, 1, 2, 4, 4, 1, 0,
                0, 1, 0, 0, 0,
                0, 1, 2, 2, 1,
                1, 1, 1, 1, 1, 1,
                1, 0, 0, 0,
                1, 1, 0};

  return PARAMS[tok];
}

// get virtual function number for token
int ConEngine::_MakeVFunc(char *tok)
{
  if (!stricmp(tok, "ACTION"))
    return ACTION;
  else if (!stricmp(tok, "ADDAMMO"))
    return ADDAMMO;
  else if (!stricmp(tok, "addinventory"))
    return ADDINVENTORY;
  else if (!stricmp(tok, "ADDKILLS"))
    return ADDKILLS;
  else if (!stricmp(tok, "ADDPHEALTH"))
    return ADDPHEALTH;
  else if (!stricmp(tok, "ADDSTRENGTH"))
    return ADDSTRENGTH;
  else if (!stricmp(tok, "ADDWEAPON"))
    return ADDWEAPON;
  else if (!stricmp(tok, "AI"))
    return AI;
  else if (!stricmp(tok, "CACTOR"))
    return CACTOR;
  else if (!stricmp(tok, "CLIPDIST"))
    return CLIPDIST;
  else if (!stricmp(tok, "COUNT"))
    return COUNT;
  else if (!stricmp(tok, "CSTAT"))
    return CSTAT;
  else if (!stricmp(tok, "CSTATOR"))
    return CSTATOR;
  else if (!stricmp(tok, "DEBRIS"))
    return DEBRIS;
  else if (!stricmp(tok, "DEBUG"))
    return DEBUG;
  else if (!stricmp(tok, "ENDOFGAME"))
    return ENDOFGAME;
  else if (!stricmp(tok, "FALL"))
    return FALL;
  else if (!stricmp(tok, "GETLASTPAL"))
    return GETLASTPAL;
  else if (!stricmp(tok, "GLOBALSOUND"))
    return GLOBALSOUND;
  else if (!stricmp(tok, "GUTS"))
    return GUTS;
  else if (!stricmp(tok, "HITRADIUS"))
    return HITRADIUS;
  else if (!stricmp(tok, "IFACTION"))
    return IFACTION;
  else if (!stricmp(tok, "IFACTIONCOUNT"))
    return IFACTIONCOUNT;
  else if (!stricmp(tok, "IFACTOR"))
    return IFACTOR;
  else if (!stricmp(tok, "IFACTORNOTSTAYPUT"))
    return IFACTORNOTSTAYPUT;
  else if (!stricmp(tok, "IFAI"))
    return IFAI;
  else if (!stricmp(tok, "IFANGDIFFL"))
    return IFANGDIFFL;
  else if (!stricmp(tok, "IFANGDIFFG"))
    return IFANGDIFFG;
  else if (!stricmp(tok, "IFAWAYFROMWALL"))
    return IFAWAYFROMWALL;
  else if (!stricmp(tok, "IFBULLETNEAR"))
    return IFBULLETNEAR;
  else if (!stricmp(tok, "IFCANSEE"))
    return IFCANSEE;
  else if (!stricmp(tok, "IFCANSEETARGET"))
    return IFCANSEETARGET;
  else if (!stricmp(tok, "IFCANSHOOTTARGET"))
    return IFCANSHOOTTARGET;
  else if (!stricmp(tok, "IFCEILINGDISTL"))
    return IFCEILINGDISTL;
  else if (!stricmp(tok, "IFCOUNT"))
    return IFCOUNT;
  else if (!stricmp(tok, "IFDEAD"))
    return IFDEAD;
  else if (!stricmp(tok, "IFFLOORDISTL"))
    return IFFLOORDISTL;
  else if (!stricmp(tok, "IFGAPZL"))
    return IFGAPZL;
  else if (!stricmp(tok, "IFGOTWEAPONCE"))
    return IFGOTWEAPONCE;
  else if (!stricmp(tok, "IFHITSPACE"))
    return IFHITSPACE;
  else if (!stricmp(tok, "IFHITWEAPON"))
    return IFHITWEAPON;
  else if (!stricmp(tok, "IFINOUTERSPACE"))
    return IFINOUTERSPACE;
  else if (!stricmp(tok, "IFINSPACE"))
    return IFINSPACE;
  else if (!stricmp(tok, "IFINWATER"))
    return IFINWATER;
  else if (!stricmp(tok, "IFMOVE"))
    return IFMOVE;
  else if (!stricmp(tok, "IFMULTIPLAYER"))
    return IFMULTIPLAYER;
  else if (!stricmp(tok, "IFNOSOUNDS"))
    return IFNOSOUNDS;
  else if (!stricmp(tok, "IFNOTMOVING"))
    return IFNOTMOVING;
  else if (!stricmp(tok, "IFONWATER"))
    return IFONWATER;
  else if (!stricmp(tok, "IFOUTSIDE"))
    return IFOUTSIDE;
  else if (!stricmp(tok, "IFP"))
    return IFP;
  else if (!stricmp(tok, "IFPDISTG"))
    return IFPDISTG;
  else if (!stricmp(tok, "IFPDISTL"))
    return IFPDISTL;
  else if (!stricmp(tok, "IFPHEALTHL"))
    return IFPHEALTHL;
  else if (!stricmp(tok, "IFPHEALTHG"))
    return IFPHEALTHG;
  else if (!stricmp(tok, "IFPINVENTORY"))
    return IFPINVENTORY;
  else if (!stricmp(tok, "IFRESPAWN"))
    return IFRESPAWN;
  else if (!stricmp(tok, "IFRND"))
    return IFRND;
  else if (!stricmp(tok, "IFSPAWNEDBY"))
    return IFSPAWNEDBY;
  else if (!stricmp(tok, "IFSPRITEPAL"))
    return IFSPRITEPAL;
  else if (!stricmp(tok, "IFSQUISHED"))
    return IFSQUISHED;
  else if (!stricmp(tok, "IFSTRENGTH"))
    return IFSTRENGTH;
  else if (!stricmp(tok, "IFWASWEAPON"))
    return IFWASWEAPON;
  else if (!stricmp(tok, "KILLIT"))
    return KILLIT;
  else if (!stricmp(tok, "LOTSOFGLASS"))
    return LOTSOFGLASS;
  else if (!stricmp(tok, "MAIL"))
    return MAIL;
  else if (!stricmp(tok, "MIKESND"))
    return MIKESND;
  else if (!stricmp(tok, "MONEY"))
    return MONEY;
  else if (!stricmp(tok, "MOVE"))
    return MOVE;
  else if (!stricmp(tok, "OPERATE"))
    return OPERATE;
  else if (!stricmp(tok, "PALFROM"))
    return PALFROM;
  else if (!stricmp(tok, "PAPER"))
    return PAPER;
  else if (!stricmp(tok, "PKICK"))
    return PKICK;
  else if (!stricmp(tok, "PSTOMP"))
    return PSTOMP;
  else if (!stricmp(tok, "QUOTE"))
    return QUOTE;
  else if (!stricmp(tok, "RESETACTIONCOUNT"))
    return RESETACTIONCOUNT;
  else if (!stricmp(tok, "RESETCOUNT"))
    return RESETCOUNT;
  else if (!stricmp(tok, "RESETPLAYER"))
    return RESETPLAYER;
  else if (!stricmp(tok, "RESPAWNHITAG"))
    return RESPAWNHITAG;
  else if (!stricmp(tok, "SHOOT"))
    return SHOOT;
  else if (!stricmp(tok, "SIZEAT"))
    return SIZEAT;
  else if (!stricmp(tok, "SIZETO"))
    return SIZETO;
  else if (!stricmp(tok, "SLEEPTIME"))
    return SLEEPTIME;
  else if (!stricmp(tok, "SOUND"))
    return SOUND;
  else if (!stricmp(tok, "SOUNDONCE"))
    return SOUNDONCE;
  else if (!stricmp(tok, "SPAWN"))
    return SPAWN;
  else if (!stricmp(tok, "SPRITEPAL"))
    return SPRITEPAL;
  else if (!stricmp(tok, "STATE"))
    return STATE;
  else if (!stricmp(tok, "STOPSOUND"))
    return STOPSOUND;
  else if (!stricmp(tok, "STRENGTH"))
    return STRENGTH;
  else if (!stricmp(tok, "TIP"))
    return TIP;
  else if (!stricmp(tok, "TOSSWEAPON"))
    return TOSSWEAPON;
  else if (!stricmp(tok, "WACKPLAYER"))
    return WACKPLAYER;

  return -1;
}

char *ConEngine::_GetVFunc(int vfunc, char *buf)
{
  switch (vfunc)
  {
    case ACTION:
      strcpy(buf, "ACTION");
      break;
    case ADDAMMO:
      strcpy(buf, "ADDAMMO");
      break;
    case ADDINVENTORY:
      strcpy(buf, "ADDINVENTORY");
      break;
    case ADDKILLS:
      strcpy(buf, "ADDKILLS");
      break;
    case ADDPHEALTH:
      strcpy(buf, "ADDPHEALTH");
      break;
    case ADDSTRENGTH:
      strcpy(buf, "ADDSTRENGTH");
      break;
    case ADDWEAPON:
      strcpy(buf, "ADDWEAPON");
      break;
    case AI:
      strcpy(buf, "AI");
      break;
    case CACTOR:
      strcpy(buf, "CACTOR");
      break;
    case CLIPDIST:
      strcpy(buf, "CLIPDIST");
      break;
    case COUNT:
      strcpy(buf, "COUNT");
      break;
    case CSTAT:
      strcpy(buf, "CSTAT");
      break;
    case CSTATOR:
      strcpy(buf, "CSTATOR");
      break;
    case DEBRIS:
      strcpy(buf, "DEBRIS");
      break;
    case DEBUG:
      strcpy(buf, "DEBUG");
      break;
    case ENDOFGAME:
      strcpy(buf, "ENDOFGAME");
      break;
    case FALL:
      strcpy(buf, "FALL");
      break;
    case GETLASTPAL:
      strcpy(buf, "GETLASTPAL");
      break;
    case GLOBALSOUND:
      strcpy(buf, "GLOBALSOUND");
      break;
    case GUTS:
      strcpy(buf, "GUTS");
      break;
    case HITRADIUS:
      strcpy(buf, "HITRADIUS");
      break;
    case IFACTION:
      strcpy(buf, "IFACTION");
      break;
    case IFACTIONCOUNT:
      strcpy(buf, "IFACTIONCOUNT");
      break;
    case IFACTOR:
      strcpy(buf, "IFACTOR");
      break;
    case IFACTORNOTSTAYPUT:
      strcpy(buf, "IFACTORNOTSTAYPUT");
      break;
    case IFAI:
      strcpy(buf, "IFAI");
      break;
    case IFANGDIFFL:
      strcpy(buf, "IFANGDIFFL");
      break;
    case IFANGDIFFG:
      strcpy(buf, "IFANGDIFFG");
      break;
    case IFAWAYFROMWALL:
      strcpy(buf, "IFAWAYFROMWALL");
      break;
    case IFBULLETNEAR:
      strcpy(buf, "IFBULLETNEAR");
      break;
    case IFCANSEE:
      strcpy(buf, "IFCANSEE");
      break;
    case IFCANSEETARGET:
      strcpy(buf, "IFCANSEETARGET");
      break;
    case IFCANSHOOTTARGET:
      strcpy(buf, "IFCANSHOOTTARGET");
      break;
    case IFCEILINGDISTL:
      strcpy(buf, "IFCEILINGDISTL");
      break;
    case IFCOUNT:
      strcpy(buf, "IFCOUNT");
      break;
    case IFDEAD:
      strcpy(buf, "IFDEAD");
      break;
    case IFFLOORDISTL:
      strcpy(buf, "IFFLOORDISTL");
      break;
    case IFGAPZL:
      strcpy(buf, "IFGAPZL");
      break;
    case IFGOTWEAPONCE:
      strcpy(buf, "IFGOTWEAPONCE");
      break;
    case IFHITSPACE:
      strcpy(buf, "IFHITSPACE");
      break;
    case IFHITWEAPON:
      strcpy(buf, "IFHITWEAPON");
      break;
    case IFINOUTERSPACE:
      strcpy(buf, "IFINOUTERSPACE");
      break;
    case IFINSPACE:
      strcpy(buf, "IFINSPACE");
      break;
    case IFINWATER:
      strcpy(buf, "IFINWATER");
      break;
    case IFMOVE:
      strcpy(buf, "IFMOVE");
      break;
    case IFMULTIPLAYER:
      strcpy(buf, "IFMULTIPLAYER");
      break;
    case IFNOSOUNDS:
      strcpy(buf, "IFNOSOUNDS");
      break;
    case IFNOTMOVING:
      strcpy(buf, "IFNOTMOVING");
      break;
    case IFONWATER:
      strcpy(buf, "IFONWATER");
      break;
    case IFOUTSIDE:
      strcpy(buf, "IFOUTSIDE");
      break;
    case IFP:
      strcpy(buf, "IFP");
      break;
    case IFPDISTG:
      strcpy(buf, "IFPDISTG");
      break;
    case IFPDISTL:
      strcpy(buf, "IFPDISTL");
      break;
    case IFPHEALTHL:
      strcpy(buf, "IFPHEALTHL");
      break;
    case IFPHEALTHG:
      strcpy(buf, "IFPHEALTHG");
      break;
    case IFPINVENTORY:
      strcpy(buf, "IFPINVENTORY");
      break;
    case IFRESPAWN:
      strcpy(buf, "IFRESPAWN");
      break;
    case IFRND:
      strcpy(buf, "IFRND");
      break;
    case IFSPAWNEDBY:
      strcpy(buf, "IFSPAWNEDBY");
      break;
    case IFSPRITEPAL:
      strcpy(buf, "IFSPRITEPAL");
      break;
    case IFSQUISHED:
      strcpy(buf, "IFSQUISHED");
      break;
    case IFSTRENGTH:
      strcpy(buf, "IFSTRENGTH");
      break;
    case IFWASWEAPON:
      strcpy(buf, "IFWASWEAPON");
      break;
    case KILLIT:
      strcpy(buf, "KILLIT");
      break;
    case LOTSOFGLASS:
      strcpy(buf, "LOTSOFGLASS");
      break;
    case MAIL:
      strcpy(buf, "MAIL");
      break;
    case MIKESND:
      strcpy(buf, "MIKESND");
      break;
    case MONEY:
      strcpy(buf, "MONEY");
      break;
    case MOVE:
      strcpy(buf, "MOVE");
      break;
    case OPERATE:
      strcpy(buf, "OPERATE");
      break;
    case PALFROM:
      strcpy(buf, "PALFROM");
      break;
    case PAPER:
      strcpy(buf, "PAPER");
      break;
    case PKICK:
      strcpy(buf, "PKICK");
      break;
    case PSTOMP:
      strcpy(buf, "PSTOMP");
      break;
    case QUOTE:
      strcpy(buf, "QUOTE");
      break;
    case RESETACTIONCOUNT:
      strcpy(buf, "RESETACTIONCOUNT");
      break;
    case RESETCOUNT:
      strcpy(buf, "RESETCOUNT");
      break;
    case RESETPLAYER:
      strcpy(buf, "RESETPLAYER");
      break;
    case RESPAWNHITAG:
      strcpy(buf, "RESPAWNHITAG");
      break;
    case SHOOT:
      strcpy(buf, "SHOOT");
      break;
    case SIZEAT:
      strcpy(buf, "SIZEAT");
      break;
    case SIZETO:
      strcpy(buf, "SIZETO");
      break;
    case SLEEPTIME:
      strcpy(buf, "SLEEPTIME");
      break;
    case SOUND:
      strcpy(buf, "SOUND");
      break;
    case SOUNDONCE:
      strcpy(buf, "SOUNDONCE");
      break;
    case SPAWN:
      strcpy(buf, "SPAWN");
      break;
    case SPRITEPAL:
      strcpy(buf, "SPRITEPAL");
      break;
    case STATE:
      strcpy(buf, "STATE");
      break;
    case STOPSOUND:
      strcpy(buf, "STOPSOUND");
      break;
    case STRENGTH:
      strcpy(buf, "STRENGTH");
      break;
    case TIP:
      strcpy(buf, "TIP");
      break;
    case TOSSWEAPON:
      strcpy(buf, "TOSSWEAPON");
      break;
    case WACKPLAYER:
      strcpy(buf, "WACKPLAYER");
      break;
    case CALL:
      strcpy(buf, "(CALL)");
      break;
    case JUMP:
      strcpy(buf, "(JUMP)");
      break;
    case RETURN:
      strcpy(buf, "(RETURN)");
      break;
    default:
      sprintf(buf, "(%i)", vfunc);
  }

  return buf;
}

void ConEngine::_Dump()
{
  map<string, int>::iterator str_int;
  int i, j;
  char buf[256];

  printf("Defines:\n");
  str_int = _defines.begin();
  while (str_int != _defines.end())
  {
    printf("%25s %i\n", str_int->first.c_str(), str_int->second);
    str_int++;
  }
  printf("\n");

  printf("Virtual Code:\n");
  for(i = 0; i < (int)_code.size(); i++)
  {
    printf("%05i: %5i  %s", i, _code[i], _GetVFunc(_code[i], buf));
    if (_VFuncParams(_code[i]) > 0)
    {
      j = i;
      if ((_code[i] >= IFACTION) && (_code[i] <= IFWASWEAPON))
      {
        i++;
        printf("(%i)", _code[i]);
      }
      do
      {
        i++;
        printf(" %i", _code[i]);
      } while ((i - j) < _VFuncParams(_code[j]));
    }
    printf("\n");
  }
  printf("\n");
}

static void Error(char *str, ...)
{
  va_list argptr;
  char buf[256];

  if (str)
  {
    sprintf(buf, "ConEngine Error: %s\n", str);
    va_start(argptr, str);
    vprintf(buf, argptr);
    va_end(argptr);
  }
  // Exiting here is bad - DDOI
  //exit(1);
}

static void strccat(char *str, int c)
{
  char buf[2];

  buf[0] = (char)c;
  buf[1] = 0;
  strcat(str, buf);
}

static void strtolower(char *str)
{
  size_t i;

  for(i = 0; i < strlen(str); i++)
    str[i] = (char)tolower(str[i]);
}

