/*
 *  OpenDuke
 *  Copyright (C) 1999  Rusty Wagner
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */


#pragma hdrstop
#include "Direct3D.h"
#include "View.h"
#include "Texture.h"

D3DMATRIX projMatrix={2, 0, 0, 0,
					  0, 2, 0, 0,
					  0, 0, 1, 1,
					  0, 0,-1, 0};

D3DMATRIX worldMatrix={1, 0, 0, 0,
					   0, 1, 0, 0,
					   0, 0, 1, 0,
					   0, 0, 0, 1};

D3DMATRIX initViewMatrix={1, 0, 0, 0,
						  0, 1, 0, 0,
						  0, 0, 1, 0,
						  0, 0, 0, 1};

extern char errorStr[256];
extern int error;
int software=0;

static unsigned short oldFcw;

#define DO_ERROR(str) \
{ \
	if (errorStr) strcpy(errorStr,str); \
	error=1; \
	return; \
}

static CDirectDraw *dd;
static HRESULT WINAPI EnumZBufferFormatsCallback(DDPIXELFORMAT *fmt,
    void* ddpf)
{
    if (fmt==NULL) return D3DENUMRET_CANCEL;
    if ((fmt->dwFlags&(DDPF_ZBUFFER|DDPF_STENCILBUFFER))&&
        (fmt->dwStencilBitDepth>=8)&&(fmt->dwZBufferBitDepth==
        dd->GetScrnDepth()))
        memcpy(ddpf,fmt,sizeof(DDPIXELFORMAT));
    else if ((!(((DDPIXELFORMAT*)ddpf)->dwFlags&DDPF_STENCILBUFFER))&&
        (fmt->dwZBufferBitDepth==dd->GetScrnDepth()))
        memcpy(ddpf,fmt,sizeof(DDPIXELFORMAT));
    return D3DENUMRET_OK;
}

CDirect3D::CDirect3D(CDirectDraw *_dd,IDirectDrawSurface7* target)
{
	DDSURFACEDESC2 ddsd;
	D3DMATERIAL material;
	IDirect3DMaterial3 *lpMaterial;

	DDraw=_dd;
	view=NULL;
	lpRenderTarget=target;

	if (!lpRenderTarget)
		DO_ERROR("No render target found");

	// Find the width and height of the render target
	ddsd.dwSize=sizeof(ddsd);
	lpRenderTarget->GetSurfaceDesc(&ddsd);
	scrnX=ddsd.dwWidth;
	scrnY=ddsd.dwHeight;

    projMatrix._11=1;
    projMatrix._22=(float)scrnX/(float)scrnY;

	// Create Direct3D object
	if (DDraw->GetDD()->QueryInterface(IID_IDirect3D7,(void **)&lpD3D)!=DD_OK)
		DO_ERROR("Could not initialize Direct3D");

	// Create a z-buffer
	ddsd.dwSize=sizeof(ddsd);
	ddsd.dwFlags=DDSD_WIDTH|DDSD_HEIGHT|DDSD_CAPS|DDSD_PIXELFORMAT;
	ddsd.ddsCaps.dwCaps=DDSCAPS_ZBUFFER|(software?
    	DDSCAPS_SYSTEMMEMORY:DDSCAPS_VIDEOMEMORY);
	ddsd.dwWidth=scrnX;
	ddsd.dwHeight=scrnY;
    ddsd.ddpfPixelFormat.dwSize=sizeof(DDPIXELFORMAT);
    dd=_dd;
    ddsd.ddpfPixelFormat.dwFlags=0;
    lpD3D->EnumZBufferFormats(IID_IDirect3DHALDevice,
        EnumZBufferFormatsCallback,&ddsd.ddpfPixelFormat);
    stencil=ddsd.ddpfPixelFormat.dwFlags&DDPF_STENCILBUFFER;
	if (DDraw->GetDD()->CreateSurface(&ddsd,&lpZBuffer,NULL)!=DD_OK)
		DO_ERROR("Could not create z-buffer");

	// Attach the z-buffer
	if (lpRenderTarget->AddAttachedSurface(lpZBuffer)!=DD_OK)
		DO_ERROR("Could not attach z-buffer");

	// Create hardware device object
	if (lpD3D->CreateDevice(software?IID_IDirect3DRGBDevice:
    	IID_IDirect3DTnLHalDevice,lpRenderTarget,
		&lpD3DDevice)!=DD_OK)
    {
    	if (lpD3D->CreateDevice(software?IID_IDirect3DRGBDevice:
        	IID_IDirect3DHALDevice,lpRenderTarget,
		    &lpD3DDevice)!=DD_OK)
            DO_ERROR("No Direct3D hardware accelerators found");
    }

    // Disable floating point exceptions
    // (VC++ probably does this already, as a divide
    //  by zero exception occurs in Microsoft's D3D code
    //  only on Borland C++)
    unsigned short fcw;
    asm fstcw [fcw]
    asm mov ax,[fcw]
    asm mov [oldFcw],ax
    asm or word ptr [fcw],03fh
    asm fldcw [fcw]

	// Set initial transform matrices
	if (lpD3DDevice->SetTransform(D3DTRANSFORMSTATE_VIEW,
 		&initViewMatrix)!=DD_OK)
		DO_ERROR("Could not set view matrix");
	if (lpD3DDevice->SetTransform(D3DTRANSFORMSTATE_WORLD,
		&worldMatrix)!=DD_OK)
		DO_ERROR("Could not set world matrix");
	if (lpD3DDevice->SetTransform(D3DTRANSFORMSTATE_PROJECTION,
		&projMatrix)!=DD_OK)
		DO_ERROR("Could not set projection matrix");

    // Restore floating point exceptions and ignore
    // any that occured in Microsoft's D3D code
    asm fclex
    asm fldcw [oldFcw]

    D3DMATERIAL7 mat;
    mat.diffuse.r=0; mat.diffuse.g=0; mat.diffuse.b=0; mat.diffuse.a=0;
    mat.ambient.r=1; mat.ambient.g=1; mat.ambient.b=1; mat.ambient.a=1;
    mat.specular.r=0; mat.specular.g=0; mat.specular.b=0; mat.specular.a=0;
    mat.emissive.r=0; mat.emissive.g=0; mat.emissive.b=0; mat.emissive.a=0;
    mat.power=0;
    lpD3DDevice->SetMaterial(&mat);

    // Obtain the capabilities of the device
    memset(&devCaps,0,sizeof(devCaps));
    lpD3DDevice->GetCaps(&devCaps);

    maxVert=0xffff;
    vertSectionSize=maxVert-(maxVert%6);
}

CDirect3D::~CDirect3D()
{
    if (lpZBuffer) { lpZBuffer->Release(); lpZBuffer=NULL; }
	if (lpD3DDevice) { lpD3DDevice->Release(); lpD3DDevice=NULL; }
	if (lpD3D) { lpD3D->Release(); lpD3D=NULL; }
}

void CDirect3D::SetWorldMatrix(D3DMATRIX *mat)
{
	if (mat==NULL)
   	mat=&worldMatrix;
	if (lpD3DDevice->SetTransform(D3DTRANSFORMSTATE_WORLD,mat)!=DD_OK)
		DO_ERROR("Could not set world matrix");
}

void CDirect3D::SetView(CView *_view)
{
	if (view)
	{
		view->Activate(0);
		view=NULL;
	}
	view=_view;
	if (!view)
		return;
	lpD3DDevice->SetViewport(view->GetViewport());
	if (lpD3DDevice->SetTransform(D3DTRANSFORMSTATE_VIEW,&view->
		GetMatrix())!=DD_OK)
		throw "Could not set view matrix";
	view->Activate(1);
}

void CDirect3D::BeginScene()
{
	lpD3DDevice->BeginScene();
}

void CDirect3D::EndScene()
{
	lpD3DDevice->EndScene();
}

void CDirect3D::DrawPrimitive(D3DPRIMITIVETYPE pt,Vertex* vert,
	int count,DWORD flags)
{
    if (count>=maxVert)
    {
        for (int i=0;i<count;i+=vertSectionSize,count-=vertSectionSize)
        {
        	lpD3DDevice->DrawPrimitive(pt,D3DFVF_DIFFUSE|
                D3DFVF_XYZ|D3DFVF_TEX2,
                &(vert[i]),(count<vertSectionSize)?
                count:vertSectionSize,flags);
        }
    }
	lpD3DDevice->DrawPrimitive(pt,D3DFVF_DIFFUSE|
        D3DFVF_XYZ|D3DFVF_TEX2,vert,count,flags);
}

void CDirect3D::SetRenderState(D3DRENDERSTATETYPE state,DWORD val)
{
	lpD3DDevice->SetRenderState(state,val);
}

void CDirect3D::SetRenderStateFloat(D3DRENDERSTATETYPE state,float val)
{
	lpD3DDevice->SetRenderState(state,*(DWORD *)(&val));
}

void CDirect3D::BeginRender()
{
	// Check to see if we still have our surfaces
	// If not, restore them
	if (lpRenderTarget->IsLost() == DDERR_SURFACELOST)
	{
		lpRenderTarget->Restore();
		lpZBuffer->Restore();
	}

    lpD3DDevice->Clear(0,NULL,D3DCLEAR_TARGET|(stencil?D3DCLEAR_STENCIL:0)|
        D3DCLEAR_ZBUFFER,RGBA_MAKE(0,0,0,0),1,0);

    // Disable floating point exceptions
    // (VC++ probably does this already, as a divide
    //  by zero exception occurs in Microsoft's D3D code
    //  only on Borland C++)
    unsigned short fcw;
    asm fstcw [fcw]
    asm mov ax,[fcw]
    asm mov [oldFcw],ax
    asm or word ptr [fcw],03fh
    asm fldcw [fcw]
}

void CDirect3D::ClearZBuffer()
{
    lpD3DDevice->Clear(0,NULL,D3DCLEAR_ZBUFFER,RGBA_MAKE(128,128,128,0),1,0);
}

void CDirect3D::ClearStencilBuffer()
{
    if (stencil)
        lpD3DDevice->Clear(0,NULL,D3DCLEAR_STENCIL,RGBA_MAKE(128,128,128,0),1,0);
}

void CDirect3D::EndRender()
{
    // Restore floating point exceptions and ignore
    // any that occured in Microsoft's D3D code
    asm fclex
    asm fldcw [oldFcw]

	DDraw->Update();
}

void CDirect3D::SetTexture(int stage,CTexture *tex)
{
	if (tex) lpD3DDevice->SetTexture(stage,tex->GetTexture());
}

void CDirect3D::SetTextureStageState(int stage,
    D3DTEXTURESTAGESTATETYPE state,DWORD value)
{
    lpD3DDevice->SetTextureStageState(stage,state,value);
}

