/*****************************************************************************
 * Copyright (c) 2014-2024 OpenRCT2 developers
 *
 * For a complete list of all authors, please refer to contributors.md
 * Interested in contributing? Visit https://github.com/OpenRCT2/OpenRCT2
 *
 * OpenRCT2 is licensed under the GNU General Public License version 3.
 *****************************************************************************/

#include "openrct2/localisation/LanguagePack.h"

#include "openrct2/localisation/Language.h"
#include "openrct2/localisation/StringIds.h"

#include <gtest/gtest.h>

using namespace OpenRCT2;

class LanguagePackTest : public testing::Test
{
protected:
    static const utf8* LanguageEnGB;
    static const unsigned char LanguageZhTW[];
};

TEST_F(LanguagePackTest, create_empty)
{
    auto empty = LanguagePackFactory::FromText(0, "");
    ASSERT_EQ(empty->GetId(), 0);
    ASSERT_EQ(empty->GetCount(), 0u);
}

TEST_F(LanguagePackTest, create_mutable_id_1)
{
    auto lang = LanguagePackFactory::FromText(1, "STR_0000:\n");
    ASSERT_EQ(lang->GetId(), 1);
    ASSERT_EQ(lang->GetCount(), 1u);
    ASSERT_STREQ(lang->GetString(0), nullptr);
    lang->SetString(0, "xx");
    ASSERT_EQ(lang->GetCount(), 1u);
    ASSERT_STREQ(lang->GetString(0), "xx");
}

TEST_F(LanguagePackTest, language_pack_simple)
{
    auto lang = LanguagePackFactory::FromText(0, LanguageEnGB);
    ASSERT_EQ(lang->GetId(), 0);
    ASSERT_EQ(lang->GetCount(), 4u);
    ASSERT_STREQ(lang->GetString(2), "Spiral Roller Coaster");
    ASSERT_EQ(lang->GetScenarioOverrideStringId("Arid Heights", 0), 0x7000);
    ASSERT_STREQ(lang->GetString(0x7000), "Arid Heights scenario string");
    // Test some negatives too
    ASSERT_EQ(lang->GetString(1000), nullptr);
    ASSERT_EQ(lang->GetScenarioOverrideStringId("No such park", 0), STR_NONE);
}

TEST_F(LanguagePackTest, language_pack_multibyte)
{
    auto lang = LanguagePackFactory::FromText(0, reinterpret_cast<const utf8*>(LanguageZhTW));
    ASSERT_EQ(lang->GetId(), 0);
    ASSERT_EQ(lang->GetCount(), 4u);
    ASSERT_STREQ(lang->GetString(2), u8"懸吊式雲霄飛車");
    ASSERT_EQ(lang->GetScenarioOverrideStringId("Forest Frontiers", 0), 0x7000);
    ASSERT_EQ(lang->GetScenarioOverrideStringId("Forest Frontiers", 2), 0x7002);
    ASSERT_STREQ(lang->GetString(0x7000), "Forest Frontiers");
    ASSERT_STREQ(lang->GetString(0x7002), u8"在隱藏於森林深處的清空範圍中, 建造一個很受歡迎的樂園");
}

const utf8* LanguagePackTest::LanguageEnGB = "# STR_XXXX part is read and XXXX becomes the string id number.\n"
                                             "# Everything after the colon and before the new line will be saved as the "
                                             "string.\n"
                                             "# Use # at the beginning of a line to leave a comment.\n"
                                             "STR_0000    :\n"
                                             "STR_0001    :{STRINGID} {COMMA16}\n"
                                             "STR_0002    :Spiral Roller Coaster\n"
                                             "STR_0003    :Stand-up Roller Coaster\n"
                                             "<Arid Heights>\n"
                                             "STR_SCNR    :Arid Heights scenario string\n"
                                             "STR_PARK    :Arid Heights park string\n"
                                             "STR_DTLS    :Free of any financial limits, your challenge is to develop "
                                             "this desert park while keeping the guests happy\n"
                                             "[CONDORRD]\n"
                                             "STR_NAME    :my test ride\n"
                                             "STR_DESC    :ride description\n"
                                             "STR_CPTY    :ride capacity\n";

// This includes a few entries extracted from zh-TW localisation.
// It has to be declared as `unsigned char`, or else the values overflow signed byte.
const unsigned char LanguagePackTest::LanguageZhTW[] = {
    0x53, 0x54, 0x52, 0x5f, 0x30, 0x30, 0x30, 0x30, 0x20, 0x20, 0x20, 0x20, 0x3a, 0xe8, 0x9e, 0xba, 0xe6, 0x97, 0x8b, 0xe5,
    0xbc, 0x8f, 0xe9, 0x9b, 0xb2, 0xe9, 0x9c, 0x84, 0xe9, 0xa3, 0x9b, 0xe8, 0xbb, 0x8a, 0x0a, 0x53, 0x54, 0x52, 0x5f, 0x30,
    0x30, 0x30, 0x31, 0x20, 0x20, 0x20, 0x20, 0x3a, 0xe7, 0xab, 0x99, 0xe7, 0xab, 0x8b, 0xe5, 0xbc, 0x8f, 0xe9, 0x9b, 0xb2,
    0xe9, 0x9c, 0x84, 0xe9, 0xa3, 0x9b, 0xe8, 0xbb, 0x8a, 0x0a, 0x53, 0x54, 0x52, 0x5f, 0x30, 0x30, 0x30, 0x32, 0x20, 0x20,
    0x20, 0x20, 0x3a, 0xe6, 0x87, 0xb8, 0xe5, 0x90, 0x8a, 0xe5, 0xbc, 0x8f, 0xe9, 0x9b, 0xb2, 0xe9, 0x9c, 0x84, 0xe9, 0xa3,
    0x9b, 0xe8, 0xbb, 0x8a, 0x0a, 0x53, 0x54, 0x52, 0x5f, 0x30, 0x30, 0x30, 0x33, 0x20, 0x20, 0x20, 0x20, 0x3a, 0xe5, 0x8f,
    0x8d, 0xe8, 0xbd, 0x89, 0xe5, 0xbc, 0x8f, 0xe9, 0x9b, 0xb2, 0xe9, 0x9c, 0x84, 0xe9, 0xa3, 0x9b, 0xe8, 0xbb, 0x8a, 0x0a,
    0x3c, 0x46, 0x6f, 0x72, 0x65, 0x73, 0x74, 0x20, 0x46, 0x72, 0x6f, 0x6e, 0x74, 0x69, 0x65, 0x72, 0x73, 0x3e, 0x0a, 0x53,
    0x54, 0x52, 0x5f, 0x53, 0x43, 0x4e, 0x52, 0x20, 0x20, 0x20, 0x20, 0x3a, 0x46, 0x6f, 0x72, 0x65, 0x73, 0x74, 0x20, 0x46,
    0x72, 0x6f, 0x6e, 0x74, 0x69, 0x65, 0x72, 0x73, 0x0a, 0x53, 0x54, 0x52, 0x5f, 0x50, 0x41, 0x52, 0x4b, 0x20, 0x20, 0x20,
    0x20, 0x3a, 0x46, 0x6f, 0x72, 0x65, 0x73, 0x74, 0x20, 0x46, 0x72, 0x6f, 0x6e, 0x74, 0x69, 0x65, 0x72, 0x73, 0x0a, 0x53,
    0x54, 0x52, 0x5f, 0x44, 0x54, 0x4c, 0x53, 0x20, 0x20, 0x20, 0x20, 0x3a, 0xe5, 0x9c, 0xa8, 0xe9, 0x9a, 0xb1, 0xe8, 0x97,
    0x8f, 0xe6, 0x96, 0xbc, 0xe6, 0xa3, 0xae, 0xe6, 0x9e, 0x97, 0xe6, 0xb7, 0xb1, 0xe8, 0x99, 0x95, 0xe7, 0x9a, 0x84, 0xe6,
    0xb8, 0x85, 0xe7, 0xa9, 0xba, 0xe7, 0xaf, 0x84, 0xe5, 0x9c, 0x8d, 0xe4, 0xb8, 0xad, 0x2c, 0x20, 0xe5, 0xbb, 0xba, 0xe9,
    0x80, 0xa0, 0xe4, 0xb8, 0x80, 0xe5, 0x80, 0x8b, 0xe5, 0xbe, 0x88, 0xe5, 0x8f, 0x97, 0xe6, 0xad, 0xa1, 0xe8, 0xbf, 0x8e,
    0xe7, 0x9a, 0x84, 0xe6, 0xa8, 0x82, 0xe5, 0x9c, 0x92, 0x0a, 0x5b, 0x43, 0x4f, 0x4e, 0x44, 0x4f, 0x52, 0x52, 0x44, 0x5d,
    0x0a, 0x53, 0x54, 0x52, 0x5f, 0x4e, 0x41, 0x4d, 0x45, 0x20, 0x20, 0x20, 0x20, 0x3a, 0xe7, 0xa5, 0x9e, 0xe9, 0xb7, 0xb9,
    0xe6, 0x9a, 0xa2, 0xe9, 0x81, 0x8a, 0x0a, 0x53, 0x54, 0x52, 0x5f, 0x44, 0x45, 0x53, 0x43, 0x20, 0x20, 0x20, 0x20, 0x3a,
    0xe4, 0xb9, 0x98, 0xe5, 0xae, 0xa2, 0xe4, 0xb9, 0x98, 0xe5, 0x9d, 0x90, 0xe6, 0x96, 0xbc, 0xe8, 0xbb, 0x8c, 0xe9, 0x81,
    0x93, 0xe4, 0xb8, 0x8b, 0xe7, 0x9a, 0x84, 0xe7, 0xa5, 0x9e, 0xe9, 0xb7, 0xb9, 0xe9, 0x80, 0xa0, 0xe5, 0x9e, 0x8b, 0xe5,
    0x88, 0x97, 0xe8, 0xbb, 0x8a, 0xe4, 0xb8, 0x8a, 0x2c, 0x20, 0xe5, 0xb0, 0x87, 0xe6, 0x9c, 0x83, 0xe6, 0x96, 0xbc, 0xe9,
    0xa3, 0x9b, 0xe9, 0xa6, 0xb3, 0xe4, 0xb8, 0xad, 0xe9, 0xab, 0x94, 0xe9, 0xa9, 0x97, 0xe9, 0xa3, 0x9b, 0xe4, 0xb8, 0x80,
    0xe8, 0x88, 0xac, 0xe7, 0x9a, 0x84, 0xe5, 0xbf, 0xab, 0xe6, 0x84, 0x9f, 0x0a, 0x53, 0x54, 0x52, 0x5f, 0x43, 0x50, 0x54,
    0x59, 0x20, 0x20, 0x20, 0x20, 0x3a, 0xe6, 0xaf, 0x8f, 0xe8, 0xbb, 0x8a, 0xe5, 0x8d, 0xa1, 0x34, 0xe4, 0xbd, 0x8d, 0xe4,
    0xb9, 0x98, 0xe5, 0xae, 0xa2, 0x0a, 0x00,
};
