/*
**	mmu_support.c
**
**	This file is subject to the terms and conditions of the GNU General Public
**	License.  See the file COPYING in the main directory of this archive
**	for more details.
**
**
*/

#include <setjmp.h>

#include "asm_page.h"
#include "asm_setup.h"
#include "penguin_prototypes.h"
#include "bootstrap_prototypes.h"
#include "MMU.h"
#include "MMU_V2.h"

/* Prototypes */
static unsigned long	testLogicalAddr (unsigned long logAddr);
static void				splitFirstRAMentry (void);

/*
 *	testLogicalAddr
 *
 */
static unsigned long testLogicalAddr (unsigned long logAddr)
{
	int					i;
	unsigned long		a, l , p;
	
	for (i = 0; i < gMemoryMappings.index; i++)
	{
		a = gMemoryMappings.log2phys[i].logical;
		l = gMemoryMappings.log2phys[i].length;
		p = gMemoryMappings.log2phys[i].physical;
		
		/* adjust length so that 
		 *		0x50000000 + 0xB000000
		 * does NOT become 0x0, but instead becomes 0xFFFFFFFF.
		 * (which does not mean NOT_MAPPED but instead
		 * denotes the last valid byte for the address range!)
		 */
		l -= 1;	
		
		if (p != NOT_MAPPED)
		{
			if ((logAddr >= a) && (logAddr <= (a + l)))
			{
				return 1;
			}
		}
	}
	
	return 0;
}

/*
 *	findLogicalAddrSize
 *
 */
unsigned long findLogicalAddrSize (void)
{
	unsigned long maxLogAddr, searchDelta, searchRight;
	
	searchDelta = PAGE_SIZE;
	while (testLogicalAddr (searchDelta))
	{
		searchDelta *= 2;
	}
	
	maxLogAddr = searchDelta;
	searchRight = false;
	while (searchDelta > PAGE_SIZE)
	{
		searchDelta /= 2;
		
		if (searchRight)
			maxLogAddr += searchDelta;
		else
			maxLogAddr -= searchDelta;
		
		if (testLogicalAddr (maxLogAddr) == 0)
		{
			if (searchRight)
				searchRight = 0;
		}
		else
		{
			if (! searchRight)
				searchRight = 1;
		}
		
	}
	
	return maxLogAddr + PAGE_SIZE;
}

/*
 *	printLog2PhysTable
 *
 */
void printLog2PhysTable (void)
{
	int					i;
	unsigned long		a, l , p;

	cprintf ("Logical To Physical Mapping table %s\n", "(V2)");
	cprintf ("Logical -> physical : length\n");
	
	for (i = 0; i < gMemoryMappings.index; i++)
	{
		a = gMemoryMappings.log2phys[i].logical;
		l = gMemoryMappings.log2phys[i].length;
		p = gMemoryMappings.log2phys[i].physical;
		
		if (p != NOT_MAPPED)
		{
			if ((p == 0 && l > 0x00008000) || (p != 0))
			{
				cprintf ("0x%0.8X -> 0x%0.8X : 0x%0.8X\n", a, p, l);
			}
		}
	}
}

/*
 *	LogicalToPhysical
 *
 */
unsigned long LogicalToPhysical (unsigned long logAddr)
{
	int					i;
	unsigned long		a, l , p;

	a = LogicalToPhysical_V2(logAddr, &p);
	if (a)
		return p;
	
	UnsupportedNum ("Logical address (0x%lX) has no physical mapping.", logAddr);

	return 0;
}

/*
 *	PhysicalToLogical
 *
 */
unsigned long PhysicalToLogical (unsigned long physAddr)
{
	int					i;
	unsigned long		a, l , p;

	for (i = 0; i < gMemoryMappings.index; i++)
	{
		a = gMemoryMappings.log2phys[i].logical;
		l = gMemoryMappings.log2phys[i].length;
		p = gMemoryMappings.log2phys[i].physical;
		
		/* adjust length so that 
		 *		0x50000000 + 0xB000000
		 * does NOT become 0x0, but instead becomes 0xFFFFFFFF.
		 * (which does not mean NOT_MAPPED but instead
		 * denotes the last valid byte for the address range!)
		 */
		l -= 1;	

		if (p != NOT_MAPPED)
		{
			if ((physAddr >= p) && (physAddr <= (p + l)))
			{
				return physAddr - p + a;
			}
		}
	}
	
	UnsupportedNum ("Physical address (0x%lX) has no logical mapping.", physAddr);
	
	return 0;
}

/*
 *	splitFirstRAMentry
 *
 *	This routine will split the first mapping if it
 *	does not begin 256K aligned.
 */
static void splitFirstRAMentry (void)
{
	unsigned long		off;
	int					i;
	
	off = gMemoryMappings.log2phys[0].physical & 0x0003FFFF;
	if (off)
	{
		for (i = gMemoryMappings.index; i > 0; i--)
		{
			gMemoryMappings.log2phys[i] = gMemoryMappings.log2phys[i-1];
		}
		
		off = 0x00040000 - off;
		
		gMemoryMappings.log2phys[0].length = off;
		
		gMemoryMappings.log2phys[1].physical += off;
		gMemoryMappings.log2phys[1].logical += off;
		gMemoryMappings.log2phys[1].length -= off;
	}
}

/*
 *	findRAM
 *
 */
void findRAM (unsigned long *num_memory, struct mem_info *memory)
{
	int					i, j, N;
	unsigned long		a, l , p;
	struct mem_info		t;
	Boolean				sortAction;
	
	N = 0;
	for (i = 0; i < gMemoryMappings.index; i++)
	{
		a = gMemoryMappings.log2phys[i].logical;
		l = gMemoryMappings.log2phys[i].length;
		p = gMemoryMappings.log2phys[i].physical;
		
		if (p != NOT_MAPPED && a < 0x40000000)
		{
			for (j = 0; j < N; j++)
			{
				if ((memory[j].addr + memory[j].size) == p)
				{
					memory[j].size += l;
					break;
				}
			}
			
			if (j == N)
			{
				memory[j].addr = p;
				memory[j].size = l;
				N += 1;
			}
		}
	}

	/*	Bubble sort the memory array
	 *	Sort so that bigger physical sized blocks are before smaller ones
	 */
	do
	{
		sortAction = false;
		
		for (j = 0; j < N-1; j++)
		{
			if (memory[j].size < memory[j+1].size)
			{
				t = memory[j+1];
				memory[j+1] = memory[j];
				memory[j] = t;
				
				sortAction = true;
			}
		}
	} while (sortAction);
	
	*num_memory = N;
}

/*
 *	check020_kernel_pos
 *
 *	boot_init_bootinfo(), findRAM() must have been called before
 *	this routine is used.
 */
void
check020_kernel_pos (u32 num_memory, struct mem_info *memory)
{
	int					j;
	struct mem_info		t;
	Boolean				sortAction;

	/* Hack for 020/68851. Kernel "head.S" does not handle
	 * 020 with > 1 memory segment and kernel not in first
	 * segment. Force kernel into first memory segment on
	 * these machines.
	 */

	/* Skip if not 68020 or number of segments <= 1
	 */
	if ( ( !(bi.cputype & CPU_68020) ) || (num_memory <= 1) )
		return;

	/*	Bubble sort the memory array
	 *	Sort so that lower physical addresses comes first
	 */

	do
	{
		sortAction = false;
		
		for (j = 0; j < num_memory-1; j++)
		{
			if (memory[j].addr > memory[j+1].addr)
			{
				t = memory[j+1];
				memory[j+1] = memory[j];
				memory[j] = t;
				
				sortAction = true;
			}
		}
	} while (sortAction);
}

/*
 *	Unsupported
 *
 */
void Unsupported (char *str)
{
	cprintf ("Error: this program has run into a condition that is unsupported:\n");
	cprintf ("	Please send this information to mikaelf@comenius.se\n");

	cprintf ("\n\t%s\n", str);

	longjmp (jmpState, -1);
}

/*
 *	UnsupportedNum
 *
 */
void UnsupportedNum (char *str, unsigned long n)
{
	cprintf ("Error: this program has run into a condition that is unsupported:\n");
	cprintf ("	Please send this information to mikaelf@comenius.se\n");

	cprintf (str, n);
	cprintf ("\n");

	longjmp (jmpState, -1);
}

/*
 *	Error
 *
 */
void Error (char *str)
{
	cprintf ("Error:\n");
	cprintf (str);
	cprintf ("\n");

	longjmp (jmpState, -1);
}

/*
 *	ErrorNum
 *
 */
void ErrorNum (char *str, unsigned long n)
{
	cprintf ("Error:\n");
	cprintf (str, n);
	cprintf ("\n");

	longjmp (jmpState, -1);
}

/*
 *	ErrorNumNum
 *
 */
void ErrorNumNum (char *str, unsigned long n1, unsigned long n2)
{
	cprintf ("Error:\n");
	cprintf (str, n1, n2);
	cprintf ("\n");

	longjmp (jmpState, -1);
}

/*
 *	tableReadData
 *
 */
void tableReadData (void *addr)
{
//#pragma unused (addr)
	
}

/*
 *	addReadData
 *
 */
void addReadData (void *addr, unsigned long data)
{
//#pragma unused (addr, data)

}
