/****************************************************************************
** ImagePreview class
**
** Created: Tue Feb 02 22:06:51 2004
**      by: Varol Okan using Kate
**
** This class is the encapsulation of the ImagePreview from the FileSelection
** Dialog.
** It is also used in the ImageDialog.
**
****************************************************************************/
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qpopupmenu.h>
#include <qfileinfo.h>
#include <qlistview.h>
#include <qlineedit.h>
#include <qpainter.h>
#include <qwmatrix.h>
#include <qpixmap.h>
#include <qimage.h>
#include <qdom.h>

#include "xml_dvd.h"
#include "menuobject.h"
#include "frameobject.h"
#include "imageobject.h"
#include "dialogmatrix.h"
#include "dialoggeometry.h"
#include "dialoganimation.h"

ImageObject::ImageObject (QWidget *pParent)
	: MenuObject (pParent)
{
	m_qsObjectType = QString (IMAGE_OBJECT);
}

ImageObject::~ImageObject ()
{
}

void ImageObject::setPixmap(QPixmap &thePixmap)
{
	// Note: the Pixmap could be rotated, thus the bounding Rect might
	// be of different size than the width or height of the Pixmap.
	m_pixmap = thePixmap;
}

void ImageObject::setFile (QString fileName)
{
	QString qsExtractionPath;
	QFileInfo fileInfo(fileName);
	m_qsFileName = fileInfo.fileName();
	setName(m_qsFileName);
	qsExtractionPath = fileInfo.baseName ();
	setExtractionPath(qsExtractionPath);
	m_qsFileName = fileName;
//	m_pixmap = QPixmap (fileName);
}

void ImageObject::setZoom (float fZoom)
{
	// For the purpose of this class the Zoom is alreaduy incorporated into m_pixmap ...
	m_imageManipulator.fZoom = fZoom;
}

void ImageObject::setManipulator(ImageManipulator &theOther)
{
	m_imageManipulator = theOther;
}

ImageManipulator &ImageObject::manipulator()
{
	return m_imageManipulator;
}

Modifiers *ImageObject::modifiers()
{
	// Return the base class only ...
	// Note: this is needed to use the same MatrixDialog for text and images.
	return (Modifiers *) &m_imageManipulator;
}

QString &ImageObject::fileName()
{
	return m_qsFileName;
}

QPixmap &ImageObject::pixmap ()
{
	return m_pixmap;
}

float ImageObject::zoom()
{
	return m_imageManipulator.fZoom;
}

void ImageObject::drawContents (QPainter *painter)
{
	QWMatrix theMatrix;
	// Here we calculate the center point of gravity (rotation)
	QPoint centerPos;
	centerPos.setX (rect().x() + (int)((float)rect().width () / 2.0));
	centerPos.setY (rect().y() + (int)((float)rect().height() / 2.0));
	// Here we define the cenetered rect.
	QRect theRect ((int)-(rect().width()/ 2.0), (int)-(rect().height()/2.0), rect().width(), rect().height());

	QPen thePen (QColor (255,0,0), 2, Qt::DashDotLine);
	painter->setPen(thePen);

	// From the Qt documentation ...
	// This transformation engine is a three-step pipeline
	// The first step uses the world transformation matrix.
	// The second step uses the window.
	// The third step uses the viewport.

	// First we translate to the appropriate location,
	theMatrix.translate ((double)centerPos.x(), (double)centerPos.y());

	//. then we apply the other modifications ...
//	theMatrix.scale  (manipulator().fZoom, manipulator().fZoom);
	theMatrix.scale  (modifiers()->fScaleX, modifiers()->fScaleY);
	theMatrix.rotate (manipulator().fRotate);
	theMatrix.shear  (manipulator().fShearX, manipulator().fShearY);

	// Here we get the rect that encompasses the button for this object.
	m_boundingRect = theMatrix.mapRect(theRect);
	if (m_bDrawRect)
		painter->drawRect(m_boundingRect);

	painter->setWorldMatrix(theMatrix);

	// Here we re-set the world matrix and translate the newly drawn ... thing ...
	if (m_bDrawRect)
		painter->drawRect(theRect);
	
	theMatrix.reset();
	painter->setWorldMatrix(theMatrix);
	
	// check if we have a complete transparency, in which case we don't even need to bother...
	if (manipulator().fTransparency == 1.0)
		return;

	if (!m_bDrawRect)	{
		int x, y;
		QRgb thePixel;
		int iSourceHeight = pixmap().height();
		int iSourceWidth  = pixmap().width ();
		int iSourceStartX = 0;
		int iSourceStartY = 0;
		int iTargetStartX = painter->viewport().x() + m_boundingRect.x();
		int iTargetStartY = painter->viewport().y() + m_boundingRect.y();
		// Use the fast method whenever possible
		if ((	(manipulator().fRotate ==   0.0) || (manipulator().fRotate ==  90.0) ||
			(manipulator().fRotate == 180.0) || (manipulator().fRotate == 270.0) ) &&
			( (manipulator().fShearX == 0.0) && (manipulator().fShearY ==   0.0) && 
			  (manipulator().fTransparency == 0.0) )
		   )
			bitBlt (painter->device(), iTargetStartX, iTargetStartY, &pixmap(), iSourceStartX, iSourceStartY, iSourceWidth, iSourceHeight, Qt::CopyROP, true);
		// Okay if we can not use bitBlt we have to go through the exact calculations ...
		else	{
			// First we need to create a Pixmap which holds the Target pixels
			QImage theImage = pixmap().convertToImage();
			QColor sourceColor, targetColor, theColor;
			QRgb sourcePixel;
			QImage sourceImg;
			float fTargetTrans, fSourceTrans;
			fTargetTrans = 1.0 - manipulator().fTransparency;
			fSourceTrans = manipulator().fTransparency;
			if (fTargetTrans != 0.0)	{
				QPixmap sourcePix (iSourceWidth, iSourceHeight);
				bitBlt (&sourcePix, 0, 0, painter->device(), iTargetStartX, iTargetStartY, iSourceWidth, iSourceHeight, Qt::CopyROP, true);
				// To get Pixel information we need a QImage however ...
				sourceImg = sourcePix.convertToImage();
				// Now we can get the original Pixel information ...
//				Note: This ought to be done in the updatePixmap () - function
//					- Also there we leave all info untouched which is (0,0,0)
//					- Also when we move AND when fTransparency != 0.0 then we need to do updatePixmap everytime.
			}			
			// Note that this function is actually using the viewport - transformation provided by Qt. The previous bitBlt's
			// did paint directly onto the device, thus we had to manually add the viewport coords. Now we have this info already
			// in the painter and thus we can substract them here.
			iTargetStartX -= painter->viewport().x();
			iTargetStartY -= painter->viewport().y();
			for (y=0; y < iSourceHeight; y++)	{
				for (x=0; x < iSourceWidth; x++)	{
					thePixel = theImage.pixel(x+iSourceStartX, y+iSourceStartY);
					thePixel &= 0x00ffffff;			// filter out the transparency part of the color
					if (thePixel != 0x000000)	{	// And check if this is a pixel we need to handle ...
						if (fTargetTrans != 0.0)	{		// If there is a transparency part, we need to 
							sourcePixel = sourceImg.pixel(x, y);	// calculate the mix of background and foreground.
							sourceColor = QColor (sourcePixel);
							targetColor = QColor ( thePixel );
							theColor = QColor (
								(int)(fSourceTrans * sourceColor.red()   + (int)(fTargetTrans * targetColor.red())),
								(int)(fSourceTrans * sourceColor.green() + (int)(fTargetTrans * targetColor.green())),
								(int)(fSourceTrans * sourceColor.blue()  + (int)(fTargetTrans * targetColor.blue()))
							);
							painter->setPen ( theColor );
						}
						else	// No transparency is slightly easier ...
							painter->setPen ( QColor ( thePixel ) );
						painter->drawPoint(x+iTargetStartX, y+iTargetStartY);
					}
				}
			}
		}
	}
}

void ImageObject::drawContents (QPainter *pPainter, int iRenderFrameNumber, int)
{
	// Next is to set the AnimationAttributes
	for (uint t=0;t<m_listAnimationAttributes.count();t++)
		m_listAnimationAttributes[t]->setValue (iRenderFrameNumber);
//printf ("<%ld>X1=<%d> Y1=<%d> trans=<%03f>\n", iRenderFrameNumber, m_rect.x(), m_rect.y(), m_imageManipulator.fTransparency);

	drawContents (pPainter);
}

/* Without transparency ...
void ImageObject::drawContents (QPainter *painter)
{
	QWMatrix theMatrix;
	// Here we calculate the center point of gravity (rotation)
	QPoint centerPos;
	centerPos.setX (rect().x() + (int)((float)rect().width () / 2.0));
	centerPos.setY (rect().y() + (int)((float)rect().height() / 2.0));
	// Here we define the cenetered rect.
	QRect theRect ((int)-(rect().width()/ 2.0), (int)-(rect().height()/2.0), rect().width(), rect().height());

	QPen thePen (QColor (255,0,0), 2, Qt::DashDotLine);
	painter->setPen(thePen);

	// From the Qt documentation ...
	// This transformation engine is a three-step pipeline
	// The first step uses the world transformation matrix.
	// The second step uses the window.
	// The third step uses the viewport.

	// First we translate to the appropriate location,
	theMatrix.translate ((double)centerPos.x(), (double)centerPos.y());

	//. then we apply the other modifications ...
//	theMatrix.scale  (manipulator().fZoom, manipulator().fZoom);
	theMatrix.rotate (manipulator().fRotate);
	theMatrix.shear  (manipulator().fShearX, manipulator().fShearY);

	// Here we get the rect that encompasses the button for this object.
	m_boundingRect = theMatrix.mapRect(theRect);

	if (m_bDrawRect)
		painter->drawRect(m_boundingRect);

	painter->setWorldMatrix(theMatrix);

	// Here we re-set the world matrix and translate the newly drawn ... thing ...
	if (m_bDrawRect)
		painter->drawRect(theRect);

	theMatrix.reset();
	painter->setWorldMatrix(theMatrix);
	if (!m_bDrawRect)	{
		int x, y;
		QRgb thePixel;
		int iSourceHeight = pixmap().height();
		int iSourceWidth  = pixmap().width ();
		int iSourceStartX = 0;
		int iSourceStartY = 0;
		int iTargetStartX = painter->viewport().x() + m_boundingRect.x();
		int iTargetStartY = painter->viewport().y() + m_boundingRect.y();
		if (( (manipulator().fRotate ==   0.0) || (manipulator().fRotate ==  90.0) ||
			  (manipulator().fRotate == 180.0) || (manipulator().fRotate == 270.0) ) &&
			( (manipulator().fShearX == 0.0) && (manipulator().fShearY ==   0.0) )
		   )
			bitBlt (painter->device(), iTargetStartX, iTargetStartY, &pixmap(), iSourceStartX, iSourceStartY, iSourceWidth, iSourceHeight, Qt::CopyROP, true);
		else	{
			QImage theImage = pixmap().convertToImage();
			// Okay if the rotation is any angle, then we get black background even with background image enabled.
			// Here we copy only those pixel over, which are part of the rotated image.
			for (y=0; y < iSourceHeight; y++)	{
				for (x=0; x < iSourceWidth; x++)	{
					thePixel = theImage.pixel(x+iSourceStartX, y+iSourceStartY);
					thePixel &= 0x00ffffff;	// filter out the transparency part
					if (thePixel != 0x000000)	{
						painter->setPen (QColor (thePixel));
						painter->drawPoint(x+iTargetStartX, y+iTargetStartY);
					}
				}
			}
		}
	}
}
*/
void ImageObject::updatePixmap()
{
	// This function will take the input image and pass it through the whole modifications pipeline,
	// so we get a nice and clean rotated / colored image again.

	// In order to speed up the whole procedure, we set the matrix first to scale the image,
	// then we handle the color functions on the smaller image and lastly we apply the rotation,shearing...
	int iRed, iGreen, iBlue;
	int x, y;
	bool bLighter = true;
	float fAdjust = 0.0;
	QColor theColor;
	QRgb thePix;	// QRgb defined as :  0xAARRGGBB;
	QWMatrix theMatrix;
	// Here we calculate the center point of gravity (rotation)
	QPoint centerPos;

	if (fileName().isEmpty())
		return;

	QImage theImage(fileName());
	if (theImage.isNull())
		return;
	theImage = theImage.smoothScale ( rect ().width(), rect().height(), QImage::ScaleFree);

	centerPos.setX (rect().x() + (int)((float)rect().width () / 2.0));
	centerPos.setY (rect().y() + (int)((float)rect().height() / 2.0));
	// First we translate to the appropriate location,
	theMatrix.translate ((double)centerPos.x(), (double)centerPos.y());
	// then we apply the other modifications ...
	theMatrix.scale  (modifiers()->fScaleX, modifiers()->fScaleY);
//	theMatrix.scale  (manipulator().fZoom, manipulator().fZoom);
	// Here we create the scaled image ...
	theImage = theImage.xForm ( theMatrix );

	// create empty image.
	QImage imageColored = QImage (theImage.width(), theImage.height(), theImage.depth());
	if (manipulator().fBrightness > 0.0)	{
		bLighter = true;
		fAdjust = 100 + manipulator().fBrightness * 300;
	}
	else	{
		bLighter = false;
		fAdjust = 100 + manipulator().fBrightness * -300;
	}
	for (y=0;y<theImage.height();y++)	{
		for (x=0;x<theImage.width();x++)	{
			thePix = theImage.pixel(x, y);
			iRed = (int)((float)qRed(thePix) * manipulator().fRed);
			iBlue = (int)((float)qBlue(thePix) * manipulator().fBlue);
			iGreen = (int)((float)qGreen(thePix) * manipulator().fGreen);

			theColor = QColor (iRed, iGreen, iBlue);
			if (bLighter)
				theColor = theColor.light ((int)fAdjust);
			else
				theColor = theColor.dark ((int)fAdjust);
			// This test is to mask out the transparency color (0, 0, 0)
			if (theColor == QColor (0,0,0))
				theColor = QColor (0,0,1);
			imageColored.setPixel (x, y, theColor.rgb());
		}
	}
	// Next we re-set the matrix ...
	theMatrix.reset ();
	// and plug in the remaining operations ...
	theMatrix.rotate (manipulator().fRotate);
	theMatrix.shear  (manipulator().fShearX, manipulator().fShearY);
	theImage = imageColored.xForm( theMatrix );
	m_pixmap.convertFromImage(theImage);
	emit (signalUpdatePixmap());
}

bool ImageObject::readProjectFile  (QDomNode &theNode)
{
	QDomElement theElement = theNode.toElement();
	// The video node has only attributes and nothing else.
	QDomAttr attribute = theElement.attributeNode ( IMAGE_OBJECT_FILE_NAME );
	if (!attribute.isNull())
		setFile ( attribute.value() );

	QDomNode manNode = theNode.firstChild();
	while (!manNode.isNull())	{
		QDomElement manElement = manNode.toElement();
		if (manElement.tagName() == MANIPULATOR)	{
			attribute = manElement.attributeNode ( MANIPULATOR_FILE_NAME );
			if (!attribute.isNull())
				manipulator().backgroundFileName = attribute.value();

			attribute = manElement.attributeNode ( MANIPULATOR_SHOW_BACKGROUND );
			if (!attribute.isNull())
				 manipulator().bShowBackground = (attribute.value() == QString ("yes"));

	 		attribute = manElement.attributeNode ( MANIPULATOR_STARTX );
			if (!attribute.isNull())
				 manipulator().iStartX = attribute.value().toInt();

	 		attribute = manElement.attributeNode ( MANIPULATOR_STARTY );
			if (!attribute.isNull())
				 manipulator().iStartY = attribute.value().toInt();

	 		attribute = manElement.attributeNode ( MANIPULATOR_XRES );
			if (!attribute.isNull())
				 manipulator().iXRes = attribute.value().toInt();

	 		attribute = manElement.attributeNode ( MANIPULATOR_YRES );
			if (!attribute.isNull())
				 manipulator().iYRes = attribute.value().toInt();

	 		attribute = manElement.attributeNode ( MANIPULATOR_TRANSFORMATION_MODE );
			if (!attribute.isNull())
				 manipulator().iTransformationMode = attribute.value().toInt();

	 		attribute = manElement.attributeNode ( MANIPULATOR_ASPECT_RATIO );
			if (!attribute.isNull())
				 manipulator().fAspectRatio = attribute.value().toFloat();

	 		attribute = manElement.attributeNode ( MANIPULATOR_RED );
			if (!attribute.isNull())
				 manipulator().fRed = attribute.value().toFloat();

	 		attribute = manElement.attributeNode ( MANIPULATOR_GREEN );
			if (!attribute.isNull())
				 manipulator().fGreen = attribute.value().toFloat();

	 		attribute = manElement.attributeNode ( MANIPULATOR_BLUE );
			if (!attribute.isNull())
				 manipulator().fBlue = attribute.value().toFloat();

	 		attribute = manElement.attributeNode ( MANIPULATOR_BRIGHTNESS );
			if (!attribute.isNull())
				 manipulator().fBrightness = attribute.value().toFloat();
		}
		else if (manElement.tagName() == MENU_OBJECT)
			MenuObject::readProjectFile( theNode );

		manNode = manNode.nextSibling();
	}
	this->updatePixmap ();
	return true;
}

bool ImageObject::writeProjectFile (QDomElement &theElement)
{
	QDomDocument xmlDoc = theElement.ownerDocument();
	QDomElement imageNode = xmlDoc.createElement( IMAGE_OBJECT );	// <FrameObject>
	// Here we set the attributes of the <dvdauthor> tag
	if (!m_qsFileName.isNull())
		imageNode.setAttribute( IMAGE_OBJECT_FILE_NAME, m_qsFileName );

	// Store the ImageManipulator - cariables (if neccesarry) ...
	ImageManipulator initialValues;
	if ((!manipulator().backgroundFileName.isEmpty())                                ||
		(manipulator().bShowBackground != initialValues.bShowBackground)         ||
		(manipulator().iStartX != initialValues.iStartX)                         ||
		(manipulator().iStartY != initialValues.iStartY)                         ||
		(manipulator().iXRes != initialValues.iXRes)                             ||
		(manipulator().iYRes != initialValues.iYRes)                             ||
		(manipulator().iTransformationMode != initialValues.iTransformationMode) ||
		(manipulator().fAspectRatio != initialValues.fAspectRatio)               ||
		(manipulator().fRed != initialValues.fRed)                               ||
		(manipulator().fGreen != initialValues.fGreen)                           ||
		(manipulator().fBlue != initialValues.fBlue)                             ||
		(manipulator().fBrightness != initialValues.fBrightness) )                {

		QDomElement manipulatorNode = xmlDoc.createElement ( MANIPULATOR );
		if (!manipulator().backgroundFileName.isEmpty())
			manipulatorNode.setAttribute( MANIPULATOR_FILE_NAME, manipulator().backgroundFileName );
		if (manipulator().bShowBackground != initialValues.bShowBackground)
			manipulatorNode.setAttribute( MANIPULATOR_SHOW_BACKGROUND, manipulator().bShowBackground ? QString ("yes") : QString ("no"));
		if (manipulator().iStartX != initialValues.iStartX)
			manipulatorNode.setAttribute( MANIPULATOR_STARTX, QString ("%1").arg(manipulator().iStartX) );
		if (manipulator().iStartY != initialValues.iStartY)
			manipulatorNode.setAttribute( MANIPULATOR_STARTY, QString ("%1").arg(manipulator().iStartY) );
		if (manipulator().iXRes != initialValues.iXRes)
			manipulatorNode.setAttribute( MANIPULATOR_XRES, QString ("%1").arg(manipulator().iXRes) );
		if (manipulator().iYRes != initialValues.iYRes)
			manipulatorNode.setAttribute( MANIPULATOR_YRES, QString ("%1").arg(manipulator().iYRes) );
		if (manipulator().iTransformationMode != initialValues.iTransformationMode)
			manipulatorNode.setAttribute( MANIPULATOR_TRANSFORMATION_MODE, QString ("%1").arg(manipulator().iTransformationMode) );
		if (manipulator().fAspectRatio != initialValues.fAspectRatio)
			manipulatorNode.setAttribute( MANIPULATOR_ASPECT_RATIO, QString ("%1").arg(manipulator().fAspectRatio) );
		if (manipulator().fRed != initialValues.fRed)
			manipulatorNode.setAttribute( MANIPULATOR_RED, QString ("%1").arg(manipulator().fRed) );
		if (manipulator().fGreen != initialValues.fGreen)
			manipulatorNode.setAttribute( MANIPULATOR_GREEN, QString ("%1").arg(manipulator().fGreen) );
		if (manipulator().fBlue != initialValues.fBlue)
			manipulatorNode.setAttribute( MANIPULATOR_BLUE, QString ("%1").arg(manipulator().fBlue) );
		if (manipulator().fBrightness != initialValues.fBrightness)
			manipulatorNode.setAttribute( MANIPULATOR_BRIGHTNESS, QString ("%1").arg(manipulator().fBrightness) );
		imageNode.appendChild( manipulatorNode );
	}

	theElement.appendChild( imageNode );
	// And here we write the base class ...
	return MenuObject::writeProjectFile( imageNode );
}

void ImageObject::createStructure (QListViewItem *pParentItem)
{
	QListViewItem *pButtonItem;
	QListViewItem *pModifierItem;
	QFileInfo fileInfo(m_qsFileName);
	QString qsName;
	
	qsName = QString ("Image Object");

	pButtonItem = new QListViewItem(pParentItem, qsName, fileInfo.fileName());
	pButtonItem->setExpandable (TRUE);

	ImageObject initialValues;
	// And finally we create the modifiers info ...
	if ( !(	(manipulator().fRotate     == initialValues.manipulator().fRotate) &&
			(manipulator().fShearX       == initialValues.manipulator().fShearX) &&
			(manipulator().fShearY       == initialValues.manipulator().fShearY) &&
			(manipulator().fZoom         == initialValues.manipulator().fZoom)   &&
			(manipulator().fBrightness   == initialValues.manipulator().fBrightness) &&
			(manipulator().fTransparency == initialValues.manipulator().fTransparency) &&
			(manipulator().fRed          == initialValues.manipulator().fRed)    &&
			(manipulator().fGreen        == initialValues.manipulator().fGreen)  &&
			(manipulator().fBlue         == initialValues.manipulator().fBlue)
			) )	{
		pModifierItem = new QListViewItem(pButtonItem, QString ("Modifiers"), NULL);
		pButtonItem->setExpandable (TRUE);

		if ((manipulator().fRed        != initialValues.manipulator().fRed)   ||
			(manipulator().fGreen      != initialValues.manipulator().fGreen) ||
			(manipulator().fBlue       != initialValues.manipulator().fBlue) )
			new QListViewItem(pModifierItem, QString ("Color Adjust"),
				QString ("%1, %2, %3").arg(manipulator().fRed).arg(manipulator().fGreen).arg(manipulator().fBlue));
		if (manipulator().fBrightness  != initialValues.manipulator().fBrightness)
			new QListViewItem(pModifierItem, QString ("Brightness"),
				QString ("%1").arg(manipulator().fBrightness));
		if (manipulator().fTransparency  != initialValues.manipulator().fTransparency)
			new QListViewItem(pModifierItem, QString ("Transparency"),
				QString ("%1").arg(manipulator().fTransparency));
		if ((manipulator().fShearX  != initialValues.manipulator().fShearX) ||
			(manipulator().fShearY  != initialValues.manipulator().fShearY)  )
			new QListViewItem(pModifierItem, QString ("Shear (x, y)"),
				QString ("%1, %2").arg(manipulator().fShearX).arg(manipulator().fShearY));
		if (manipulator().fZoom  != initialValues.manipulator().fZoom)
			new QListViewItem(pModifierItem, QString ("Zoom"),
				QString ("%1").arg(manipulator().fZoom));
		if (manipulator().fRotate  != initialValues.manipulator().fRotate)
			new QListViewItem(pModifierItem, QString ("Rotate"),
				QString ("%1 deg").arg(manipulator().fRotate));
	}

	new QListViewItem(pButtonItem, QString ("Geometry"),
		QString ("%1, %2, %3, %4").
		arg(rect().x()).arg(rect().y()).arg(rect().width()).arg(rect().height()));

}

bool ImageObject::mouseMoveEvent (QMouseEvent *pEvent)
{
	// This will move the textObject around and around and around we go ...
	QRect theRect = rect();
	int iX, iY, iWidth, iHeight;
	iWidth  = rect().width ();
	iHeight = rect().height();
	iX = rect().x() - (m_currentMousePos.x() - pEvent->pos().x());
	iY = rect().y() - (m_currentMousePos.y() - pEvent->pos().y());
	theRect.setX(iX);
	theRect.setY(iY);
	theRect.setWidth(iWidth);
	theRect.setHeight(iHeight);

	setRect(theRect);
	m_currentMousePos = pEvent->pos();
	return false;
}

bool ImageObject::mousePressEvent (QMouseEvent *pEvent)
{
	m_currentMousePos = pEvent->pos();
	if (pEvent->button() == Qt::RightButton)	{
		QPoint globalPos = pEvent->globalPos();
		QPopupMenu *pStackMenu = new QPopupMenu(m_pParent);
		pStackMenu->insertItem ( tr ("To Front")  , this, SLOT(slotToFront()));
		pStackMenu->insertItem ( tr ("To Back")   , this, SLOT(slotToBack()));
		globalPos.setX (globalPos.x() - pStackMenu->sizeHint().width()); // -100);
		pStackMenu->popup(globalPos, 1);

		if (m_pContextMenu)
			delete m_pContextMenu;
		m_pContextMenu = new QPopupMenu(m_pParent);
		m_pContextMenu->insertItem ( QString ("Edit ...")      , this, SLOT(slotEdit()));
		m_pContextMenu->insertItem ( QString ("Properties ..."), this, SLOT(slotProperties()));
		m_pContextMenu->insertItem ( QString ("Matrix ...")    , this, SLOT(slotMatrix()));
		m_pContextMenu->insertItem ( QString ("Delete")        , this, SLOT(slotDelete()));
		m_pContextMenu->insertSeparator();
		m_pContextMenu->insertItem ( QString ("Rotate 90")     , this, SLOT(slotRotate90 ()));
		m_pContextMenu->insertItem ( QString ("Rotate 180")    , this, SLOT(slotRotate180()));
		m_pContextMenu->insertItem ( QString ("Rotate 270")    , this, SLOT(slotRotate270()));
		m_pContextMenu->insertSeparator();
		m_pContextMenu->insertItem ( QString ("Add Frame")     , this, SLOT(slotAddFrame()));
		m_pContextMenu->insertItem ( QString ("Add Text")      , this, SLOT(slotAddText()));
		m_pContextMenu->insertSeparator();
		m_pContextMenu->insertItem ( QString ("Define as Button") , this, SLOT(slotDefineAsButton()));
		// Blocking call.
		m_pContextMenu->exec(pEvent->globalPos(), 6);
		delete pStackMenu;
		if (m_pContextMenu)
			delete m_pContextMenu;
		m_pContextMenu = NULL;
		// Here we mark that the user called a menu item thus we don't want the
		// base classes to continue with the mouse event
//		if (iReturn != -1)
			return true;
	}
	else
		m_bDrawRect = true;
	return false;
}

bool ImageObject::mouseReleaseEvent (QMouseEvent *)
{
	m_bDrawRect = false;
	emit (signalUpdatePixmap());
	emit (signalUpdateStructure());
	return false;
}

bool ImageObject::mouseDoubleClickEvent (QMouseEvent *)
{
	// Back to the MenuPreview - class, where I can handle this request ...
	emit (signalModifyMe(this));
	return false;
}

void ImageObject::slotEdit()
{
	mouseDoubleClickEvent(NULL);
}

void ImageObject::slotProperties()
{
	DialogGeometry *pGeometry = new DialogGeometry ();
	pGeometry->initMe(rect());
	
	connect ( pGeometry->m_pButtonAnimation, SIGNAL(clicked()), this, SLOT(slotAnimation()) );
	
	if (pGeometry->exec() == QDialog::Accepted)	{
		int iX, iY, iWidth, iHeight;
		iX      = pGeometry->m_pEditX->text().toInt();
		iY      = pGeometry->m_pEditY->text().toInt();
		iWidth  = pGeometry->m_pEditWidth ->text().toInt();
		iHeight = pGeometry->m_pEditHeight->text().toInt();
		QRect rect (iX, iY, iWidth, iHeight);
		setRect (rect);
		updatePixmap ();
		emit (signalUpdatePixmap());
	}
	delete pGeometry;
}

void ImageObject::slotAnimation ()
{
	// This function gets the Animation Dialog
	// Within this dialo, you can create a script
	// (or hardcode) Attributes of the object.
	// Before rendering the menu the script will 
	// get executed and all data sent to stdout 
	// will be captured in the data file, wich 
	// will provide the data for the animation.
	QString qsEmpty;
	DialogAnimation dialog;
	dialog.initMe ( animation(), 300, 25.0, DialogAnimation::TypeImage );
	if ( dialog.exec () == QDialog::Accepted )
		setAnimation ( dialog.string () );

	if (animation().length () < 5)
		setAnimation ( qsEmpty );
}

void ImageObject::slotRotate90()
{
	rotate (90.0);
}

void ImageObject::slotRotate180()
{
	rotate (180.0);
}

void ImageObject::slotRotate270()
{
	rotate (270.0);
}

void ImageObject::rotate(float fRot)
{
//printf ("ImageObject::rotate (%f)<%f>\n", manipulator().fRotate, modifiers()->fRotate);
	QWMatrix theMatrix;
	theMatrix.rotate (fRot);
	float fRotate = manipulator().fRotate + fRot;
	if (fRotate > 180.0)
		fRotate -= 360.0;
	manipulator().fRotate = fRotate;
	m_pixmap = m_pixmap.xForm( theMatrix );
	emit (signalUpdatePixmap());
	parent()->update();
}

void ImageObject::slotMatrix()
{
	QMessageBox::warning (NULL, QString ("Warning"), QString ("This Dialog is very instabel ...\n"), QMessageBox::Ok,  QMessageBox::NoButton);

	DialogMatrix *pDialog = new DialogMatrix(MenuObject::parent());
	pDialog->initMe(this);
	connect (this, SIGNAL(signalDeleteMe(MenuObject *)), pDialog, SLOT(slotDeleteObject(MenuObject *)));
	pDialog->show();
}

void ImageObject::slotAddFrame()
{
	QMessageBox::warning (NULL, QString ("Not Implemented yet."), QString ("Not Implemented yet."),
		QMessageBox::Ok ,  QMessageBox::Cancel);
}

void ImageObject::slotAddText()
{
	QMessageBox::warning (NULL, QString ("Not Implemented yet."), QString ("Not Implemented yet."),
		QMessageBox::Ok ,  QMessageBox::Cancel);
}

void ImageObject::slotDefineAsButton()
{
	emit (signalDefineAsButton(this));
}

MenuObject *ImageObject::clone()
{
	ImageObject *pNewObject = new ImageObject (MenuObject::parent());
	pNewObject->setPixmap(pixmap());
	pNewObject->setFile (fileName());
	pNewObject->setZoom (zoom());
	// The following two are not really needed, since we get those
	// informations solely from the Normal State - objects ...
	pNewObject->setRect(rect());
	pNewObject->setManipulator(manipulator());
//	pNewObject->setModifiers(*modifiers());

	return pNewObject;
}

AnimationAttribute *ImageObject::getSpecificAttributes (long iMaxNumberOfFrames, QString qsProperty)
{
	AnimationAttribute *pAnimAttr = NULL;

	if (qsProperty == "zoom")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.fZoom, &m_imageManipulator.fZoom);
	else if (qsProperty == "rotate")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.fRotate, &m_imageManipulator.fRotate);
	else if (qsProperty == "shear.x")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.fShearX, &m_imageManipulator.fShearX);
	else if (qsProperty == "shear.y")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.fShearY, &m_imageManipulator.fShearY);
	else if (qsProperty == "scale.x")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.fScaleX, &m_imageManipulator.fScaleX);
	else if (qsProperty == "scale.y")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.fScaleY, &m_imageManipulator.fScaleY);
	else if (qsProperty == "transparency")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.fTransparency, &m_imageManipulator.fTransparency);
	else if (qsProperty == "backgroundFileName")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.backgroundFileName, &m_imageManipulator.backgroundFileName);
	else if (qsProperty == "showBackground")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.bShowBackground, &m_imageManipulator.bShowBackground);
	else if (qsProperty == "startPos.x")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.iStartX, &m_imageManipulator.iStartX);
	else if (qsProperty == "startPos.y")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.iStartY, &m_imageManipulator.iStartY);
	else if (qsProperty == "res.x")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.iXRes, &m_imageManipulator.iXRes);
	else if (qsProperty == "res.y")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.iYRes, &m_imageManipulator.iYRes);
	else if (qsProperty == "aspectRatio")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.fAspectRatio, &m_imageManipulator.fAspectRatio);
	else if (qsProperty == "blendColor.red")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.fRed, &m_imageManipulator.fRed);
	else if (qsProperty == "blendColor.green")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.fGreen, &m_imageManipulator.fGreen);
	else if (qsProperty == "blendColor.blue")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.fBlue, &m_imageManipulator.fBlue);
	else if (qsProperty == "brightness")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_imageManipulator.fBrightness, &m_imageManipulator.fBrightness);

	return pAnimAttr;
}

