/****************************************************************************
** ImagePreview class
**
** Created: Tue Feb 02 22:06:51 2004
**      by: Varol Okan using Kate
**
** This class is the encapsulation of the ImagePreview from the FileSelection
** Dialog.
** It is also used in the ImageDialog.
**
****************************************************************************/

#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qtextedit.h>
#include <qpainter.h>
#include <qcursor.h>
#include <qslider.h>
#include <qimage.h>

#include "global.h"
#include "xml_dvd.h"
#include "qslideshow/xml_slideshow.h"
#include "buttonpreview.h"
#include "dialogtextfont.h"
#include "dialogbutton.h"
#include "menupreview.h"
#include "menuobject.h"
#include "frameobject.h"
#include "textobject.h"
#include "imageobject.h"
#include "movieobject.h"
#include "dialogimage.h"
#include "buttonobject.h"
#include "sourcefileentry.h"
#include "objectcollection.h"

MenuPreview::MenuPreview(QWidget * parent, const char * name, WFlags f)
	: ButtonPreview (parent, name, f)
{
	m_iCursorStyle       = Qt::ArrowCursor;
	m_bActiveCursor      = false;
	m_iCreateObjectStyle = TypeNone;
	m_pActiveObject      = NULL;
	m_pDialogButton      = NULL;
	m_bIsSubMenu         = false;
	m_pPgcColors        = NULL;
}

MenuPreview::~MenuPreview()
{

}

void MenuPreview::initMe (QColor *pColors)
{
	m_pPgcColors = pColors;
}

void MenuPreview::setIsSubMenu(bool bIsSubmenu)
{
	// This function is only here to flag that the MenuPreview is part of a SubMenu and
	// Not a VMGM, or something else.
	m_bIsSubMenu = bIsSubmenu;
}

void MenuPreview::mousePressEvent (QMouseEvent *pEvent)
{
	// First we let the base class have it ...
	m_bMouseEventDone = false;
	ButtonPreview::mousePressEvent(pEvent);
	// Here we check if the mouse click appeared withtin one of the MenuObjects,
	// in which case the Object will take over the mouse handling ...
	if (m_bMouseEventDone)
		return;
	QPoint thePos = pEvent->pos();
	MenuObject *pObject = childContains(thePos);
	if (pObject != NULL)	{
		m_pActiveObject = pObject;
		update();
		return;
	}
	// Okay the user actually clicked in the MenuPreview ...
	// The left mouse button was clicked.
	// If the user is in creation mode, then we handle this one
	if (pEvent->button () == LeftButton) 	{
		// otherwise we simply return.
		if (m_iCursorStyle == Qt::ArrowCursor)
			return;
		// Okay, at this point we know the user is creating an object.
		m_bActiveCursor = true;
		m_rectCurrentObject.setX(pEvent->pos().x());
		m_rectCurrentObject.setY(pEvent->pos().y());
		return;
	}
	// The right mousebutton was pressed means that we should display the context drop down menu.
	m_iCursorStyle = Qt::ArrowCursor;
	QCursor myCursor (m_iCursorStyle);
	setCursor(myCursor);
	m_bActiveCursor = false;

	QPopupMenu *pMenu = new QPopupMenu (this);
	pMenu->insertItem ( tr ("Add Frame"), this, SLOT(slotAddFrameObject()));
	pMenu->insertItem ( tr ("Add Text") , this, SLOT(slotAddTextObject()));
	pMenu->insertItem ( tr ("Add Image"), this, SLOT(slotAddImageObject()));
	pMenu->insertItem ( tr ("Add Movie"), this, SLOT(slotAddMovieObject()));
	pMenu->insertItem ( tr ("Add Collection"), this, SLOT(slotAddCollection()));
	if (m_bIsSubMenu)	{
		pMenu->insertSeparator ();
		pMenu->insertItem (tr ("Rename Menu"), this, SLOT(slotRenameMenu()));
		pMenu->insertItem (tr ("Delete Menu"), this, SLOT(slotDeleteMenu()));
	}
	pMenu->exec(pEvent->globalPos());
}

void MenuPreview::mouseReleaseEvent (QMouseEvent *pEvent)
{
	// First we let the base class have it ...
	m_bMouseEventDone = false;
	ButtonPreview::mouseReleaseEvent(pEvent);
	// Here we check if the mouse click appeared withtin one of the MenuObjects,
	// in which case the Object will take over the mouse handling ...
	if (m_bMouseEventDone)
		return;
	if (childContains((QPoint &)pEvent->pos()) != NULL)	{
		m_pActiveObject = NULL;
//		updatePixmap();	taken care of in the child deja ...
		update();
		return;
	}
	m_pActiveObject = NULL;
	// The user actually released the mousebutton in the MenuPreview - class.
	// The left mouse button was released.
	if (pEvent->button () == LeftButton) 	{
		// If the user is not drawing a button, we'll return.
		if (m_iCursorStyle == Qt::ArrowCursor)
			return;
		// The user is drawing a Button. Let us activate the drawing algol.
		m_rectCurrentObject.setRight (pEvent->pos().x());
		m_rectCurrentObject.setBottom (pEvent->pos().y());
		m_iCursorStyle = Qt::ArrowCursor;
		QCursor myCursor (m_iCursorStyle);
		setCursor(myCursor);
		m_bActiveCursor = false;
		// Here we create the chosen object ...
		switch (m_iCreateObjectStyle)	{
		case MenuPreview::FrameType :
			createFrameObject ();
		break;
		case MenuPreview::TextType :
			createTextObject();
		break;
		case MenuPreview::ImageType :
			createImageObject();
		break;
		case MenuPreview::MovieType :
			createMovieObject ();
		break;
		}
		update ();
	}
}

void MenuPreview::mouseDoubleClickEvent	(QMouseEvent *pEvent)
{
	m_pActiveObject = NULL;
	// First we let the base class have it ...
	ButtonPreview::mouseDoubleClickEvent(pEvent);
}

void MenuPreview::mouseMoveEvent (QMouseEvent *pEvent)
{
	// This part will move the active object around ...
	if (m_pActiveObject)	{
		m_pActiveObject->mouseMoveEvent(pEvent);
		update ();
		return;
	}
	if (!m_bActiveCursor)
		return;
	// Okay at this point we have m_rectCurrentObject.x, and .y set
	// Here we draw the dynamicly changing size of the rect.

	// First we clear the contents of the previous rectangle
	repaint(m_rectCurrentObject, FALSE);
	QPainter thePainter (this);
	QPen thePen (QColor (255, 30,30), 2, Qt::DashDotLine);
	thePainter.setPen(thePen);
	m_rectCurrentObject.setRight(pEvent->pos().x());
	m_rectCurrentObject.setBottom(pEvent->pos().y());
	thePainter.drawRect (m_rectCurrentObject);
}

bool MenuPreview::readProjectFile (QDomNode &theNode)
{
	uint t;
	if (m_pDialogButton)
		delete m_pDialogButton;
	m_pDialogButton = NULL;
	QDomNode xmlPreview = theNode.firstChild();
	while ( !xmlPreview.isNull () )	{
		// Here we created a MenuObject, we also want to
		MenuObject *pNewObject = readObject (xmlPreview);
		// add it to the list ...
		if (pNewObject)	{
			pNewObject->readProjectFile ( xmlPreview );
			m_listMenuObjects.append ( pNewObject );
		}
		// So lets get the next sibling ... until we hit hte end of DVDMenu ...
		xmlPreview = xmlPreview.nextSibling();
	}
	
	// we need to make one more step here 
	// since we stored the Display Name for ButtonObjects and we need the SourceFileEntry pointer ...
	ButtonObject tempButton;
	for (t=0;t<m_listMenuObjects.count();t++)	{
		if (m_listMenuObjects.at (t)->objectType() == tempButton.objectType())	{
			m_pActiveObject = m_listMenuObjects.at(t);
			emit (signalRequestSourceFiles());
		}
	}
	return true;
}

MenuObject *MenuPreview::readObject (QDomNode &objectNode)
{
	// This function wil create an MenuObject-derived object
	// depending on the info from the QDomNode - object
	QDomElement theElement = objectNode.toElement();
	QString tagName = theElement.tagName();
	QString nodeText = theElement.text ();

	// Okay, this is ugly but after all it is the simplest of all xml file structure.
	// No need to get fancy ...
	MenuObject *pNewObject = NULL;
	if (tagName == FRAME_OBJECT)
		pNewObject = createFrameObject (false);
	else if (tagName == TEXT_OBJECT)
		pNewObject = createTextObject (false);
	else if (tagName == IMAGE_OBJECT)
		pNewObject = createImageObject (false);
	else if (tagName == MOVIE_OBJECT)
		pNewObject = createMovieObject (false);
	else if (tagName == BUTTON_OBJECT)	{
		pNewObject = createButtonObject (false);
		// A small special handling for the Buttons ...
		// Funky, since the above function itself calls this function here ...
		((ButtonObject *)pNewObject)->readObjects(objectNode, this);
	}
	else if (tagName == OBJECT_COLLECTION)
		pNewObject = createObjectCollection (false);
	else
		printf ("Warning: MenuPreview::readObject -=> wrong XML Node <%s>\nContinuing ...\n",
				(const char *)tagName);
	// And finally, if we created a MenuObject, we also want to
	// add it to the list ...
	if (pNewObject)
		pNewObject->readProjectFile ( objectNode );
	// So lets get the next sibling ... until we hit hte end of DVDMenu ...
	return pNewObject;
}

bool MenuPreview::writeProjectFile (QDomElement &theElement)
{
	uint t;
	for (t=0;t<m_listMenuObjects.count();t++)	{
		if (!m_listMenuObjects.at(t)->writeProjectFile( theElement ))
			return false;
	}
	return true;
}

// The following three slots will add a button to the Menu.
void MenuPreview::slotAddFrameObject()
{
	m_iCursorStyle = Qt::CrossCursor;	// signals to be awaiting the next mouse click
		// which will then generate the rectangle for this text button.
	m_iCreateObjectStyle = MenuPreview::FrameType;
	QCursor myCursor(m_iCursorStyle);
	setCursor(myCursor);
}

void MenuPreview::slotAddTextObject()
{
	m_iCursorStyle = Qt::CrossCursor;	// signals to be awaiting the next mouse click
		// which will then generate the rectangle for this text button.
	m_iCreateObjectStyle = MenuPreview::TextType;
	QCursor myCursor(m_iCursorStyle);
	setCursor(myCursor);
}

void MenuPreview::slotAddImageObject ()
{
	m_iCursorStyle = Qt::CrossCursor;	// signals to be awaiting the next mouse click
		// which will then generate the rectangle for this text button.
	m_iCreateObjectStyle = MenuPreview::ImageType;
	QCursor myCursor(m_iCursorStyle);
	setCursor(myCursor);
}

void MenuPreview::slotAddMovieObject ()
{
	m_iCursorStyle = Qt::CrossCursor;	// signals to be awaiting the next mouse click
		// which will then generate the rectangle for this text button.
	m_iCreateObjectStyle = MenuPreview::MovieType;
	QCursor myCursor(m_iCursorStyle);
	setCursor(myCursor);
}

void MenuPreview::slotAddCollection()
{
	m_iCursorStyle = Qt::CrossCursor;	// signals to be awaiting the next mouse click
		// which will then generate the rectangle for this text button.
	m_iCreateObjectStyle = MenuPreview::CollectionType;
	QCursor myCursor(m_iCursorStyle);
	setCursor(myCursor);
}

MenuObject *MenuPreview::createFrameObject(bool bShowDialog)
{
	FrameObject *pFrameObject = new FrameObject (this);
	pFrameObject->setRect		(m_rectCurrentObject);
	pFrameObject->setFrameWidth  (4);
	pFrameObject->setFrameColor (QColor (START_FRAME_COLOR));
	if (bShowDialog)	{
		m_listMenuObjects.append (pFrameObject);
		emit (signalUpdateStructure());
		updatePixmap();
	}
	// Here we connect the signal to the slot ...
	connect (pFrameObject, SIGNAL(signalDefineAsButton(MenuObject *)), this, SLOT(slotDefineAsButton(MenuObject *)));
	connect (pFrameObject, SIGNAL(signalDeleteMe(MenuObject *)), this, SLOT(slotDeleteObject(MenuObject *)));
	connect (pFrameObject, SIGNAL(signalUpdateStructure()), this, SLOT(slotUpdateStructure()));
	connect (pFrameObject, SIGNAL(signalUpdatePixmap()), this, SLOT(slotUpdatePixmap()));

	return pFrameObject;
}

MenuObject *MenuPreview::createTextObject(bool bShowDialog)
{
	TextObject *pTextObject = NULL;
	if (bShowDialog)	{
		DialogTextFont fontDialog (this);
		if (fontDialog.exec() == QDialog::Rejected)
			return NULL;

		pTextObject = new TextObject (this);
		QString qsText = fontDialog.getText();
		pTextObject->setRect (m_rectCurrentObject);
		pTextObject->setFont (fontDialog.getFont());
		pTextObject->setText (qsText);
		pTextObject->setTextAlign(fontDialog.getTextAlign());
		pTextObject->setBackgroundColor (fontDialog.getBackgroundColor());
		pTextObject->setForegroundColor (fontDialog.getForegroundColor());
		// Checks wether the user wants to fit the size of the button to the text or not.
		if (fontDialog.getFit())	{
			QRect rect = pTextObject->rect();
			rect.setWidth (1);
			rect.setHeight(1);
			pTextObject->setRect(rect);
		}
		m_listMenuObjects.append (pTextObject);
		emit (signalUpdateStructure());
		updatePixmap();
		pTextObject->update ();
	}
	else
		pTextObject = new TextObject (this);

	// Here we connect the signal to the slot ...
	connect (pTextObject, SIGNAL(signalDefineAsButton(MenuObject *)), this, SLOT(slotDefineAsButton(MenuObject *)));
	connect (pTextObject, SIGNAL(signalDeleteMe(MenuObject *)), this, SLOT(slotDeleteObject(MenuObject *)));
	connect (pTextObject, SIGNAL(signalUpdateStructure()), this, SLOT(slotUpdateStructure()));
	connect (pTextObject, SIGNAL(signalUpdatePixmap()), this, SLOT(slotUpdatePixmap()));
	return pTextObject;
}

MenuObject *MenuPreview::createImageObject(bool bShowDialog)
{
	ImageObject *pImageObject = NULL;

	if (bShowDialog)	{
		// The first step is to get the image name
		QString qsImageName = QFileDialog::getOpenFileName(m_qsCurrentPath, tr("Image Files (*.jpg *.jpeg *.png *.xbm *.bmp *.JPG *.JPEG *.PNG *.XBM *.BMP)"), this,
		tr("Select background image"), tr("Select the background image."));
		if (qsImageName.isNull())
			return NULL;

		QFileInfo fileInfo  (qsImageName);
		m_qsCurrentPath = fileInfo.filePath();

		// Here we create the Pixmap in the right scale
		QImage theImage (qsImageName);
		int iWidth, iHeight;
		float fZoom;
		// Keep aspect ratio
		iWidth  = m_rectCurrentObject.width();
		iHeight = (int)((float)theImage.height()/theImage.width()*m_rectCurrentObject.width());
		fZoom = ((float)m_rectCurrentObject.width() / theImage.width());
		// Convert the image and generate the Pixmap
		theImage = theImage.smoothScale (iWidth, iHeight);	//, QImage::ScaleMin);
		QPixmap thePixmap;
		thePixmap.convertFromImage(theImage);
		// Clear memory ...
		theImage = QImage ();
		// And here we adopt the rect the user has drawn.
		m_rectCurrentObject.setHeight(iHeight);
		// Finally we create the ImageObject
		pImageObject = new ImageObject (this);
		pImageObject->setRect (m_rectCurrentObject);
		pImageObject->setZoom (fZoom);	// We want to say Zoom = 1.0 even if the original size is already zoomed.
		pImageObject->setFile(qsImageName);
		pImageObject->setPixmap(thePixmap);

		m_listMenuObjects.append (pImageObject);
		emit (signalUpdateStructure());
		updatePixmap();
	}
	else
		pImageObject = new ImageObject (this);

	// And last we connect the signals
	connect (pImageObject, SIGNAL(signalDefineAsButton(MenuObject *)), this, SLOT(slotDefineAsButton(MenuObject *)));
	connect (pImageObject, SIGNAL(signalDeleteMe(MenuObject *)), this, SLOT(slotDeleteObject(MenuObject *)));
	connect (pImageObject, SIGNAL(signalModifyMe(MenuObject *)), this, SLOT(slotModifyObject(MenuObject *)));
	connect (pImageObject, SIGNAL(signalUpdateStructure()), this, SLOT(slotUpdateStructure()));
	connect (pImageObject, SIGNAL(signalUpdatePixmap()), this, SLOT(slotUpdatePixmap()));

	return pImageObject;
}

MenuObject *MenuPreview::createMovieObject(bool)
{
	QMessageBox::warning (NULL, tr ("Not Implemented yet."), tr ("createMovieObject : Not Implemented yet."),
		QMessageBox::Ok ,  QMessageBox::Cancel);
	updatePixmap();
	return NULL;
}

MenuObject *MenuPreview::createButtonObject(bool bShowDialog)
{
	ButtonObject *pButtonObject = new ButtonObject (this);

	if (bShowDialog)	{
		m_listMenuObjects.append (pButtonObject);
		emit (signalUpdateStructure());
		updatePixmap();
	}

	// And finally we connect everything ...
	connect (pButtonObject,   SIGNAL(signalDeleteMe(MenuObject *)), this, SLOT(slotDeleteObject(MenuObject *)));
//	connect (pNewSelected,    SIGNAL(signalDeleteMe(MenuObject *)), this, SLOT(slotDeleteObject(MenuObject *)));
//	connect (pNewHighlighted, SIGNAL(signalDeleteMe(MenuObject *)), this, SLOT(slotDeleteObject(MenuObject *)));
	connect (pButtonObject,   SIGNAL(signalUnbuttonMe(ButtonObject *)), this, SLOT(slotUnbutton(ButtonObject *)));
	connect (pButtonObject,   SIGNAL(signalCreateButtonDialog(ButtonObject *)), this, SLOT(slotCreateButtonDialog(ButtonObject *)));

	return pButtonObject;
}

MenuObject *MenuPreview::createObjectCollection(bool)
{
	QMessageBox::warning (NULL, tr ("Not Implemented yet."), tr ("createObjectCollection : Not Implemented yet."),
		QMessageBox::Ok ,  QMessageBox::Cancel);
	updatePixmap();
	return NULL;
}

void MenuPreview::slotDefineAsButton(MenuObject *pTheObject)
{
	// This is interesting ...
	// first we need to create a ButtonObject,
	// which we then furnish with three copies of this TextObject
	// Namely: Normal, Selected, and Highlighted ...
	m_listMenuObjects.remove (pTheObject);
	ButtonObject *pButtonObject = new ButtonObject (this);
	// Now we generate two copies ...
	MenuObject *pNewSelected, *pNewHighlighted;
	TextObject  textObject;
	FrameObject frameObject;

	QColor colorTransparent (TRANSPARENT_COLOR);
 	QColor colorHighlighted (START_HIGHLIGHTED_COLOR);
	QColor colorSelected    (START_SELECTED_COLOR);
	QColor colorSpare       (TRANSPARENT_COLOR);
	if (m_pPgcColors)	{
printf ("MenuPreview::slotDefineAsButton <Found Colors>\n");
		colorTransparent 	= m_pPgcColors[0];
 		colorHighlighted	= m_pPgcColors[1];
		colorSelected		= m_pPgcColors[2];
		colorSpare			= m_pPgcColors[3];
	}
else
printf ("MenuPreview::slotDefineAsButton <Did'nt find Colors>\n");
	pNewSelected = pNewHighlighted = NULL;
	if (pTheObject->objectType() == textObject.objectType())	{
		pNewSelected = pTheObject->clone();
		pNewHighlighted = pTheObject->clone();
		// Next we give some color differences for the different states (TextObjects)
		((TextObject *)(pNewSelected))   ->setForegroundColor (colorSelected);
		((TextObject *)(pNewHighlighted))->setForegroundColor (colorHighlighted);
		// Also here we ensure that the Textbackground for selected / highlighted is always transparent
		// The background color has already been put onto the actual background image.
		((TextObject *)(pNewSelected))   ->setBackgroundColor (colorTransparent);
		((TextObject *)(pNewHighlighted))->setBackgroundColor (colorTransparent);
	}
	else	{	// All but Text Objects get a frame for starters
		// Okay we want 1:1 for FrameObjects
		if (pTheObject->objectType() == frameObject.objectType())	{
			pNewSelected = pTheObject->clone();
			pNewHighlighted = pTheObject->clone();
		}
		else	{	// for ImageObject, and MovieObject a standard frame
			pNewSelected = new FrameObject;
			pNewHighlighted = new FrameObject;
			pNewSelected->setRect(pTheObject->rect());
			pNewHighlighted->setRect(pTheObject->rect());
			pNewSelected->setModifiers(*pTheObject->modifiers());
			pNewHighlighted->setModifiers(*pTheObject->modifiers());
			((FrameObject *)(pNewSelected))->setFrameWidth(10);
			((FrameObject *)(pNewHighlighted))->setFrameWidth(10);
		}
		// Next we give some color the the diffenrent states
		((FrameObject *)(pNewSelected))   ->setFrameColor (colorSelected);
		((FrameObject *)(pNewHighlighted))->setFrameColor (colorHighlighted);
	}
	pButtonObject->appendNormal (pTheObject);
	pButtonObject->appendSelected (pNewSelected);
	pButtonObject->appendHighlighted (pNewHighlighted);

	pButtonObject->setName(newButtonName());
	// Next we should give the button the same attributes as the NormalState ...
	// rect and boundingRect are taken care of in drawContent
	pButtonObject->setModifiers(*pTheObject->modifiers());

	// and here we append the button object as a new item in the MenuObject list.
	m_listMenuObjects.append(pButtonObject);
	m_pActiveObject = NULL;
	// Create the buttonDialog, so the user can change things around ...
	slotCreateButtonDialog(pButtonObject);
	if (m_pDialogButton)
		m_pDialogButton->setButtonCreation(true);
	// And finally we connect everything ...
	connect (pButtonObject,   SIGNAL(signalDeleteMe(MenuObject *)), this, SLOT(slotDeleteObject(MenuObject *)));
	connect (pNewSelected,    SIGNAL(signalDeleteMe(MenuObject *)), this, SLOT(slotDeleteObject(MenuObject *)));
	connect (pNewHighlighted, SIGNAL(signalDeleteMe(MenuObject *)), this, SLOT(slotDeleteObject(MenuObject *)));
	connect (pButtonObject,   SIGNAL(signalUnbuttonMe(ButtonObject *)),        this, SLOT(slotUnbutton(ButtonObject *)));
	connect (pButtonObject,   SIGNAL(signalCreateButtonDialog(ButtonObject *)),this, SLOT(slotCreateButtonDialog(ButtonObject *)));
}

void MenuPreview::slotCreateButtonDialog(ButtonObject *pButtonObject)
{
	uint t;
	// Here we generate a list of all Buttons in the current Menu. This is needed in the ButtonDialog
	// to  offer a list of possible targets.
	QStringList listMenuButtons;
	ButtonObject tempButton;
	for (t=0;t<m_listMenuObjects.count();t++)	{
		if (m_listMenuObjects.at(t)->objectType() == tempButton.objectType())	{
			listMenuButtons.append(m_listMenuObjects.at(t)->name());
		}
	}

	// Second step is to invoke the ButtonDialog ...
	m_pDialogButton = new DialogButton(this);
	m_pDialogButton->initMe(pButtonObject, this);
	m_pDialogButton->setMenuButtons (listMenuButtons);
	m_pDialogButton->show();
	connect (m_pDialogButton, SIGNAL(signalUpdateStructure()), this, SLOT(slotUpdateStructure()));
	connect (m_pDialogButton, SIGNAL(signalUpdatePixmap()),    this, SLOT(slotUpdatePixmap()));
	connect (m_pDialogButton, SIGNAL(destroyed()),             this, SLOT(slotDestroyedButtonDialog()));

	m_pActiveObject = NULL;
	// Go to DVDAuthor::slotRerquestSourceFIles.
	// Will return to respondSourceFiles() ...
	emit (signalRequestSourceFiles());
	// Will return to respondSubMenus() ...
	emit (signalRequestSubMenus());
	emit (signalUpdateStructure());
}

void MenuPreview::respondSourceFiles (QValueList<SourceFileEntry *>listSourceFileEntries)
{
	// If m_pActiveObject is set this means that we called for the SourceFileList from
	// readObject ...
	if (m_pActiveObject)	{
		ButtonObject *pCreatedButton = (ButtonObject *)m_pActiveObject;
		// Here we connect the Button, which was created in readObject
		uint t;
		for (t=0;t<listSourceFileEntries.count();t++)	{
//			QString qs = pCreatedButton->sourceDisplayName();
// printf ("MenuPreview::respondSourceFiles <%s>\n", (const char *)pCreatedButton->sourceDisplayName());
			if (listSourceFileEntries[t]->qsDisplayName == pCreatedButton->sourceDisplayName())	{
				pCreatedButton->setSourceFileEntry(listSourceFileEntries[t]);
				break;
			}
		}
		// And set this one to NULL
		m_pActiveObject = NULL;
	}
	// First check if this was triggered from slotCreateButtonDialog
	else if (m_pDialogButton)
		m_pDialogButton->setSourceFiles(listSourceFileEntries);
}

void MenuPreview::respondSubMenus (QStringList listSubMenus)
{
	// Okay, this is the return of the above emit(signalRequestSubMenus()) - call
	// Coming back from DVDMenu::slotRequestSubMenus()
	if (!m_pDialogButton)
		return;
	m_pDialogButton->setSubMenus(listSubMenus);
}

void MenuPreview::slotDestroyedButtonDialog()
{
	if (m_pDialogButton)
		delete m_pDialogButton;
	m_pDialogButton = NULL;
}

void MenuPreview::slotUpdateStructure()
{
	// simply reache this call on to CDVDMenu ...
	emit (signalUpdateStructure());
}

void MenuPreview::slotUpdatePixmap()
{
	updatePixmap();
}

void MenuPreview::slotDeleteObject(MenuObject *pObject)
{
	// simply do the same as the base class but also update the structureView ...
	ButtonPreview::slotDeleteObject(pObject);
	updatePixmap();
	emit (signalUpdateStructure());
}

// This is the slow version but 100% functional, I will correct this some time soon.
void MenuPreview::slotModifyObject(MenuObject *pObject)
{
	// This function is only used from the ImageObject so far.

	// Need to modify internal handling and eliminate the dependance of the
	// m_pImage->src - fileName.
	// Rather check if the Image is already in Memory.
	ImageObject *pImage = (ImageObject *)pObject;
	// The following struct will hold the modifier information ...
	ImagePreview *pPreview;		// The ImagePreview - class
	QString qsBackgroundImage;	// Null is okay ... for now.
	CXmlSlideshow::img_struct theImageStruct;
	ImageManipulator *pManipulator = new ImageManipulator;

	// Here we set the name of the Image
	theImageStruct.src = pImage->fileName();
	theImageStruct.pModifier = (void *)pManipulator;

	// Here we init the Manipulator values.
	*pManipulator = pImage->manipulator();
	// Note that we store the fRealZoom under fStartZoom in the ImagePreview.
	pManipulator->fZoom   = 1.0;
	pManipulator->iStartX = pImage->rect().x();
	pManipulator->iStartY = pImage->rect().y();
//	pManipulator->backgroundFileName = qsBackgroundImage;

	DialogImage imageDialog(this);
	// Before we initialize this we need to look at the modifiers of this ImageObject.

	imageDialog.initMe ((void *)&theImageStruct, qsBackgroundImage);
	pPreview = imageDialog.getPreview();
	// This is the StartZoom factor, we take every call to this Dialog at Zoom 1.0 in the GUI.
	pPreview->setStartZoom (pImage->zoom());
//	pPreview->setImage     (pImage->pixmap());
	if (!m_backgroundPixmap.isNull())	{
		// Here we create the background without the current object
		// (we want to move that one around right ?)
		QPixmap thePixmap(m_backgroundPixmap);
		QPainter thePainter(&thePixmap);
		for (uint t=0;t<m_listMenuObjects.count();t++)	{
			if (m_listMenuObjects.at(t) != pImage)
				m_listMenuObjects.at(t)->drawContents (&thePainter);
		}
		pPreview->setBackground(thePixmap, false);
	}
	pPreview->refreshPreview();
	if (imageDialog.exec() == QDialog::Rejected)
		return;
	// Here we recoup all information ...
	pManipulator = (ImageManipulator *)theImageStruct.pModifier;
	pImage->manipulator() = *pManipulator; 
	pImage->setPixmap (pPreview->getObject());
	QRect newRect = pPreview->getRect();
	// Next we need to compensate for the ImageObjects drawing offset (rect/2)
	pImage->setRect(newRect);
	updatePixmap();
}

/* Here we have the fast version.
 the difference is that the image is taken from pPreview->setImage(pImage->pixmap()); (think thumbnail),
 rather then from the pImage->fileName - original file.
 I could also eliminater the setStartZoom thingy ...

 Problem :
 The problem here however is the wrong location at start of the dialog and also the
 pixalation after a couple of zoom, new dialog, stretch, new dialog rotate etc.
 Solution :
 Need to implement a re-fresh of the image from the file after the dialog is done
 This would be a perfect background task which refreshes the pImage->pixmap() automatically after
 the next drawContents() - call
 Benefit :
 This would save the loading of the (probably big) pixmap from file and the re-sizing to the
 thumbnail size.
 It would speed up the creation of the ImageDialog
void MenuPreview::slotModifyObject(MenuObject *pObject)
{
	// This function is only used from the ImageObject so far.

	// Need to modify internal handling and eliminate the dependance of the
	// m_pImage->src - fileName.
	// Rather check if the Image is already in Memory.
	ImageObject *pImage = (ImageObject *)pObject;
	// The following struct will hold the modifier information ...
	ImagePreview *pPreview;		// The ImagePreview - class
	QString qsBackgroundImage;	// Null is okay ... for now.
	CXmlSlideshow::img_struct theImageStruct;
	ImageManipulator *pManipulator = new ImageManipulator;

	// Here we set the name of the Image
//	theImageStruct.src = pImage->fileName();
	theImageStruct.pModifier = (void *)pManipulator;

	// Here we init the Manipulator values.
	*pManipulator = pImage->manipulator();
	// Note that we store the fRealZoom under fStartZoom in the ImagePreview.
	pManipulator->fZoom   = 1.0;
	pManipulator->iStartX = pImage->rect().x();
	pManipulator->iStartY = pImage->rect().y();
//	pManipulator->backgroundFileName = qsBackgroundImage;

	DialogImage imageDialog(this);
	// Before we initialize this we need to look at the modifiers of this ImageObject.

	imageDialog.initMe ((void *)&theImageStruct, qsBackgroundImage);
	pPreview = imageDialog.getPreview();
	// This is the StartZoom factor, we take every call to this Dialog at Zoom 1.0 in the GUI.
//	pPreview->setStartZoom (pImage->zoom());
	pPreview->setImage     (pImage->pixmap());
	if (!m_backgroundPixmap.isNull())	{
		// Here we create the background without the current object
		// (we want to move that one around right ?)
		QPixmap thePixmap(m_backgroundPixmap);
		QPainter thePainter(&thePixmap);
		for (uint t=0;t<m_listMenuObjects.count();t++)	{
			if (m_listMenuObjects[t] != pImage)
				m_listMenuObjects[t]->drawContents (&thePainter);
		}
		pPreview->setBackground(thePixmap, false);
	}
	pPreview->refreshPreview();
	if (imageDialog.exec() == QDialog::Rejected)
		return;
	// Here we recoup all information ...
	pManipulator = (ImageManipulator *)theImageStruct.pModifier;
	pImage->manipulator() = *pManipulator;
	pImage->setPixmap (pPreview->getObject());
	QRect newRect = pPreview->getRect();
	// Next we need to compensate for the ImageObjects drawing offset (rect/2)
	pImage->setRect(newRect);
	updatePixmap();
}
*/

void MenuPreview::slotUnbutton(ButtonObject *pButton)
{
	uint t=0;
	MenuObject *pObject;
	m_listMenuObjects.remove(pButton);

	for (t=0;t<pButton->getNormalCount();t++)	{
		pObject = pButton->getNormal(t);
		m_listMenuObjects.append (pObject);
	}
	for (t=0;t<pButton->getHighlightedCount();t++)	{
		pObject = pButton->getHighlighted(t);
		m_listMenuObjects.append (pObject);
	}
	for (t=0;t<pButton->getSelectedCount();t++)	{
		pObject = pButton->getSelected(t);
		m_listMenuObjects.append (pObject);
	}
	delete pButton;
	emit (signalUpdateStructure());
}

void MenuPreview::slotDeleteMenu()
{
	// This function is called when the user wants to delete the current DVDMenu.
	// It passes the request right through to the QDVDAuthor - class
	emit (signalDeleteMe ());
}

void MenuPreview::slotRenameMenu()
{
	// This function is called when the user wants to change the name of this SubMenu.
	// It passes the request right through the the main QDVDAuthor class.
	emit (signalRenameMe());
}

QString &MenuPreview::newButtonName()
{
	// This function searches the existing Buttons for it's names and picks the first name which is not in use.
	uint t, iButtonNr;
	bool bContinue = true;
	iButtonNr = 2;
	ButtonObject tempButton;
	static QString qsButtonName;
	qsButtonName = QString ("Button 1");
	while (bContinue)	{
		bContinue = false;	// default is the assumption we won't find the same name, thus exit after ...
		for (t=0;t<m_listMenuObjects.count();t++)	{
			if (m_listMenuObjects.at(t)->objectType() == tempButton.objectType())	{
				if (m_listMenuObjects.at(t)->name () == qsButtonName)	{
					qsButtonName = QString ("Button %1").arg(iButtonNr++);
					bContinue = true;	// crap, got to do it again ...
					break;
				}
			}
		}
	}
	return qsButtonName;
}
