/****************************************************************************
** ImagePreview class
**
** Created: Tue Feb 02 22:06:51 2004
**      by: Varol Okan using Kate
**
** This class is the encapsulation of the ImagePreview from the FileSelection
** Dialog.
** It is also used in the ImageDialog.
**
****************************************************************************/

#include <qpen.h>
#include <qfont.h>
#include <qpainter.h>
#include <qpopupmenu.h>
#include <qlistview.h>
#include <qwmatrix.h>
#include <qmessagebox.h>

#include "global.h"
#include "xml_dvd.h"
#include "dialogtextfont.h"
#include "dialogmatrix.h"
#include "menuobject.h"
#include "textobject.h"	// derived from menuobject

TextObject::TextObject (QWidget *pParent)
	: MenuObject (pParent)
{
	m_qsObjectType = QString (TEXT_OBJECT);
	m_colorBackground = QColor(TRANSPARENT_COLOR);	// indicate the transparent color.
}

TextObject::~TextObject ()
{
}

void TextObject::setFont (QFont &theFont)
{
	m_Font = theFont;
}

QFont &TextObject::getFont ()
{
	return m_Font;
}

void TextObject::setText (QString &theText)
{
	m_qsText = theText;
}

QString &TextObject::getText ()
{
	return m_qsText;
}

void TextObject::setTextAlign (QPoint &newAlign)
{
	m_ptTextAlign = newAlign;
}

QPoint &TextObject::getTextAlign ()
{
	return m_ptTextAlign;
}

void TextObject::replaceColor (QColor theColor, QColor oldColor)
{
	if (getForegroundColor () == oldColor)
		setForegroundColor (theColor);
	if (getBackgroundColor () == oldColor)
		setBackgroundColor (theColor);
}

void TextObject::setForegroundColor (QColor &theColor)
{
	m_colorForeground = theColor;
}

void TextObject::setBackgroundColor (QColor &theColor)
{
	m_colorBackground = theColor;
}

QColor &TextObject::getForegroundColor ()
{
	return m_colorForeground;
}

QColor &TextObject::getBackgroundColor ()
{
	return m_colorBackground;
}


void TextObject::drawContents (QPainter *painter)
{
	QWMatrix theMatrix;// = matrix();
	// Here we calculate the center point of gravity (rotation)
	QPoint centerPos;
	centerPos.setX (rect().x() + (int)(rect().width () / 2.0));
	centerPos.setY (rect().y() + (int)(rect().height() / 2.0));
	// Here we define the cenetered rect.
	QRect theRect ((int)-(rect().width()/ 2.0), (int)-(rect().height()/2.0), rect().width(), rect().height());

	// From the Qt documentation ...
	// This transformation engine is a three-step pipeline
	// The first step uses the world transformation matrix.
	// The second step uses the window.
	// The third step uses the viewport.

	// First we translate to the appropriate location,
	theMatrix.translate ((double)centerPos.x(), (double)centerPos.y());

	//. then we apply the other modifications ...
	theMatrix.scale  (modifiers()->fScaleX, modifiers()->fScaleY);
	theMatrix.rotate (modifiers()->fRotate);
	theMatrix.shear  (modifiers()->fShearX, modifiers()->fShearY);
	
	QPen thePen (m_colorForeground, 2, Qt::DashDotLine);
	painter->setPen(thePen);
	
	// Here we get the rect that encompasses the button for this object.
	m_boundingRect = theMatrix.mapRect(theRect);
	if (m_bDrawRect)
		painter->drawRect(m_boundingRect);
	//  then we actually set the world matrix 
	painter->setWorldMatrix(theMatrix);

	// and continue to draw the contents ...
	if (m_colorBackground != QColor(TRANSPARENT_COLOR))	{
		// No frame wanted, we take care of that ourself if requested ...
		painter->setPen(Qt::NoPen);
		painter->setBrush (QBrush(m_colorBackground));
		painter->drawRect(theRect);
		// and re-set the PEN.
		painter->setPen(thePen);
	}
	else
		painter->setBrush(Qt::NoBrush);
 
	m_Font.setStyleStrategy (QFont::NoAntialias);
	painter->setFont(m_Font);
	QRect boundingRect;

	// Okay, I abused a QPoint class to hold some other infos ... so what ???
	int iFlags = Qt::DontClip|Qt::WordBreak|m_ptTextAlign.x()|m_ptTextAlign.y();

	if (m_bDrawRect)
		painter->drawRect(theRect);

	painter->drawText(theRect, iFlags, m_qsText, -1, &boundingRect);
	// Here we expand the rectangle if needs be, and update again, to re-draw.
	if (m_rect.width() < boundingRect.width())	{
		m_rect.setWidth  (boundingRect.width());
		m_pParent->update ();
	}
	if (m_rect.height() < boundingRect.height())	{
		m_rect.setHeight (boundingRect.height());
		m_pParent->update ();
	}

	// Here we re-set the world matrix and translate the newly drawn ... thing ...
	theMatrix.reset();
	painter->setWorldMatrix(theMatrix);
}

/*

void TextObject::drawContents (QPainter *painter)
{
	QWMatrix theMatrix;// = matrix();
	// Here we calculate the center point of gravity (rotation)
	QPoint centerPos;
	centerPos.setX (rect().x() + (int)(rect().width () / 2.0));
	centerPos.setY (rect().y() + (int)(rect().height() / 2.0));
	// Here we define the cenetered rect.
	QRect theRect ((int)-(rect().width()/ 2.0), (int)-(rect().height()/2.0), rect().width(), rect().height());

	// From the Qt documentation ...
	// This transformation engine is a three-step pipeline
	// The first step uses the world transformation matrix.
	// The second step uses the window.
	// The third step uses the viewport.

	// First we translate to the appropriate location,
	theMatrix.translate ((double)centerPos.x(), (double)centerPos.y());

	//. then we apply the other modifications ...
	theMatrix.scale  (modifiers()->fScaleX, modifiers()->fScaleY);
	theMatrix.rotate (modifiers()->fRotate);
	theMatrix.shear  (modifiers()->fShearX, modifiers()->fShearY);
	painter->setWorldMatrix(theMatrix);
	// Here we get the rect that encompasses the button for this object.
	m_boundingRect = theMatrix.mapRect(theRect);

	QPen thePen (m_colorForeground, 2, Qt::DashDotLine);
	if (m_colorBackground != QColor(TRANSPARENT_COLOR))	{
		painter->setPen(Qt::NoPen);
		painter->setBrush (QBrush(m_colorBackground));
		painter->drawRect(theRect);
	}
	else
		painter->setBrush(Qt::NoBrush);

	painter->setPen(thePen);
	painter->setFont(m_Font);
	QRect boundingRect;

	int iFlags = Qt::DontClip|Qt::WordBreak|m_ptTextAlign.x()|m_ptTextAlign.y();

	if (m_bDrawRect)
		painter->drawRect(theRect);

	painter->drawText(theRect, iFlags, m_qsText, -1, &boundingRect);

	if (m_rect.width()  < boundingRect.width())	{
		m_rect.setWidth  (boundingRect.width());
		m_pParent->update ();
	}
	if (m_rect.height() < boundingRect.height())	{
		m_rect.setHeight (boundingRect.height());
		m_pParent->update ();
	}
	// Here we re-set the world matrix and translate the newly drawn ... thing ...
	theMatrix.reset();
	painter->setWorldMatrix(theMatrix);
	// for now to see what is going on ...
	painter->setBrush(Qt::NoBrush);
	painter->setPen(thePen);
	if (m_bDrawRect)
		painter->drawRect(m_boundingRect);
}

*/

bool TextObject::readProjectFile  (QDomNode &theNode)
{
	QDomElement theElement = theNode.toElement();
	QDomAttr attribute;

	attribute = theElement.attributeNode ( TEXT_OBJECT_TEXT );
	if (!attribute.isNull())
		m_qsText = attribute.value();

	attribute = theElement.attributeNode ( TEXT_OBJECT_TEXT_ALIGN );
	if (!attribute.isNull())	{
		// QString "x,y" - we abused a QPoint object for the Align info.
		QStringList list = QStringList::split (",", attribute.value());
		QPoint textAlign;
		if (list.count() == 2)
			textAlign = QPoint(list[0].toInt(), list[1].toInt());
		setTextAlign (textAlign);
	}

	attribute = theElement.attributeNode ( TEXT_OBJECT_FONT);
	if (!attribute.isNull())	{
		QFont theFont;
		if (theFont.fromString (attribute.value()))
			setFont (theFont);
	}

	attribute = theElement.attributeNode ( TEXT_OBJECT_BACKGROUND_COLOR );
	if (!attribute.isNull())	{
		QColor backgroundColor (attribute.value());
		setBackgroundColor (backgroundColor);
	}

	attribute = theElement.attributeNode ( TEXT_OBJECT_FOREGROUND_COLOR );
	if (!attribute.isNull())	{
		QColor foregroundColor (attribute.value());
		setForegroundColor (foregroundColor);
	}

	MenuObject::readProjectFile( theNode );
	return true;
}

bool TextObject::writeProjectFile (QDomElement &theElement)
{
	QDomDocument xmlDoc = theElement.ownerDocument();
	QDomElement textNode = xmlDoc.createElement( TEXT_OBJECT );	// <FrameObject>
	
	TextObject initialValues;
	// Here we set the attributes of the <dvdauthor> tag
	if (!getText().isNull())
		textNode.setAttribute( TEXT_OBJECT_TEXT, m_qsText );
	if (getTextAlign() != initialValues.getTextAlign())
		textNode.setAttribute( TEXT_OBJECT_TEXT_ALIGN, QString("%1,%2").arg(getTextAlign().x()).arg(getTextAlign().y()) );
	if (getFont() != initialValues.getFont())
		textNode.setAttribute( TEXT_OBJECT_FONT, getFont().toString() );
	if (getBackgroundColor() != initialValues.getBackgroundColor())
		textNode.setAttribute( TEXT_OBJECT_BACKGROUND_COLOR, getBackgroundColor().name() );
	if (getForegroundColor() != initialValues.getForegroundColor())
		textNode.setAttribute( TEXT_OBJECT_FOREGROUND_COLOR, getForegroundColor().name() );
	
	theElement.appendChild( textNode );
	// And here we write the base class ...
	return MenuObject::writeProjectFile( textNode );
}

void TextObject::createStructure (QListViewItem *pParentItem)
{
	QListViewItem *pButtonItem;
	QListViewItem *pModifierItem;

	QString qsName = tr ("Text Object");
	pButtonItem = new QListViewItem(pParentItem, qsName, m_qsText);
	pButtonItem->setExpandable (TRUE);

	// And finally we create the modifiers info ...
	if ( !((m_modifiers.fRotate == 0.0) && (m_modifiers.fShearX == 0.0) && (m_modifiers.fShearY == 0.0) &&
		   (m_modifiers.fZoom   == 1.0) && (m_modifiers.fScaleX == 1.0) && (m_modifiers.fScaleY == 1.0)) )	{
		pModifierItem = new QListViewItem(pButtonItem, tr ("Modifiers"), NULL);
		pButtonItem->setExpandable (TRUE);

		new QListViewItem(pModifierItem, tr ("Shear (x, y)"),
			QString ("%1, %2").arg(m_modifiers.fShearX).arg(m_modifiers.fShearY));
		new QListViewItem(pModifierItem, tr ("Zoom"),
			QString ("%1").arg(m_modifiers.fZoom));
		new QListViewItem(pModifierItem, tr ("Rotate"),
			QString ("%1 deg").arg(m_modifiers.fRotate));
	}

	if (m_colorBackground == QColor (TRANSPARENT_COLOR))
		new QListViewItem(pButtonItem, tr ("Background"), tr ("Transparent"));
	else
		new QListViewItem(pButtonItem, tr ("Background"),
			tr ("color(%1, %2, %3)").
				arg(m_colorBackground.red()).
				arg(m_colorBackground.green()).
				arg(m_colorBackground.blue()));

	new QListViewItem(pButtonItem, tr ("Foreground"),
		QString ("color(%1, %2, %3)").
			arg(m_colorForeground.red()).
			arg(m_colorForeground.green()).
			arg(m_colorForeground.blue()));

	new QListViewItem(pButtonItem, tr ("Geometry"),
		QString ("%1, %2, %3, %4").
		arg(rect().x()).arg(rect().y()).arg(rect().width()).arg(rect().height()));
}

bool TextObject::mouseMoveEvent (QMouseEvent *pEvent)
{
	// This will move the textObject around and around and around we go ...
	QRect theRect = rect();
	int iX, iY, iWidth, iHeight;
	iWidth  = rect().width ();
	iHeight = rect().height();
	iX = rect().x() - (m_currentMousePos.x() - pEvent->pos().x());
	iY = rect().y() - (m_currentMousePos.y() - pEvent->pos().y());
	theRect.setX(iX);
	theRect.setY(iY);
	theRect.setWidth(iWidth);
	theRect.setHeight(iHeight);

	setRect(theRect);
	m_currentMousePos = pEvent->pos();
	return false;
}

bool TextObject::mousePressEvent (QMouseEvent *pEvent)
{
	m_currentMousePos = pEvent->pos();
	if (pEvent->button() == Qt::RightButton)	{
		QPopupMenu *pMenu = new QPopupMenu(m_pParent);
		pMenu->insertItem ( tr ("Edit ...")  , this, SLOT(slotEdit()));
		pMenu->insertItem ( tr ("Matrix ..."), this, SLOT(slotMatrix()));
		pMenu->insertItem ( tr ("Delete")    , this, SLOT(slotDelete()));
		pMenu->insertSeparator();
		pMenu->insertItem ( tr ("Add Frame"), this, SLOT(slotAddFrame()));
		pMenu->insertItem ( tr ("Add Image"), this, SLOT(slotAddImage()));
		pMenu->insertItem ( tr ("Add Movie"), this, SLOT(slotAddMovie()));
		pMenu->insertSeparator();
		pMenu->insertItem ( tr ("Define as Button") , this, SLOT(slotDefineAsButton()));
		int iReturn = pMenu->exec(pEvent->globalPos());
		// Here we mark that the user called a menu item thus we don't want the
		// base classes to continue with the mouse event
		if (iReturn != -1)
			return true;
	}
	else
		m_bDrawRect = true;
	return false;
}

bool TextObject::mouseReleaseEvent (QMouseEvent *)
{
	m_bDrawRect = false;
	emit (signalUpdatePixmap());
	emit (signalUpdateStructure());
	return false;
}

bool TextObject::mouseDoubleClickEvent (QMouseEvent *)
{
	QFont theFont;
	DialogTextFont fontDialog (m_pParent);
	fontDialog.setFont (getFont());
	fontDialog.setText (getText());
	fontDialog.setTextAlign (getTextAlign());
	fontDialog.setBackgroundColor (getBackgroundColor());
	fontDialog.setForegroundColor (getForegroundColor());

	if (fontDialog.exec() == QDialog::Rejected)
		return false;

	setFont(fontDialog.getFont());
	QString theString = fontDialog.getText();
	setText(theString);
	if (fontDialog.getBackgroundColor() != getBackgroundColor())	{
		// This emit signal is only caught when in ButtonDialog and the Selected, or Highlighted mask 
		// was changed (I.e. This Object is in one of those two frames).
		// Otherwise the signal is ignored.
		emit (signalMaskColorChanged(fontDialog.getBackgroundColor(), getBackgroundColor()));
		// P.s. replaceColor (QColor, QColor) of all ButtonObjects will be called, inclusive this object)
	}
	setBackgroundColor (fontDialog.getBackgroundColor());
	if (fontDialog.getForegroundColor() != getForegroundColor())	{
		// This emit signal is only caught when in ButtonDialog and the Selected, or Highlighted mask 
		// was changed (I.e. This Object is in one of those two frames).
		// Otherwise the signal is ignored.
		emit (signalMaskColorChanged(fontDialog.getForegroundColor(), getForegroundColor()));
		// P.s. replaceColor (QColor, QColor) of all ButtonObjects will be called, inclusive this object)
	}
	setForegroundColor (fontDialog.getForegroundColor());
	setTextAlign(fontDialog.getTextAlign());
	if (fontDialog.getFit())	{
		rect().setWidth (1);
		rect().setHeight(1);
	}

	m_pParent->update ();
	emit (signalUpdatePixmap());
	emit (signalUpdateStructure());
	return false;
}

void TextObject::slotEdit ()
{
	mouseDoubleClickEvent(NULL);
}

void TextObject::slotMatrix()
{
	DialogMatrix *pDialog = new DialogMatrix(MenuObject::parent());
	pDialog->initMe(this);
	pDialog->show();
}

void TextObject::slotDelete ()
{
	emit (signalDeleteMe(this));
}

void TextObject::slotAddFrame ()
{
	QMessageBox::warning (NULL, tr ("Not Implemented yet."), tr ("Not Implemented yet."),
		QMessageBox::Ok ,  QMessageBox::Cancel);
}

void TextObject::slotAddImage ()
{
	QMessageBox::warning (NULL, tr ("Not Implemented yet."), tr ("Not Implemented yet."),
		QMessageBox::Ok ,  QMessageBox::Cancel);
}

void TextObject::slotAddMovie ()
{
	QMessageBox::warning (NULL, tr ("Not Implemented yet."), tr ("Not Implemented yet."),
		QMessageBox::Ok ,  QMessageBox::Cancel);
}

void TextObject::slotDefineAsButton ()
{
	// We want to convert this TextObject into a ButtonObject.
	// Thus the class TextObject is no longer the logical place for doing this, now is it ?
	// Okay we call the MenuPreview and let this handle the task.
	emit (signalDefineAsButton(this));
}

MenuObject *TextObject::clone()
{
	TextObject *pNewObject = new TextObject (MenuObject::parent());
	pNewObject->setFont (getFont());
	pNewObject->setText (getText());
	pNewObject->setTextAlign (getTextAlign());
	pNewObject->setForegroundColor (getForegroundColor());
	pNewObject->setBackgroundColor (getBackgroundColor());
	// The following two are not really needed, since we get those 
	// informations solely from the Normal State - objects ...
	pNewObject->setRect(rect());
	pNewObject->setModifiers(*modifiers());

	return pNewObject;
}

