/****************************************************************************
** ShadowObject class
**
** Created: Tue Aug 08 22:06:51 2006
**      by: Varol Okan using XEmacs
**
** This class 
**
****************************************************************************/

#include <math.h>
#include <qpainter.h>

#include "global.h"
#include "xml_dvd.h"
#include "menuobject.h"
#include "textobject.h"
#include "imageobject.h"
#include "movieobject.h"
#include "frameobject.h"
#include "kimageeffect.h"
#include "structuretoolbar.h"

#include "shadowobject.h"

#define qMax(a,b) (a>b)?a:b
#define qMin(a,b) (a<b)?a:b

// Exponential blur, Jani Huhtanen, 2006
//
template<int aprec, int zprec>
static inline void blurinner(unsigned char *bptr, int &zR, int &zG, int &zB, int &zA, int alpha);

template<int aprec,int zprec>
static inline void blurrow( QImage & im, int line, int alpha);

template<int aprec, int zprec>
static inline void blurcol( QImage & im, int col, int alpha);

/*
*  expblur(QImage &img, int radius)
*
*  In-place blur of image 'img' with kernel
*  of approximate radius 'radius'.
*
*  Blurs with two sided exponential impulse
*  response.
*
*  aprec = precision of alpha parameter 
*  in fixed-point format 0.aprec
*
*  zprec = precision of state parameters
*  zR,zG,zB and zA in fp format 8.zprec
*/
template<int aprec,int zprec>
void expblur( QImage &img, int radius )
{
  if(radius<1)
    return;

  /* Calculate the alpha such that 90% of 
     the kernel is within the radius.
     (Kernel extends to infinity) 
  */
  int alpha = (int)((1<<aprec)*(1.0f-expf(-2.3f/(radius+1.f))));

  for(int row=0;row<img.height();row++)
  {
    blurrow<aprec,zprec>(img,row,alpha);
  }

  for(int col=0;col<img.width();col++)
  {
    blurcol<aprec,zprec>(img,col,alpha);
  }
  return;
}

template<int aprec, int zprec>
static inline void blurinner(unsigned char *bptr, int &zR, int &zG, int &zB, int &zA, int alpha)
{
  int R,G,B,A;
  R = *bptr;
  G = *(bptr+1);
  B = *(bptr+2);
  A = *(bptr+3);

  zR += (alpha * ((R<<zprec)-zR))>>aprec;
  zG += (alpha * ((G<<zprec)-zG))>>aprec;
  zB += (alpha * ((B<<zprec)-zB))>>aprec;
  zA += (alpha * ((A<<zprec)-zA))>>aprec;

  *bptr =     zR>>zprec;
  *(bptr+1) = zG>>zprec;
  *(bptr+2) = zB>>zprec;
  *(bptr+3) = zA>>zprec;
}

template<int aprec,int zprec>
static inline void blurrow( QImage & im, int line, int alpha)
{
  int zR,zG,zB,zA;

  QRgb *ptr = (QRgb *)im.scanLine(line);

  zR = *((unsigned char *)ptr    )<<zprec;
  zG = *((unsigned char *)ptr + 1)<<zprec;
  zB = *((unsigned char *)ptr + 2)<<zprec;
  zA = *((unsigned char *)ptr + 3)<<zprec;

  for(int index=1; index<im.width(); index++)
  {
    blurinner<aprec,zprec>((unsigned char *)&ptr[index],zR,zG,zB,zA,alpha);
  }
  for(int index=im.width()-2; index>=0; index--)
  {
    blurinner<aprec,zprec>((unsigned char *)&ptr[index],zR,zG,zB,zA,alpha);
  }
}

template<int aprec, int zprec>
static inline void blurcol( QImage & im, int col, int alpha)
{
  int zR,zG,zB,zA;

  QRgb *ptr = (QRgb *)im.bits();
  ptr+=col;

  zR = *((unsigned char *)ptr    )<<zprec;
  zG = *((unsigned char *)ptr + 1)<<zprec;
  zB = *((unsigned char *)ptr + 2)<<zprec;
  zA = *((unsigned char *)ptr + 3)<<zprec;

  for(int index=im.width(); index<(im.height()-1)*im.width(); index+=im.width())
  {
    blurinner<aprec,zprec>((unsigned char *)&ptr[index],zR,zG,zB,zA,alpha);
  }

  for(int index=(im.height()-2)*im.width(); index>=0; index-=im.width())
  {
    blurinner<aprec,zprec>((unsigned char *)&ptr[index],zR,zG,zB,zA,alpha);
  }

}

ShadowObject::ShadowObject ( MenuObject *pMenuObject, QWidget *pParent ) 
  : ImageObject  ( pParent ), 
    m_fSunElevation (45.0f)
{
  //  m_pShadowImage  = NULL;
  m_pMenuObject   = pMenuObject;
  m_iDeltaX       =   -1;
  m_iDeltaY       =   -1;
  m_fSunAngle     =  0.0;
  m_fDistance     = -1.0;
  m_fTransparency =  0.0;
  m_iBlurRadius   =    3;
  m_enType        = TYPE_SHADOW; //TYPE_MIRROR; // TYPE_SHADOW;
}

ShadowObject::~ShadowObject ( ) 
{

}

ShadowObject &ShadowObject::operator = ( ShadowObject &theOther )
{
  setColor        ( theOther.color        ( ) );
  setTransparency ( theOther.transparency ( ) );
  setDistance     ( theOther.distance     ( ) );
  setBlur         ( theOther.blur         ( ) );
  setSunAngle     ( theOther.sunAngle     ( ) );

  return *this;
}

void ShadowObject::drawContents ( QPainter *pPainter )
{
  if ( ( m_fDistance <= 0.0 ) || ( m_fTransparency >= 1.0 ) )
    return;

  if ( m_imageShadow.isNull ( ) )
    updateShadow ( );

  if ( ! m_pMenuObject )
    return;

  QRect objectRect = m_pMenuObject->rect ( );

  QWMatrix theMatrix;
  //ImageManipulator *pManipulator = ;
  Modifiers *pModifiers = m_pMenuObject->modifiers ( );
  // Here we calculate the center point of gravity (rotation)
  QPoint centerPos;
  centerPos.setX ( objectRect.x ( ) + (int) ( (float) objectRect.width  ( ) / 2.0 ) );
  centerPos.setY ( objectRect.y ( ) + (int) ( (float) objectRect.height ( ) / 2.0 ) );
  // Here we define the cenetered rect.
  QRect theRect ((int)-( objectRect.width ( ) / 2.0), (int)-( objectRect.height ( ) / 2.0 ), objectRect.width ( ), objectRect.height ( ) );
  
  // From the Qt documentation ...
  // This transformation engine is a three-step pipeline
  // The first step uses the world transformation matrix.
  // The second step uses the window.
  // The third step uses the viewport.
  
  // First we translate to the appropriate location,
  theMatrix.translate ( (double)centerPos.x ( ), (double)centerPos.y ( ) );
  
  // then we apply the other modifications ...
  theMatrix.scale  ( pModifiers->fScaleX, pModifiers->fScaleY );
  theMatrix.shear  ( pModifiers->fShearX, pModifiers->fShearY );
  theMatrix.rotate ( pModifiers->fRotate );

  pPainter->setWorldMatrix ( theMatrix );
  pPainter->drawImage ( theRect.x ( ) + deltaX ( ), theRect.y ( ) + deltaY ( ), m_imageShadow );

  theMatrix.reset ( );
  pPainter->setWorldMatrix ( theMatrix );
}

void ShadowObject::drawContents ( QPainter *pPainter, int iRenderFrameNumber, int )
{
  // Next is to set the AnimationAttributes
  for ( uint t=0; t<m_listAnimationAttributes.count ( ) ;t++ )
    m_listAnimationAttributes[t]->setValue ( iRenderFrameNumber );

  // And then we ought to refresh the pixmap ...
  if ( m_listAnimationAttributes.count () > 0 )
    updateShadow ( );
  
  drawContents ( pPainter );
}

bool ShadowObject::mousePressEvent ( QMouseEvent   * )
{
  return false;
}

bool ShadowObject::mouseReleaseEvent ( QMouseEvent   * )
{
  return false;
}

bool ShadowObject::mouseDoubleClickEvent ( QMouseEvent   * )
{
  return false;
}

void ShadowObject::createStructure ( QListViewItem *pParentItem )
{
  StructureItem *pShadowItem;
  
  QString qsName  = tr("Shadow Object");
  QString qsInfo ( tr("Shadow dist=%1 blur=%2 angle=%3 transparency=%4").
		   arg ( distance ( ) ).arg ( blur         ( ) ).
		   arg ( sunAngle ( ) ).arg ( transparency ( ) ) );
  
  pShadowItem = new StructureItem(this, pParentItem, qsName, qsInfo);
  pShadowItem->setExpandable (TRUE);
  
  new StructureItem(this, pShadowItem, tr ("Shadow Color"),
		    tr ("color(%1, %2, %3)").
		    arg ( m_color.red   ( ) ).
		    arg ( m_color.green ( ) ).
		    arg ( m_color.blue  ( ) ) );
  
  new StructureItem(this, pShadowItem, tr ("Transparency"), QString ("%1").arg ( transparency ( ) ) );
  new StructureItem(this, pShadowItem, tr ("Blur Radius"),  QString ("%1").arg ( blur     ( ) ) );
  new StructureItem(this, pShadowItem, tr ("Distance"),     QString ("%1").arg ( distance ( ) ) );
  new StructureItem(this, pShadowItem, tr ("Sun Angle"),    QString ("%1").arg ( sunAngle ( ) ) );
}

bool ShadowObject::createContextMenu ( QPoint   )
{
  return false;
}

AnimationAttribute *ShadowObject::getSpecificAttributes (long, QString)
{
  return NULL;
}

void ShadowObject::updateShadow ( )
{
  if ( ! m_pMenuObject )
    return;

  float fSunAngle = m_fSunAngle / 180.0 * PI;

  m_iDeltaX = (int) ( sin ( 45.0 / 180.0 * PI ) * m_fDistance * cos ( fSunAngle ) );
  m_iDeltaY = (int) ( sin ( 45.0 / 180.0 * PI ) * m_fDistance * sin ( fSunAngle ) );

  // Next we create the image
  int         iFrameWidth = 0;
  QString     emptyAnimation;
  Modifiers   emptyModifier;
  TextObject  tempText;
  FrameObject tempFrame;
  MenuObject *pMenuObject = NULL;
  QColor      backgroundColor (   1,   1,   1 );
  QColor      foregroundColor ( 255, 255, 255 );

  if ( m_pMenuObject->objectType ( ) == tempText.objectType ( ) ) {
    pMenuObject = m_pMenuObject->clone ( );
    pMenuObject->setShadow ( NULL );
    // Remember that the modifiers will be part of the Imagebject
    pMenuObject->setModifiers ( emptyModifier  );
    pMenuObject->setAnimation ( emptyAnimation );
    if ( m_enType != TYPE_MIRROR ) {
      TextObject *pText = (TextObject *)pMenuObject;
      pText->setForegroundColor ( foregroundColor );
      pText->setBackgroundColor ( backgroundColor );
    }
  }
  else if ( m_pMenuObject->objectType  ( ) == tempFrame.objectType ( ) ) {
    pMenuObject = m_pMenuObject->clone ( );
    pMenuObject->setShadow ( NULL );
    // Remember that the modifiers will be part of the Imagebject
    pMenuObject->setModifiers ( emptyModifier  );
    pMenuObject->setAnimation ( emptyAnimation );
    FrameObject *pFrame = (FrameObject *)pMenuObject;
     iFrameWidth = pFrame->width ( );
    m_iDeltaX = m_iDeltaX - (int)( iFrameWidth / 2.0 );
    m_iDeltaY = m_iDeltaY - (int)( iFrameWidth / 2.0 );
    if ( m_enType != TYPE_MIRROR ) 
      pFrame->setFrameColor ( foregroundColor );
  }
  else {
    // ImageObject, MoveObject
    // Otherwise we simply want to draw a rect in the right color / size
    if ( m_enType == TYPE_SHADOW ) {
      TextObject *pText = new TextObject;
      pMenuObject = pText;
      pText->setRect ( m_pMenuObject->rect ( ) );
      pText->setBackgroundColor ( foregroundColor );
    }
    else if ( m_enType == TYPE_MIRROR ) {
      pMenuObject = m_pMenuObject->clone ( );
      pMenuObject->setShadow ( NULL ); 
      // Remember that the modifiers will be part of the Imagebject
      pMenuObject->setModifiers ( emptyModifier  );
      pMenuObject->setAnimation ( emptyAnimation );
    }
  }

  if ( m_enType == TYPE_SHADOW )
    createShadow ( pMenuObject, iFrameWidth );
  else if ( m_enType == TYPE_REFLECTION )
    createShadow ( pMenuObject, iFrameWidth );
  else if ( m_enType == TYPE_MIRROR )
    createMirror ( pMenuObject, iFrameWidth );

  delete pMenuObject;

}

void ShadowObject::createMirror ( MenuObject *pMenuObject, int iFrameWidth )
{
  QColor backgroundColor (   1,   1,   1 );
  QRect r = m_pMenuObject->rect ( );
  // Now that we have the object to draw we should create the QImage - object ...
  QImage theImage;
  {
    int iOffset1 = (int) ( iFrameWidth/2.0 + m_iBlurRadius + 10 );
    int iOffset2 = (int) ( iFrameWidth + 2.0*m_iBlurRadius + 10 );
    QRect    theRect     ( iOffset1, iOffset1, r.width ( ), r.height ( ) );
    QPixmap  thePix      ( r.width ( ) + iOffset2, r.height ( ) + iOffset2 );
    QPainter thePainter  ( &thePix );

    thePainter.fillRect  ( thePix.rect ( ), backgroundColor );
    pMenuObject->setRect ( theRect );
    pMenuObject->drawContents ( &thePainter );
    // At this point we should have thePix as we want/need with a shadowee backdrop
    theImage = thePix.convertToImage ( );
    theImage.setAlphaBuffer ( true );
  }
  
  // Lastly we should filter out color keys if present (MovieObject, and Imagebject only)
  filterColorKeys ( theImage );

  // At this point we have the frame/text/rect and need to blur this into a semi transparent image
  int x, y;
  QRgb *pLine, currentPixel, newPixel;
  int iRed, iGreen, iBlue, iAlpha;
  m_imageShadow = theImage;

  // What we have now is a greyscale image. The values actually represent the 
  // transparency of the final shadow.
  // Thus we will create an image that has only one color and only modulate the 
  // transparency.
  iRed   = m_color.red   ( );
  iGreen = m_color.green ( );
  iBlue  = m_color.blue  ( );
  iAlpha = (int)( (1.0 - m_fTransparency) * 255 );
  for ( y=0;  y < m_imageShadow.height ( ); y++ ) {
    pLine = (QRgb *) m_imageShadow.scanLine ( y );
    for ( x=0; x < m_imageShadow.width ( ); x++ ) {
      currentPixel = *(pLine + x);
      newPixel = currentPixel | 0x10000000; //( iAlpha << 24 );
      *(pLine + x) = newPixel;
    }
  }
}

void ShadowObject::createShadow ( MenuObject *pMenuObject, int iFrameWidth )
{
  QColor backgroundColor (   1,   1,   1 );
  QRect r = m_pMenuObject->rect ( );
  // Now that we have the object to draw we should create the QImage - object ...
  QImage theImage;
  {
    int iOffset1 = (int) ( iFrameWidth/2.0 + m_iBlurRadius + 10 );
    int iOffset2 = (int) ( iFrameWidth + 2.0*m_iBlurRadius + 10 );
    QRect    theRect     ( iOffset1, iOffset1, r.width ( ), r.height ( ) );
    QPixmap  thePix      ( r.width ( ) + iOffset2, r.height ( ) + iOffset2 );
    QPainter thePainter  ( &thePix );

    thePainter.fillRect  ( thePix.rect ( ), backgroundColor );
    pMenuObject->setRect ( theRect );
    pMenuObject->drawContents ( &thePainter );
    // At this point we should have thePix as we want/need with a shadowee backdrop
    theImage = thePix.convertToImage ( );
    theImage.setAlphaBuffer ( true );
  }
  
  // Lastly we should filter out color keys if present (MovieObject, and Imagebject only)
  filterColorKeys ( theImage );

  // At this point we have the frame/text/rect and need to blur this into a semi transparent image
  int x, y;
  QRgb *pLine, currentPixel, newPixel;
  int iRed, iGreen, iBlue, iAlpha;
  m_imageShadow = theImage;

  // What we have now is a greyscale image. The values actually represent the 
  // transparency of the final shadow.
  // Thus we will create an image that has only one color and only modulate the 
  // transparency.
  iRed   = m_color.red   ( );
  iGreen = m_color.green ( );
  iBlue  = m_color.blue  ( );
  for ( y=0;  y < m_imageShadow.height ( ); y++ ) {
    pLine = (QRgb *) m_imageShadow.scanLine ( y );
    for ( x=0; x < m_imageShadow.width ( ); x++ ) {
      currentPixel = *(pLine + x);
      //iAlpha = (int)( (1.0 - m_fTransparency) / 3.0 * ( qRed( currentPixel ) + qGreen( currentPixel ) + qBlue( currentPixel ) ) );
      iAlpha = (int)( (1.0 - m_fTransparency) * qRed( currentPixel ) );
      newPixel = qRgba ( iRed, iGreen, iBlue, iAlpha );
      *(pLine + x) = newPixel;
    }
  }
  blurShadow ( );
}

void ShadowObject::blurShadow ( )
{
  if ( m_iBlurRadius > 0 ) {
    expblur <16, 8>( m_imageShadow, m_iBlurRadius );
    // the resulting image is a bit off around the corners, thus we should fade the shadow out
    int x, y;
    QRgb *pLine, currentPixel, newPixel;
    int iRed, iGreen, iBlue, iAlpha, iWidth, iHeight, iPhaseZone;
    float fPhaseOut;
      
    // What we have now is a greyscale image. The values actually represent the 
    // transparency of the final shadow.
    // Thus we will create an image that has only one color and only modulate the 
    // transparency.
    iWidth  = m_imageShadow.width   ( );
    iHeight = m_imageShadow.height  ( );
    iPhaseZone = 5;
    for ( y=0;  y < m_imageShadow.height ( ); y++ ) {
      pLine = (QRgb *) m_imageShadow.scanLine ( y );
      for ( x=0; x < m_imageShadow.width ( ); x++ ) {
	currentPixel = *(pLine + x);
	iRed   = qRed   ( currentPixel );
	iGreen = qGreen ( currentPixel );
	iBlue  = qBlue  ( currentPixel );
	fPhaseOut = 1.0;
	if ( x < iPhaseZone )
	  fPhaseOut = x / iPhaseZone;
	if ( x > iWidth - iPhaseZone )
	  fPhaseOut = ( iWidth - x ) / iPhaseZone;
	if ( y < iPhaseZone )
	  fPhaseOut = y / iPhaseZone;
	if ( y > iHeight - iPhaseZone )
	  fPhaseOut = ( iHeight - y ) / iPhaseZone;
	if ( fPhaseOut < 1.0 ) {
	  iAlpha = (int)( fPhaseOut * qAlpha ( currentPixel ) );
	
	  // Draw a red frame around the Shadow object so we know where we are ...
	  //if ( ( x == 0 ) || ( x == iWidth - 1) || ( y == 0 ) || ( y == iHeight - 1 ) )
	  //  newPixel = qRgba ( 255, 0, 0, 255 );
	  //else
	  newPixel = qRgba ( iRed, iGreen, iBlue, iAlpha );
	  *(pLine + x) = newPixel;
	}
      }
    }
  }
}

bool ShadowObject::filterColorKeys ( QImage &theShadow )
{
  ImageObject tempImage, *pImageObject;
  MovieObject tempMovie;
  // Only Image, or Movie objects of interest for color keying.
  if ( ( m_pMenuObject->objectType ( ) != tempImage.objectType ( ) ) &&
       ( m_pMenuObject->objectType ( ) != tempImage.objectType ( ) ) )
    return false;

  // Note MovieObject is derived off ImageObject.
  pImageObject = (ImageObject *)m_pMenuObject;
  // Check if Color Keying is enabled.
  if ( pImageObject->manipulator( ).listColorKeys.count ( ) < 1 )
    return false;

  // Next we create the min/max Color keys, to check if we ought to filter the pixel out or not
  ImageManipulator *pManipulator = &pImageObject->manipulator ( );
  QImage theImage   = pImageObject->pixmap ( ).convertToImage ( );
  QValueList<QColor *>listColorKeyMin;
  QValueList<QColor *>listColorKeyMax;
  QColor *pColorMin, *pColorMax, theColor, targetColor;
  QRgb thePixel, *pLine, *pShadowLine;
  
  int t, x, y, iX, iY, iDelta, iRed, iGreen, iBlue, iColorKeyCount;
  iColorKeyCount = pManipulator->listColorKeys.count();
  for (t=0;t<iColorKeyCount;t++)	{
    iDelta = (int)(pManipulator->listColorKeys[t]->fDeltaColor * 255.0);
    theColor = QColor (pManipulator->listColorKeys[t]->theColor);
    iRed = theColor.red() - iDelta; iGreen = theColor.green() - iDelta; iBlue = theColor.blue() - iDelta;
    iRed = (iRed < 0) ? 0 : iRed; iGreen = (iGreen < 0) ? 0 : iGreen; iBlue = (iBlue < 0) ? 0 : iBlue;
    pColorMin = new QColor (iRed, iGreen, iBlue);
    iRed = theColor.red() + iDelta; iGreen = theColor.green() + iDelta; iBlue = theColor.blue() + iDelta;
    iRed = (iRed > 255) ? 255 : iRed; iGreen = (iGreen > 255) ? 255 : iGreen; iBlue = (iBlue > 255) ? 255 : iBlue;
    pColorMax = new QColor (iRed, iGreen, iBlue);
    
    listColorKeyMin.append (pColorMin);
    listColorKeyMax.append (pColorMax);
  }

  for (y=0; y < theShadow.height ( ); y++)	{
    iY = y;
    if ( iY >= theImage.height ( ) )
         iY  = theImage.height ( )-1;
    pLine = (QRgb *)theImage.scanLine  ( iY );
    iY = y + 10; 
    if ( ( iY >= 0 ) && ( iY < theShadow.height ( ) ) ) {
      pShadowLine = (QRgb *)theShadow.scanLine  ( iY  );
      for (x=0; x < theShadow.width ( ); x++)	{
	iX = x + 10;
	if ( ( iX >= 0) && ( iX < theShadow.width ( ) ) ) {
	  if ( x >= theImage.width ( ) )
	    thePixel = *(pLine + theImage.width ( ) );
	  else
	    thePixel = *(pLine + x);
	  thePixel &= 0x00ffffff;		// filter out the transparency part of the color (not used in Qt 3.x)
	  if (thePixel != 0x000000)	{	// And check if this is a pixel we need to handle ...
	    targetColor = QColor ( thePixel );
	    if ( filterColorKey ( &targetColor, iColorKeyCount, &listColorKeyMin, &listColorKeyMax ) )
	      *(pShadowLine + iX ) = qRgba ( 0, 0, 0, 0 );
	  }
	}
      }
    }
  }
  for ( t=0; t<iColorKeyCount; t++ )  {
    delete listColorKeyMin[t];
    delete listColorKeyMax[t];
  }

  return true;
}

// This function checks if pColor is within the min/max of one of the color keys.
// Same as ImageObject::filterColorKey.
bool ShadowObject::filterColorKey ( QColor *pColor, int iColorKeyCount, QValueList<QColor *> *pListColorMin, QValueList<QColor *>*pListColorMax )
{
  int iRed, iGreen, iBlue, t;

  if ( ( iColorKeyCount < 1 ) || ( ! pListColorMin ) || ( ! pListColorMax ) )
    return false;
  iRed = pColor->red(); iGreen = pColor->green(); iBlue = pColor->blue ();
  for (t=0;t<iColorKeyCount;t++)	{
    if (	(iRed   >= (*pListColorMin)[t]->red  () && iRed   <= (*pListColorMax)[t]->red  ()) && 
		(iGreen >= (*pListColorMin)[t]->green() && iGreen <= (*pListColorMax)[t]->green()) &&
		(iBlue  >= (*pListColorMin)[t]->blue () && iBlue  <= (*pListColorMax)[t]->blue ()) )
      return true;
  }
  return false;
}

bool ShadowObject::writeProjectFile ( QDomElement &theElement ) 
{
  QDomDocument xmlDoc     = theElement.ownerDocument ( );
  QDomElement  shadowNode = xmlDoc.createElement( SHADOW_OBJECT );	// <ShadowObject>

  if ( m_color.isValid ( ) )
    shadowNode.setAttribute( SHADOW_OBJECT_COLOR, m_color.name ( ) );

  if ( m_fTransparency != 0.0f )
    shadowNode.setAttribute( SHADOW_OBJECT_TRANSPARENCY, m_fTransparency );

  if ( m_fDistance != -1.0 )
    shadowNode.setAttribute( SHADOW_OBJECT_DISTANCE, m_fDistance );

  if (m_iBlurRadius != 3 )
    shadowNode.setAttribute( SHADOW_OBJECT_BLUR_RADIUS, m_iBlurRadius );

  if (m_fSunAngle != 0.0f )
    shadowNode.setAttribute( SHADOW_OBJECT_SUN_ANGLE,  m_fSunAngle );
  
  theElement.appendChild( shadowNode );

 return true;
}

bool ShadowObject::readProjectFile ( QDomNode &theNode )
{
  QDomElement theElement = theNode.toElement();
  QDomAttr attribute;

  attribute = theElement.attributeNode ( SHADOW_OBJECT_COLOR );
  if ( ! attribute.isNull ( ) )
    m_color = QColor ( attribute.value ( ) );
  
  attribute = theElement.attributeNode ( SHADOW_OBJECT_TRANSPARENCY );
  if ( ! attribute.isNull ( ) )
    m_fTransparency = attribute.value ( ).toFloat( );
  
  attribute = theElement.attributeNode ( SHADOW_OBJECT_DISTANCE );
  if ( ! attribute.isNull ( ) )
    m_fDistance = attribute.value ( ).toFloat ( );
  
  attribute = theElement.attributeNode ( SHADOW_OBJECT_BLUR_RADIUS );
  if ( ! attribute.isNull ( ) )
    m_iBlurRadius = attribute.value ( ).toInt ( );

  attribute = theElement.attributeNode ( SHADOW_OBJECT_SUN_ANGLE );
  if ( ! attribute.isNull ( ) )
    m_fSunAngle = attribute.value ( ).toFloat ( );

  return true;
}

float ShadowObject::transparency ( )
{
  return m_fTransparency;
}

int ShadowObject::blur ( )
{
  return m_iBlurRadius;
}

float ShadowObject::distance ( )
{
  return m_fDistance;
}

float ShadowObject::sunAngle ( )
{
  return m_fSunAngle;
}

void ShadowObject::setTransparency ( float fTransparency )
{
  m_fTransparency = fTransparency;
}

void ShadowObject::setColor ( QColor color )
{
  m_color = color;
}

QColor &ShadowObject::color ( )
{
  return m_color;
}

void ShadowObject::setBlur ( int iBlurRadius )
{
  if ( iBlurRadius < 3 )
    iBlurRadius = 0;
  m_iBlurRadius = iBlurRadius;
}

void ShadowObject::setDistance ( float fDistance )
{
  if ( fDistance != m_fDistance ) {
    
    if ( m_pMenuObject ) {
      float fSunAngle = m_fSunAngle / 180.0 * PI;
      m_iDeltaX = (int) ( sin ( 45.0 / 180.0 * PI ) * fDistance * cos ( fSunAngle ) );
      m_iDeltaY = (int) ( sin ( 45.0 / 180.0 * PI ) * fDistance * sin ( fSunAngle ) );
    }
  }

  m_fDistance = fDistance;
}

void ShadowObject::setSunAngle ( float fSunAngle )
{
  if ( fSunAngle != m_fSunAngle ) {
    if ( m_pMenuObject ) {
      float fSunAngle = m_fSunAngle / 180.0 * PI;
      m_iDeltaX = (int) ( sin ( 45.0 / 180.0 * PI ) * m_fDistance * cos ( fSunAngle ) );
      m_iDeltaY = (int) ( sin ( 45.0 / 180.0 * PI ) * m_fDistance * sin ( fSunAngle ) );
    }
  }
  m_fSunAngle = fSunAngle;
}

int ShadowObject::deltaX ( )
{
  return ( m_iDeltaX - m_iBlurRadius - 10 );
}

int ShadowObject::deltaY ( )
{
  return ( m_iDeltaY - m_iBlurRadius - 10 );
}

MenuObject *ShadowObject::clone ( MenuObject *pParent )
{
  ShadowObject *pNewObject = new ShadowObject( pParent, MenuObject::parent ( ) );
  pNewObject->setTransparency ( transparency ( ) );
  pNewObject->setBlur         ( blur         ( ) );
  pNewObject->setDistance     ( distance     ( ) );
  pNewObject->setSunAngle     ( sunAngle     ( ) );
  pNewObject->setColor        ( color        ( ) );
  
  // The following two are not really needed, since we get those
  // informations solely from the Normal State - objects ...
  pNewObject->setRect(rect());
  pNewObject->setModifiers(*modifiers());
  
  return pNewObject;
}
