/****************************************************************************
** ButtonObject class
**
** Created: Tue Feb 02 22:06:51 2004
**      by: Varol Okan using Kate
**
** This class is the encapsulation of the ButtonObject. 
** 
** A ButtonObject is the object in a menu which was created by
** either dragging a SourceFileEntry over to the  DVD menu or
** by choosing "define as button" from the context menu of an 
** MenuObject.
**
**   Created : 
**        by : Varol Okan using Kate
** Copyright : (c) Varol Okan
**   License : GPL v 2.0
**
****************************************************************************/

#include <qtimer.h>
#include <qobject.h>
#include <qfileinfo.h>
#include <qlistview.h>
#include <qpopupmenu.h>

#include "global.h"
#include "xml_dvd.h"
#include "dvdmenu.h"
#include "menuobject.h"
#include "textobject.h"
#include "maskobject.h"
#include "qdvdauthor.h"
#include "imageobject.h"
#include "menupreview.h"
#include "frameobject.h"
#include "dialogbutton.h"
#include "buttonobject.h"
#include "sourcefileentry.h"
#include "structuretoolbar.h"

ButtonObject::ButtonObject (QWidget *pParent) //, const char *pName, WFlags flags)
	: MenuObject (pParent)
{
	m_qsObjectType     = QString (BUTTON_OBJECT);
	m_iButtonState     = STATE_NORMAL;
	m_pSourceFileEntry = NULL;
	m_bMoveable        = true;
	m_bLoopMultiple    = false;
}

ButtonObject::~ButtonObject ( )
{
  uint t;
  for ( t=0; t<m_listHighlightedState.count ( ); t++ )
    delete m_listHighlightedState[t];
  for ( t=0; t<m_listSelectedState.count ( ); t++ )
    delete m_listSelectedState[t];
  for ( t=0; t<m_listNormalState.count ( ); t++ )
    delete m_listNormalState[t];

  m_listNormalState.clear ( );
  m_listSelectedState.clear ( );
  m_listHighlightedState.clear ( );
}

MenuObject *ButtonObject::clone ( QWidget *pParent, MenuObject * )
{
  if ( ! pParent )
    pParent = MenuObject::parent ( );
  uint t;
  ButtonObject *pNewObject = new ButtonObject ( pParent );
  for ( t=0; t<m_listNormalState.count ( ); t++ )
    pNewObject->appendNormal ( m_listNormalState[t]->clone ( pParent, pNewObject ) );
  for ( t=0; t<m_listSelectedState.count ( ); t++ )
    pNewObject->appendSelected ( m_listSelectedState[t]->clone ( pParent, pNewObject ) );
  for ( t=0; t<m_listHighlightedState.count ( ); t++ )
    pNewObject->appendHighlighted ( m_listHighlightedState[t]->clone ( pParent, pNewObject ) );
  
  pNewObject->setSourceFileEntry(sourceFileEntry());
  pNewObject->setAction   ( action    ( ) );
  pNewObject->setPreAction( preAction ( ) );
  pNewObject->setName     ( name      ( ) );
  pNewObject->setPlayMultipleList( getPlayMultipleList ( ) );
  pNewObject->setLoopMultiple    ( getLoopMultiple     ( ) );
  pNewObject->setNext     ( NEXT_BUTTON_UP,    m_qsUp    );
  pNewObject->setNext     ( NEXT_BUTTON_DOWN,  m_qsDown  );
  pNewObject->setNext     ( NEXT_BUTTON_RIGHT, m_qsRight );
  pNewObject->setNext     ( NEXT_BUTTON_LEFT,  m_qsLeft  );
  
  return pNewObject;
}

ButtonObject &ButtonObject::operator = (ButtonObject &theOther)
{
	uint t;
	clear ();
	setSourceFileEntry ( theOther.sourceFileEntry ( ) );
	m_qsAction    = theOther.action    ( );
	m_qsPreAction = theOther.preAction ( );
	m_qsName      = theOther.name      ( );
	m_qsUp        = theOther.next ( NEXT_BUTTON_UP    );
	m_qsDown      = theOther.next ( NEXT_BUTTON_DOWN  );
	m_qsRight     = theOther.next ( NEXT_BUTTON_RIGHT );
	m_qsLeft      = theOther.next ( NEXT_BUTTON_LEFT  );
	m_listPlayMultiple = theOther.getPlayMultipleList ( );
	m_bLoopMultiple    = theOther.getLoopMultiple     ( );
	// The following lists hold the information for the different button states.
	for (t=0;t<theOther.getNormalCount();t++)
		appendNormal(theOther.getNormal(t));
	for (t=0;t<theOther.getSelectedCount();t++)
		appendSelected(theOther.getSelected(t));
	for (t=0;t<theOther.getHighlightedCount();t++)
		appendHighlighted(theOther.getHighlighted(t));
	return *this;
}

void ButtonObject::clear ()
{
	uint t;

	m_iButtonState = STATE_NORMAL;
	m_qsAction     = QString ("");
	m_qsPreAction  = QString ("");
	m_qsUp         = QString ("");
	m_qsDown       = QString ("");
	m_qsRight      = QString ("");
	m_qsLeft       = QString ("");
	m_pSourceFileEntry = NULL;
	m_listPlayMultiple.clear ( );

	for ( t=0; t<m_listNormalState.count ( ); t++ )
		delete m_listNormalState[t];
	for ( t=0; t<m_listSelectedState.count ( ); t++ )
		delete m_listSelectedState[t];
	for ( t=0; t<m_listHighlightedState.count ( ); t++ )
		delete m_listHighlightedState[t];
	m_listNormalState.clear  ( );
	m_listSelectedState.clear  ( );
	m_listHighlightedState.clear ( );
}

void ButtonObject::drawContents (QPainter *pPainter)
{
	drawContents ( pPainter, BACKGROUND_IMG );
//	m_listNormalState[0]->drawContents(pPainter);
//	m_boundingRect = m_listNormalState[0]->boundingRect();
//	m_rect = m_listNormalState[0]->rect();
}

void ButtonObject::drawContents (QPainter *pPainter, uint iWhichState)
{
  switch ( iWhichState )  {
  case BACKGROUND_IMG:
    if ( m_listNormalState[0] )  {
      m_listNormalState[0]->drawContents ( pPainter );
      m_boundingRect = m_listNormalState [0]->boundingRect ( );
      m_rect = m_listNormalState[0]->rect( );
    }
    break;
  case HIGHLIGHTED_MASK:
    if ( m_listHighlightedState[0] )
      m_listHighlightedState[0]->drawContents ( pPainter );
    break;
  case SELECTED_MASK:
    if ( m_listSelectedState[0] )
      m_listSelectedState[0]->drawContents ( pPainter );
    break;
  }
}

void ButtonObject::drawContents (QPainter *pPainter, int iRenderFrameNumber, int iTotalFrames)
{
	if (m_listNormalState[0])	{
		m_listNormalState[0]->drawContents(pPainter, iRenderFrameNumber, iTotalFrames);
		m_boundingRect = m_listNormalState[0]->boundingRect();
		m_rect = m_listNormalState[0]->rect();
	}
}

bool ButtonObject::mouseMoveEvent (QMouseEvent *pEvent)
{
  if (m_bMoveable)	{
    switch ( m_objectState )  {	
    case MenuObject::StateScaleTLXY:
    case MenuObject::StateScaleTRXY:
    case MenuObject::StateScaleBLXY:
    case MenuObject::StateScaleBRXY:
    case MenuObject::StateScaleLeftX:
    case MenuObject::StateScaleRightX:
    case MenuObject::StateScaleTopY:
    case MenuObject::StateScaleBottomY:
    case MenuObject::StateRotate: {
      uint t;
      Modifiers *pModifiers, *pObjectModifiers;
      MenuObject::mouseMoveEvent ( pEvent );
      pModifiers = modifiers ();
      for (t=0;t<m_listNormalState.count();t++) {
	pObjectModifiers = m_listNormalState[t]->modifiers ();
	*pObjectModifiers = *pModifiers;
      }
      for (t=0;t<m_listSelectedState.count();t++) {
	pObjectModifiers = m_listSelectedState[t]->modifiers ();
	*pObjectModifiers = *pModifiers;
      }
      for (t=0;t<m_listHighlightedState.count();t++) {
	pObjectModifiers = m_listHighlightedState[t]->modifiers ();
	*pObjectModifiers = *pModifiers;
      }
    }
      break;
    case MenuObject::StateMoveObject:
    default: {
      // This will move the ButtonObject and all associated objects around and around and around we go ...
      uint t;
      QRect theRect = rect();
      int iX, iY, iWidth, iHeight;
      iWidth  = rect().width ();
      iHeight = rect().height();
      iX = rect().x() - (m_currentMousePos.x() - pEvent->pos().x());
      iY = rect().y() - (m_currentMousePos.y() - pEvent->pos().y());
      theRect.setX(iX);
      theRect.setY(iY);
      theRect.setWidth(iWidth);
      theRect.setHeight(iHeight);
      
      setRect(theRect);
      for (t=0;t<m_listNormalState.count();t++)
	m_listNormalState[t]->setRect (theRect);
      for (t=0;t<m_listSelectedState.count();t++)
	m_listSelectedState[t]->setRect (theRect);
      for (t=0;t<m_listHighlightedState.count();t++)
	m_listHighlightedState[t]->setRect (theRect);
      }
    }
  }
  m_currentMousePos = pEvent->pos();
  return false;
}

bool ButtonObject::mousePressEvent (QMouseEvent *pEvent)
{
	m_currentMousePos = pEvent->pos ( );
	if (pEvent->button() == Qt::RightButton)	{
		QPoint globalPos = pEvent->globalPos();
		return createContextMenu ( globalPos );
	}
	else // Here I only want to activate the drawing of the frame rather then the full thing.
	  if ( m_listNormalState[0] )
	       m_listNormalState[0]->mousePressEvent (pEvent);
	m_bDrawRect = true;
	return false;
}

bool ButtonObject::mouseReleaseEvent (QMouseEvent *pEvent)
{
	m_bDrawRect = false;
	if ( m_listNormalState[0] )
	     m_listNormalState[0]->mouseReleaseEvent (pEvent);
	emit (signalUpdatePixmap());
	emit (signalUpdateStructure());
	return false;
}

bool ButtonObject::mouseDoubleClickEvent(QMouseEvent *)
{
  // Call MenuPreview::slotCreateButtonDialog ()
  emit (signalCreateButtonDialog (this));
  return false;
}

MenuObject *ButtonObject::shadow ( )
{
  MenuObject *pNormal = getNormal ( 0 );
  if ( pNormal )
    return pNormal->shadow ( );
  return NULL;
}

void ButtonObject::updateShadow ( )
{
  MenuObject *pNormal = getNormal ( 0 );
  if ( pNormal )
       pNormal->updateShadow ( );
}

void ButtonObject::setShadow ( MenuObject *pShadow )
{
  MenuObject *pNormal = getNormal ( 0 );
  if ( pNormal )
       pNormal->setShadow ( pShadow );
}

void ButtonObject::appendNormal (MenuObject *pObject)
{
	m_listNormalState.append (pObject);
}

void ButtonObject::appendSelected (MenuObject *pObject)
{
	m_listSelectedState.append (pObject);
}

void ButtonObject::appendHighlighted (MenuObject *pObject)
{
	m_listHighlightedState.append (pObject);
}

Rgba ButtonObject::getMaskColor ( bool bHighlighted, int iIdx )
{
  Rgba returnColor ( 0, 0, 0, 0 );
  MenuObject *pObject = NULL;
  if ( bHighlighted ) {
    if ( iIdx >= (int)m_listHighlightedState.count ( ) )
      return returnColor;
    pObject = m_listHighlightedState[iIdx];
  }
  else {
    if ( iIdx >= (int)m_listSelectedState.count ( ) )
      return returnColor;
    pObject = m_listSelectedState[iIdx];
  }

  if ( pObject->objectType ( ) == MASK_OBJECT ) {
    MaskObject   *pMask = (MaskObject *)pObject;
    returnColor = pMask->color ( );
  }
  else if ( pObject->objectType ( ) == FRAME_OBJECT ) {
    FrameObject  *pFrame = (FrameObject *)pObject;
    returnColor = pFrame->color ( );
  }
  else if ( pObject->objectType ( ) == TEXT_OBJECT ) {
    TextObject   *pText = (TextObject *)pObject;
    returnColor = pText->getForegroundColor ( );
  }
  return returnColor;
}

bool ButtonObject::setMaskColor ( Rgba newColor, bool bHighlighted, int iIdx )
{
  MenuObject *pObject = NULL;
  if ( bHighlighted ) {
    if ( iIdx >= (int)m_listHighlightedState.count ( ) )
      return false;
    pObject = m_listHighlightedState[iIdx];
  }
  else {
    if ( iIdx >= (int)m_listSelectedState.count ( ) )
      return false;
    pObject = m_listSelectedState[iIdx];
  }

  if ( pObject->objectType ( ) == MASK_OBJECT ) {
    MaskObject  *pMask = (MaskObject *)pObject;
    pMask->setColor ( newColor );
  }
  else if ( pObject->objectType ( ) == FRAME_OBJECT ) {
    FrameObject *pFrame = (FrameObject *)pObject;
    pFrame->setFrameColor ( newColor );
  }
  else if ( pObject->objectType ( ) == TEXT_OBJECT ) {
    TextObject  *pText = (TextObject *)pObject;
    pText->setForegroundColor ( newColor );
  }
  else
    return false;
  return true;
}

uint ButtonObject::getNormalCount ()
{
  return m_listNormalState.count();
}

MenuObject *ButtonObject::getNormal ( uint iNr )
{
  if ( iNr >= m_listNormalState.count ( ) )
    return NULL;
  return m_listNormalState[iNr];
}

MenuObject *ButtonObject::removeNormal ( uint iNr )
{
  MenuObject *pMenuObject = getNormal ( iNr );
  if ( pMenuObject )
    m_listNormalState.remove ( pMenuObject );
  return pMenuObject;
}

bool ButtonObject::removeNormal ( MenuObject *pMenuObject )
{
  if ( ! pMenuObject )
    return false;
  return m_listNormalState.remove ( pMenuObject );
}

uint ButtonObject::getSelectedCount ()
{
  return m_listSelectedState.count();
}

MenuObject *ButtonObject::getSelected ( uint iNr )
{
  if ( iNr >= m_listSelectedState.count ( ) )
    return NULL;
  return m_listSelectedState[iNr];
}

MenuObject *ButtonObject::removeSelected ( uint iNr )
{
  MenuObject *pMenuObject = getSelected ( iNr );
  if ( pMenuObject )
    m_listSelectedState.remove ( pMenuObject );
  return pMenuObject;
}

bool ButtonObject::removeSelected ( MenuObject *pMenuObject )
{
  if ( ! pMenuObject )
    return false;
  return m_listSelectedState.remove ( pMenuObject );
}

uint ButtonObject::getHighlightedCount ()
{
  return m_listHighlightedState.count();
}

MenuObject *ButtonObject::getHighlighted ( uint iNr )
{
  if ( iNr >= m_listHighlightedState.count ( ) )
    return NULL;
  return m_listHighlightedState[iNr];
}

MenuObject *ButtonObject::removeHighlighted ( uint iNr )
{
  MenuObject *pMenuObject = getHighlighted ( iNr );
  if ( pMenuObject )
    m_listHighlightedState.remove ( pMenuObject );
  return pMenuObject;
}

bool ButtonObject::removeHighlighted ( MenuObject *pMenuObject )
{
  if ( ! pMenuObject )
    return false;
  return m_listHighlightedState.remove ( pMenuObject );
}

QStringList &ButtonObject::getPlayMultipleList ( )
{
  return m_listPlayMultiple;
}

void ButtonObject::setPlayMultipleList ( QStringList &newList )
{
  m_listPlayMultiple = newList;
}

bool ButtonObject::getLoopMultiple ( )
{
  return m_bLoopMultiple;
}

void ButtonObject::setLoopMultiple ( bool bLoopMultiple )
{
  m_bLoopMultiple = bLoopMultiple;
}

void ButtonObject::deleteMasks()
{
	// This function only handles the case of the cancelation of the Button creation
	// From the DialogButton - class.
	// I.e. the user just created the Button and decides to cancel out.
	// In case the User cancels the ButtonDialog when the ButtonObject already 
	// existed before, nothing ought to be done here.
	uint t;
	// we can leave the m_listNormalState - objects as they reflect the original Objects.
	// However the m_listSelected / Highlighted should be deleted ...
	for (t=0;t<m_listSelectedState.count();t++)
		delete m_listSelectedState[t];
	for (t=0;t<m_listHighlightedState.count();t++)
		delete m_listHighlightedState[t];
	m_listSelectedState.clear();
	m_listHighlightedState.clear();
}

bool ButtonObject::readProjectFile ( QDomNode &theNode )
{
	// Okay, here we retain the stored data from the xml file.
	QDomNode    xmlMultiple = theNode.firstChild ( );
	QDomElement theElement  = theNode.toElement();
	QDomAttr    attribute;

	attribute = theElement.attributeNode ( BUTTON_OBJECT_NAME );
	if (!attribute.isNull())
		m_qsName = attribute.value();

	attribute = theElement.attributeNode ( BUTTON_OBJECT_ACTION );
	if (!attribute.isNull())
		m_qsAction = attribute.value();

	attribute = theElement.attributeNode ( BUTTON_OBJECT_PRE_ACTION );
	if ( ! attribute.isNull ( ) )
		m_qsPreAction = attribute.value ( );

	attribute = theElement.attributeNode ( BUTTON_OBJECT_UP );
	if (!attribute.isNull())
		m_qsUp = attribute.value();
	else
		m_qsUp = QString ("-- default --");
	
	attribute = theElement.attributeNode ( BUTTON_OBJECT_DOWN );
	if (!attribute.isNull())
		m_qsDown = attribute.value();
	else
		m_qsDown = QString ("-- default --");

	attribute = theElement.attributeNode ( BUTTON_OBJECT_LEFT );
	if (!attribute.isNull())
		m_qsLeft = attribute.value();
	else
		m_qsLeft = QString ("-- default --");

	attribute = theElement.attributeNode ( BUTTON_OBJECT_RIGHT );
	if (!attribute.isNull())
		m_qsRight = attribute.value();
	else
		m_qsRight = QString ("-- default --");

	attribute = theElement.attributeNode ( BUTTON_OBJECT_MOVEABLE );
	m_bMoveable = true;
	if (!attribute.isNull())	{
		if (attribute.value () == QString ("false"))
			m_bMoveable = false;
	}

	attribute = theElement.attributeNode ( BUTTON_OBJECT_SOURCE_ENTRY);
	if (!attribute.isNull())
		m_qsSourceDisplayName = attribute.value();

	attribute = theElement.attributeNode ( BUTTON_OBJECT_LOOP_MULTIPLE );
	m_bLoopMultiple = false;
	if ( ! attribute.isNull ( ) ) {
	  if ( attribute.value  ( ) == QString ("true") )
	    m_bLoopMultiple = true;
	}

	// Please note that the nortmal / selected / highlighted objects are read
	// in by ButtonObject::readObjects(...);
	return true;
}

bool ButtonObject::readObjects (QDomNode &theNode, MenuPreview *pPreview)
{
  int t;
  QDomElement buttonElement;
  QDomNode    childNode;
  QDomNode    buttonNode = theNode.firstChild();
  MenuObject *pNewObject = NULL;
  MenuObject *pMenuObject;
  while ( ! buttonNode.isNull ( ) )  {
    buttonElement   = buttonNode.toElement  ( );
    QString tagName = buttonElement.tagName ( );
    if ( buttonElement.hasChildNodes ( ) ) { // For Masks we do not need any children.
      childNode = buttonNode.firstChild ( );
      // Next we should create the new Object but if we see the
      // MenoObject tag it means it is the MenuObject vars of the button
      // itself, thus we don't want to create anything. (It would put out a warning).
      if ( tagName == BUTTON_OBJECT_MULTIPLE ) {
	QString nodeText = buttonElement.text ( );
	m_listPlayMultiple.append ( nodeText );
      }
      else if ( tagName != MENU_OBJECT ) 
	pNewObject = pPreview->readObject ( childNode );

      if ( tagName == BUTTON_OBJECT_NORMAL )
	m_listNormalState.append ( pNewObject );

      else if ( tagName == BUTTON_OBJECT_SELECTED )
	m_listSelectedState.append( pNewObject );

      else if ( tagName == BUTTON_OBJECT_HIGHLIGHTED )
	m_listHighlightedState.append ( pNewObject );

      else if ( tagName == MENU_OBJECT )
	MenuObject::readProjectFile( theNode );

      else if ( tagName != BUTTON_OBJECT_MULTIPLE )
	printf ("Warning : ButtonObject::readObjects -=> wrong XML Node <%s>\nContinuing ...\n", (const char *)tagName);
    }
    buttonNode = buttonNode.nextSibling ();
  }

  // short sanity check ..
  if ( m_listNormalState.count ( ) < 1 )
    return false;
 
  MaskObject *pMask = NULL;
  MenuObject *pParentObject = m_listNormalState[0];
  for ( t=0; t<(int)m_listSelectedState.count ( ); t++ ) {
    pMenuObject = m_listSelectedState[t];
    if ( pMenuObject->objectType ( ) == QString ( MASK_OBJECT ) ) {
      pMask = (MaskObject *)pMenuObject;
      pMask->setParent ( pParentObject );
    }
  }
  for ( t=0; t<(int)m_listHighlightedState.count ( ); t++ ) {
    pMenuObject = m_listHighlightedState[t];
    if ( pMenuObject->objectType ( ) == QString ( MASK_OBJECT ) ) {
      pMask = (MaskObject *)pMenuObject;
      pMask->setParent ( pParentObject );
    }
  } 
  return true;
}

bool ButtonObject::writeProjectFile (QDomElement &theElement)
{
	uint t;
	QDomDocument xmlDoc = theElement.ownerDocument();
	QDomElement buttonNode = xmlDoc.createElement( BUTTON_OBJECT );	// <FrameObject>
	// Here we set the attributes of the <dvdauthor> tag
	if (!m_qsName.isNull())
		buttonNode.setAttribute( BUTTON_OBJECT_NAME,   m_qsName );
	if (!m_qsAction.isNull())
		buttonNode.setAttribute( BUTTON_OBJECT_ACTION, m_qsAction );
	if ( ! m_qsPreAction.isNull ( ) )
		buttonNode.setAttribute( BUTTON_OBJECT_PRE_ACTION, m_qsPreAction );
	if ( (!m_qsUp.isNull()) && ( m_qsUp != QString ("-- default --")) )
		buttonNode.setAttribute( BUTTON_OBJECT_UP,     m_qsUp );
	if ( (!m_qsDown.isNull()) && ( m_qsDown != QString ("-- default --")) )
		buttonNode.setAttribute( BUTTON_OBJECT_DOWN,   m_qsDown );
	if ( (!m_qsLeft.isNull()) && ( m_qsLeft != QString ("-- default --")) )
		buttonNode.setAttribute( BUTTON_OBJECT_LEFT,   m_qsLeft );
	if ( (!m_qsRight.isNull()) && ( m_qsRight != QString ("-- default --")) )
		buttonNode.setAttribute( BUTTON_OBJECT_RIGHT,  m_qsRight );
	if (!m_bMoveable)
		buttonNode.setAttribute( BUTTON_OBJECT_MOVEABLE, QString ("false") );
	if ( m_bLoopMultiple )
		buttonNode.setAttribute( BUTTON_OBJECT_LOOP_MULTIPLE, QString ("true") );

	if (m_pSourceFileEntry)
		buttonNode.setAttribute( BUTTON_OBJECT_SOURCE_ENTRY,  m_pSourceFileEntry->qsDisplayName );
//printf ("ButtonObject::writeProjectFile <%s> - <%s> \n", (const char *)m_pSourceFileEntry->qsDisplayName, (const char *)m_qsAction );
	
	// And here we store the list of multiple objects ( if present )
	for ( t=0; t<m_listPlayMultiple.count ( ); t++ ) {
	  QDomElement multipleNode = xmlDoc.createElement  ( BUTTON_OBJECT_MULTIPLE );
	  QDomText text = xmlDoc.createTextNode ( m_listPlayMultiple[t] );
	  multipleNode.appendChild ( text );
	  buttonNode.appendChild ( multipleNode );
	}

	// And here we store the ButtonObjects children ...
	QDomElement normalNode = xmlDoc.createElement      ( BUTTON_OBJECT_NORMAL );
	for ( t=0; t<m_listNormalState.count ( ); t++ )
	  m_listNormalState[t]->writeProjectFile ( normalNode );

	QDomElement selectedNode = xmlDoc.createElement    ( BUTTON_OBJECT_SELECTED );
	for ( t=0; t<m_listSelectedState.count ( ); t++ )
	  m_listSelectedState[t]->writeProjectFile ( selectedNode );

	QDomElement highlightedNode = xmlDoc.createElement ( BUTTON_OBJECT_HIGHLIGHTED );
	for (t=0;t<m_listHighlightedState.count();t++)
	  m_listHighlightedState[t]->writeProjectFile ( highlightedNode );

	buttonNode.appendChild ( normalNode      );
	buttonNode.appendChild ( selectedNode    );
	buttonNode.appendChild ( highlightedNode );
	
	theElement.appendChild ( buttonNode );
	// And here we write the base class ...
	return MenuObject::writeProjectFile( buttonNode );
}

bool ButtonObject::createContextMenu (QPoint globalPos)
{
	int     iID = -1;
	bool    bHasPropertyDialog;
	QString qsObject, qsEditObject;
        QPoint  globalPos2 = globalPos;
	QPopupMenu *pStackMenu = new QPopupMenu(m_pParent);
	pStackMenu->insertItem ( tr ("Cut") , this, SLOT ( slotCut  ( ) ) );
	pStackMenu->insertItem ( tr ("Copy"), this, SLOT ( slotCopy ( ) ) );
	pStackMenu->insertSeparator ( );
	pStackMenu->insertItem ( tr ("To Front")  , this, SLOT(slotToFront()));
	pStackMenu->insertItem ( tr ("To Back")   , this, SLOT(slotToBack()));
	globalPos.setY ( globalPos.y ( ) - 25 );
	globalPos.setX ( globalPos.x ( ) - pStackMenu->sizeHint().width()); // -100);
	pStackMenu->popup(globalPos, 1);

	bHasPropertyDialog = false;
	qsEditObject = tr ( "Edit Object ..." );
	if ( m_listNormalState[0] ) {
	  if ( m_listNormalState[0]->objectType ( ) == QString ( IMAGE_OBJECT ) ) {
	    qsEditObject = tr ( "Edit Image ..." );
	    qsObject     = tr ( "Image Properties ..." );
	    bHasPropertyDialog = true;
	  }
	  else if ( m_listNormalState[0]->objectType ( ) == QString ( MOVIE_OBJECT ) ) {
	    qsEditObject = tr ( "Edit Movie ..." );
	    qsObject     = tr ( "Movie Properties ..." );
	    bHasPropertyDialog = true;
	  }
	}

	if ( m_pContextMenu )
	     delete m_pContextMenu;
	m_pContextMenu = new QPopupMenu(m_pParent);
	m_pContextMenu->setCheckable (true);
	if ( bHasPropertyDialog )
	     m_pContextMenu->insertItem ( qsObject,          (QObject *)this, SLOT(slotObjectProperties ( ) ) );
	m_pContextMenu->insertItem ( qsEditObject,           (QObject *)this, SLOT(slotEditObject ( ) ) );
	m_pContextMenu->insertSeparator ( );
	m_pContextMenu->insertItem ( tr ("Edit Button ..."), (QObject *)this, SLOT(slotEdit       ( ) ) );
	m_pContextMenu->insertItem ( tr ("Unbutton"),        (QObject *)this, SLOT(slotUnbutton   ( ) ) );
	m_pContextMenu->insertItem ( tr ("Delete"),          (QObject *)this, SLOT(slotDelete     ( ) ) );
	iID = m_pContextMenu->insertItem ( tr ("Moveable"),  (QObject *)this, SLOT(slotMoveable   ( ) ) );
	m_pContextMenu->insertSeparator ( );
	m_pContextMenu->insertItem ( tr ("To Text"),        (QObject *)this, SLOT(slotToText    ( ) ) );
	m_pContextMenu->insertItem ( tr ("To Frame"),       (QObject *)this, SLOT(slotToFrame   ( ) ) );
	m_pContextMenu->insertItem ( tr ("To Image"),       (QObject *)this, SLOT(slotToImage   ( ) ) );
	m_pContextMenu->insertItem ( tr ("To Movie"),       (QObject *)this, SLOT(slotToMovie   ( ) ) );
	m_pContextMenu->setItemChecked (iID, m_bMoveable);

	m_pContextMenu->exec(globalPos2, 5);

	delete pStackMenu;
	if (m_pContextMenu)
		delete m_pContextMenu;
	m_pContextMenu = NULL;

	// Here we mark that the user called a menu item thus we don't want the
	// base classes to continue with the mouse event
//	if (iReturn != -1)
		return true;
}

void ButtonObject::createMask ( Rgba *pPgcColors )
{
  MaskObject *pMask   = NULL;
  MenuObject *pParent = getNormal  ( 0 );
  if ( ( ! pParent ) || ( ! pPgcColors ) )
    return;

  pMask = new MaskObject ( pParent, MenuObject::parent ( ) );
  pMask  ->setColor      ( pPgcColors[1] );
  pMask  ->updateMask    ( );
  appendHighlighted      ( pMask );

  pMask = new MaskObject ( pParent, MenuObject::parent ( ) );
  pMask  ->setColor      ( pPgcColors[2] );
  pMask  ->updateMask    ( );
  appendSelected         ( pMask );
}

StructureItem *ButtonObject::createStructure ( StructureItem *pParentItem )
{
  // Creating teh ListView structure of this Button.
  StructureItem *pStateItem, *pTemp=NULL;

  QString qsAction;
  QStringList actionList;
  
  if ( m_listPlayMultiple.count ( ) > 1 )
    qsAction = tr ( "Play Multiple Files" );
  else if ( ! m_qsAction.isEmpty ( ) ) { // For new buttons this might be empty ...
    actionList = QStringList::split (QString (STRING_SEPARATOR), m_qsAction);
    if (actionList.count() == 1)	
      qsAction = m_qsAction;
    else	{
      qsAction = actionList[0] + QString (" ") +actionList[1] + QString (" ") +actionList[2];
      if (actionList.count() == 4)
	qsAction += QString (" ") +actionList[3];
    }
  }
  if ( ! m_pStructureItem ) {
    m_pStructureItem = new StructureItem ( this, pParentItem, pParentItem->lastChild ( ), m_qsName, qsAction );
    m_pStructureItem->setExpandable      ( TRUE );
  }
  else
    m_pStructureItem->setText ( m_qsName, qsAction );
  
  uint t;
  QString qsEmpty;
  QString qsHighlighted = tr ( "Highlighted" );
  QString qsSelected    = tr ( "Selected" );

  for ( t=0; t<m_listNormalState.count ( ); t++ )
    pTemp = m_listNormalState[t]->createStructure ( m_pStructureItem );

  pStateItem =(StructureItem *) m_pStructureItem->firstChild ( );
  while ( pStateItem )  {
    if  ( pStateItem->text ( 0 ) == qsHighlighted )
      break;
    pStateItem = (StructureItem *)pStateItem->nextSibling ( );
  }
  if ( ! pStateItem ) {
    pStateItem = new StructureItem ( this, m_pStructureItem, pTemp, qsHighlighted, qsEmpty );
    pStateItem->setExpandable ( TRUE );
  }
  for ( t=0; t<m_listHighlightedState.count ( ); t++ )
    pTemp = m_listHighlightedState[t]->createStructure ( pStateItem );

  pTemp = pStateItem;
  pStateItem =(StructureItem *) m_pStructureItem->firstChild ( );
  while ( pStateItem )  {
    if  ( pStateItem->text ( 0 ) == qsSelected )
      break;
    pStateItem = (StructureItem *)pStateItem->nextSibling ( );
  }
  if ( ! pStateItem ) {
    pStateItem = new StructureItem ( this, m_pStructureItem, pTemp, qsSelected, qsEmpty );
    pStateItem->setExpandable ( TRUE );
  }
  for ( t=0; t<m_listSelectedState.count ( ); t++ )
    pTemp = m_listSelectedState[t]->createStructure ( pStateItem );

  return m_pStructureItem;
}

void ButtonObject::replaceColor ( Rgba theColor, Rgba oldColor )
{
	// We handle the actual color as a unique index. Thus we allow 
	// the user to utilise any of the 4 available colors for any purpose.
	uint t;
	
	// Does this make sense ??? I don't know yet ...
//	for (t=0;t<m_listNormalState.count();t++)	{
//		m_listNormalState[t]->replaceColor (theColor);
//	}
	for (t=0;t<m_listSelectedState.count();t++)	{
		m_listSelectedState[t]->replaceColor ( theColor, oldColor );
	}
	for (t=0;t<m_listHighlightedState.count();t++)	{
		m_listHighlightedState[t]->replaceColor ( theColor, oldColor );
	}
}

QRect &ButtonObject::boundingRect()
{
  // Here we have a little special handling for frames to include the frame width
  // for button objects.
  static QRect rect;
  FrameObject  tempFrameObject, *pFrame;
  int iHalf;
  rect = m_boundingRect;

  if ( m_listHighlightedState.count () < 1 )
    return rect;

  if ( m_listHighlightedState[0]->objectType () == tempFrameObject.objectType() ) {
    pFrame = (FrameObject *)m_listHighlightedState[0];
    iHalf  = (int)((float)pFrame->width() / 2.0);
    rect = QRect (m_boundingRect.x()-iHalf, m_boundingRect.y()-iHalf, m_boundingRect.width()+2*iHalf,  m_boundingRect.height()+2*iHalf);
  }

  return rect;
}

void ButtonObject::setAction(QString &qsAction)
{
	m_qsAction = qsAction;
}

QString &ButtonObject::action()
{
	return m_qsAction;
}

void ButtonObject::setPreAction(QString &qsPreAction)
{
  m_qsPreAction = qsPreAction;
}

QString &ButtonObject::preAction()
{
  return m_qsPreAction;
}

void ButtonObject::setSourceFileEntry(SourceFileEntry *pEntry)
{
	m_pSourceFileEntry = pEntry;
	if (pEntry)
		m_qsSourceDisplayName = pEntry->qsDisplayName;
	else
		m_qsSourceDisplayName = QString ();
}

QString &ButtonObject::sourceDisplayName ()
{
	return m_qsSourceDisplayName;
}

SourceFileEntry *ButtonObject::sourceFileEntry()
{
  return m_pSourceFileEntry;
}

SourceFileInfo *ButtonObject::sourceFileInfo ( )
{
  uint             t;
  QString          qsName;
  QFileInfo        fileInfo;
  QStringList      actionList;
  SourceFileEntry *pEntry = sourceFileEntry ( );
  SourceFileInfo  *pInfo  = NULL;

  if  (  ! pEntry )
    return pInfo;

  // default to the first info 
  pInfo = pEntry->listFileInfos[0];

  if ( ! m_qsAction.isEmpty() )  {  // For new buttons this might be empty ...
    actionList = QStringList::split ( QString (STRING_SEPARATOR), m_qsAction );
    if ( actionList.count ( ) == 4 )
      qsName = actionList[actionList.count ( ) - 2];
    else if ( actionList.count ( ) > 1 )
      qsName = actionList[1];
  }
  // find the right sourceFileInfo from the action - string
  for ( t=0; t<pEntry->listFileInfos.count ( ); t++ ) {
    fileInfo.setFile ( pEntry->listFileInfos[t]->qsFileName );
    if ( qsName == fileInfo.fileName ( ) )
      return pEntry->listFileInfos[t];
  }

  return pInfo;
}

int ButtonObject::chapter ( bool bAllEntries )
{
  uint t, iTemp;
  int iChapterNumber = 1;
  SourceFileEntry *pEntry = sourceFileEntry ();
  SourceFileInfo  *pInfo  = sourceFileInfo  ();
  if ( ( ! pEntry ) || ( ! pInfo ) )
    return 0;  // No Info present ... Error
  QStringList     actionList;

  if ( ! m_qsAction.isEmpty() )  // For new buttons this might be empty ...
    actionList = QStringList::split (QString (STRING_SEPARATOR), m_qsAction);
  if ( actionList.count() < 1 )
    return iChapterNumber;

  // To get the total chapter number you have to add up the SourceFileInfos
  // chapters in the SourceFileEntry that come before the actual SourceFileInfo
  // on which the button points.
  iChapterNumber = 1;
  if ( bAllEntries ) {
    for (t=0;t<pEntry->listFileInfos.count();t++) {
      if ( pEntry->listFileInfos[t] == pInfo )
	break; // break out of the loop
      iTemp = pEntry->listFileInfos[t]->listChapters.count();
      //iTemp = (iTemp < 1) ? 1: iTemp; // removed 20060920. No chapter, no -pit-stop-, no count increase.
      iChapterNumber += iTemp;
    }
  }

  if ( pInfo->listChapters.count () > 0 ) {
    for ( t=0;t<pInfo->listChapters.count ();t++ ) {
      if (pInfo->listChapters[t] == actionList[actionList.count()-1])	// the chapter
	iChapterNumber += t;
    }
  }
  else // In order to reach this VOB it needs to have at least one chapter to get to. Remember pInfo is associated with this button.
    pInfo->listChapters.append ( "00:00:00.000" ); 

  return iChapterNumber;  // Info present but no chapter info...
}

void ButtonObject::setNext(uint iNext, QString &qsNext)
{
	switch (iNext)	{
	case NEXT_BUTTON_UP:
		m_qsUp = qsNext;
	break;
	case NEXT_BUTTON_DOWN:
		m_qsDown = qsNext;
	break;
	case NEXT_BUTTON_RIGHT:
		m_qsRight = qsNext;
	break;
	case NEXT_BUTTON_LEFT:
		m_qsLeft = qsNext;
	break;
	};
}

QString &ButtonObject::next(uint iNext)
{
	switch (iNext)	{
	case NEXT_BUTTON_UP:
		return m_qsUp;
	break;
	case NEXT_BUTTON_DOWN:
		return m_qsDown;
	break;
	case NEXT_BUTTON_RIGHT:
		return m_qsRight;
	break;
	case NEXT_BUTTON_LEFT:
		return m_qsLeft;
	break;
	};
	return m_qsUp;
}

void ButtonObject::slotEdit ( )
{
  mouseDoubleClickEvent ( NULL );
}

void ButtonObject::slotEditObject ( )
{
  MenuObject *pMenuObject = m_listNormalState[0];
  if ( pMenuObject ) {
       pMenuObject->mouseDoubleClickEvent ( NULL );

    // Here we handle a special case if the Text / Font of a TextObject has changed
    if ( pMenuObject->objectType ( ) == QString ( TEXT_OBJECT ) ) {
      uint t;
      TextObject *pTextObject = (TextObject *)pMenuObject;
      QPoint  align   = pTextObject->getTextAlign ( );
      QString theText = pTextObject->getText ( );
      QFont   theFont = pTextObject->getFont ( );
      QRect   theRect = pTextObject->rect    ( );
      for ( t=0; t<m_listSelectedState.count ( ); t++ ) {
	pMenuObject = m_listSelectedState[t];
	if ( pMenuObject && ( pMenuObject->objectType ( ) == QString ( TEXT_OBJECT ) ) ) {
	  pTextObject = (TextObject *)pMenuObject;
	  pTextObject->setText (  theText  );
	  pTextObject->setFont (  theFont  );
	  pTextObject->setRect (  theRect  );
	  pTextObject->setTextAlign( align );
	}
      }
      for ( t=0; t<m_listHighlightedState.count ( ); t++ ) {
	pMenuObject = m_listHighlightedState[t];
	if ( pMenuObject && ( pMenuObject->objectType ( ) == QString ( TEXT_OBJECT ) ) ) {
	  pTextObject = (TextObject *)pMenuObject;
	  pTextObject->setText (  theText  );
	  pTextObject->setFont (  theFont  );
	  pTextObject->setRect (  theRect  );
	  pTextObject->setTextAlign( align );
	}
      }
    }
  }
}

void ButtonObject::slotObjectProperties ( )
{
  if ( m_listNormalState[0] ) {
    MenuObject *pObject = m_listNormalState[0];
    if ( ( pObject->objectType ( ) == QString ( IMAGE_OBJECT ) ) || 
	 ( pObject->objectType ( ) == QString ( MOVIE_OBJECT ) ) ) {
      ImageObject *pImage = (ImageObject *)pObject;
      pImage->slotProperties ( );
    }
  }
}

void ButtonObject::slotUnbutton ( )
{
  // get a slight delay to allow the ContextMenu to be destroyed ...
  QTimer::singleShot ( 10, this, SLOT ( slotEmitUnbuttonMe ( ) ) );
}

void ButtonObject::slotEmitUnbuttonMe ( )
{
  // Here we are going to split the Button object into its three distinct 
  // objects and add them to the Preview (MenuPreview that is).
  emit ( signalUnbuttonMe ( this ) );
  // The work is done in MenuPreview::slotUnbutton()
}

void ButtonObject::slotMoveable()
{
  // if it is not moveable the Button is locked in.
  m_bMoveable = !m_bMoveable;
}

void ButtonObject::slotToText ( )
{
  // Converts the current ButtonObject type to a TextButton
  DVDMenu *pCurrentMenu = Global::pApp->getCurrentSubMenu ( );
  if ( ! pCurrentMenu )
    return;

  MenuPreview *pMenuPreview = pCurrentMenu->getMenuPreview ( );
  if ( ! pMenuPreview )
    return;

  MenuObject  *pNewObject = pMenuPreview->createTextObject ( true );
  if ( ! pNewObject )
    return;

  pMenuPreview->removeObject ( pNewObject );
  MenuObject *pOldObject = m_listNormalState[0];

  pNewObject->setRect        (  pOldObject->rect      ( ) );
  pNewObject->setModifiers   ( *pOldObject->modifiers ( ) );

  m_listNormalState.remove   ( pOldObject );
  m_listNormalState.append   ( pNewObject );

  emit ( signalUpdatePixmap    ( ) );
  emit ( signalUpdateStructure ( ) );

  delete pOldObject;
}

void ButtonObject::slotToFrame ( )
{
  // Converts the current ButtonObject type to a FrameButton
  DVDMenu *pCurrentMenu = Global::pApp->getCurrentSubMenu ( );
  if ( ! pCurrentMenu )
    return;

  MenuPreview *pMenuPreview = pCurrentMenu->getMenuPreview ( );
  if ( ! pMenuPreview )
    return;

  MenuObject  *pNewObject = pMenuPreview->createFrameObject ( true );
  if ( ! pNewObject )
    return;

  pMenuPreview->removeObject ( pNewObject );
  MenuObject *pOldObject = m_listNormalState[0];

  pNewObject->setRect        (  pOldObject->rect      ( ) );
  pNewObject->setModifiers   ( *pOldObject->modifiers ( ) );

  m_listNormalState.remove   ( pOldObject );
  m_listNormalState.append   ( pNewObject );

  emit ( signalUpdatePixmap    ( ) );
  emit ( signalUpdateStructure ( ) );

  delete pOldObject;
}

void ButtonObject::slotToImage ( )
{
  // Converts the current ButtonObject type to a ImageButton
  DVDMenu *pCurrentMenu = Global::pApp->getCurrentSubMenu ( );
  if ( ! pCurrentMenu )
    return;

  MenuPreview *pMenuPreview = pCurrentMenu->getMenuPreview ( );
  if ( ! pMenuPreview )
    return;

  MenuObject  *pNewObject = pMenuPreview->createImageObject ( true );
  if ( ! pNewObject )
    return;

  pMenuPreview->removeObject ( pNewObject );
  MenuObject *pOldObject = m_listNormalState[0];

  pNewObject->setRect        (  pOldObject->rect      ( ) );
  pNewObject->setModifiers   ( *pOldObject->modifiers ( ) );

  m_listNormalState.remove   ( pOldObject );
  m_listNormalState.append   ( pNewObject );

  ImageObject *pImage = (ImageObject *)pNewObject;
  pImage->updatePixmap ( );

  emit ( signalUpdatePixmap    ( ) );
  emit ( signalUpdateStructure ( ) );

  delete pOldObject;
}

void ButtonObject::slotToMovie ( ) // movie object for the Menu that is ...
{
  // Converts the current ButtonObject type to a MovieButton
  DVDMenu *pCurrentMenu = Global::pApp->getCurrentSubMenu ( );
  if ( ! pCurrentMenu )
    return;

  MenuPreview *pMenuPreview = pCurrentMenu->getMenuPreview ( );
  if ( ! pMenuPreview )
    return;

  MenuObject  *pNewObject = pMenuPreview->createMovieObject ( true );
  if ( ! pNewObject )
    return;

  pMenuPreview->removeObject ( pNewObject );
  MenuObject *pOldObject = m_listNormalState[0];

  pNewObject->setRect        (  pOldObject->rect      ( ) );
  pNewObject->setModifiers   ( *pOldObject->modifiers ( ) );

  m_listNormalState.remove   ( pOldObject );
  m_listNormalState.append   ( pNewObject );

  emit ( signalUpdatePixmap    ( ) );
  emit ( signalUpdateStructure ( ) );

  delete pOldObject;
}

void ButtonObject::setRect ( QRect &theRect )
{
  uint t;
  for ( t=0; t<m_listHighlightedState.count ( ); t++ )
    m_listHighlightedState[t]->setRect ( theRect );
  for ( t=0; t<m_listSelectedState.count ( ); t++ )
    m_listSelectedState[t]->setRect ( theRect );
  for ( t=0; t<m_listNormalState.count ( ); t++ )
    m_listNormalState[t]->setRect ( theRect );
  MenuObject::setRect ( theRect );
}

QRect &ButtonObject::rect ( )
{
  if ( m_listNormalState.count ( ) > 0 )
    return m_listNormalState[ 0 ]->rect ( );

  return m_rect;
}

AnimationAttribute *ButtonObject::getSpecificAttributes (long, QString)
{
  return NULL;
}

