/*
 * reportctrl.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

#include <wx/wx.h>
#include <wx/imaglist.h>
#include <wx/config.h>

#include "reportctrl.h"

wxxReportCtrl::wxxReportCtrl() : data(0), wxListCtrl()
{
}

wxxReportCtrl::wxxReportCtrl(wxWindow* parent, wxWindowID id, wxxReportData* data, wxImageList* images, 
						   const wxString& configGroup, const wxPoint& pos, 
						   const wxSize& size, const wxValidator& validator, const wxString& name) 
	: data(data), configGroup(configGroup), 
	  wxListCtrl(parent, id, pos, size, wxLC_REPORT | wxLC_SINGLE_SEL, validator, name)
{
	wxASSERT_MSG(data != 0, wxT("data must be given"));
	data->ctrl = this;
	Connect(GetId(), wxEVT_COMMAND_LIST_COL_CLICK, (wxObjectEventFunction) (wxEventFunction) (wxListEventFunction) &wxxReportCtrl::OnColumnClicked);

	SetImageList(images, wxIMAGE_LIST_SMALL);
	DoColumns(true);
	DoRows();
}

wxxReportCtrl::~wxxReportCtrl()
{
	StoreColumnWidths();
	delete data;
}

bool wxxReportCtrl::Create(wxWindow* parent, wxWindowID id, wxxReportData* data, wxImageList* images, 
						  const wxString& configGroup, const wxPoint& pos, 
						  const wxSize& size, const wxValidator& validator, const wxString& name)
{
	wxASSERT_MSG(data != 0, wxT("data must be given"));
	data->ctrl = this;

	delete this->data;
	this->data = data;
	this->configGroup = configGroup;
	bool ret = wxListCtrl::Create(parent, id, pos, size, wxLC_REPORT | wxLC_SINGLE_SEL, validator, name);
	Connect(GetId(), wxEVT_COMMAND_LIST_COL_CLICK, (wxObjectEventFunction) (wxEventFunction) (wxListEventFunction) &wxxReportCtrl::OnColumnClicked);

	SetImageList(images, wxIMAGE_LIST_SMALL);
	ret &= DoColumns(true);
	ret &= DoRows();

	return ret;
}

void wxxReportCtrl::SetData(wxxReportData* newData)
{
	delete data;
	data = newData;
	data->ctrl = this;
	DoColumns();
	DoRows();
}

long wxxReportCtrl::GetSelectedRow()
{
	long item = GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(item == -1)
		return item;
	else
		return GetItemData(item);
}

wxArrayLong wxxReportCtrl::GetSelectedRows()
{
	wxArrayLong ret;
	long item = -1;
	do{
		item = GetNextItem(item, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
		ret.Add(item);
	}while(item >= 0);
	
	for(int i = ret.GetCount() - 1; i >= 0; i--)
		ret[i] = GetItemData(item);

	return ret;
}

bool wxxReportCtrl::DoColumns(bool initial)
{
	if(!data) return false;
	if(!initial) 
	{
		StoreColumnWidths();
		ClearAll();
	}
	int numCol = data->GetColumnCount();
	for(int i = 0; i < numCol; i++)
		InsertColumn(i, data->GetColumnTitle(i), data->GetColumnAlign(i), data->GetColumnWidth(i));
	return ReadColumnWidths();
}

bool wxxReportCtrl::DoRows(int start, int end)
{
	if(end < start) // do all
	{
		DeleteAllItems();
		int todo = data->GetRowCount();
		int cols = data->GetColumnCount();
		bool hidden = !IsShown();
		if(todo > 100)
			Show(false);
		for(int i = 0; i < todo; i++)
		{
			long tmp = InsertItem(i, data->GetItemText(i, 0), data->GetImage(i));
			SetItemData(tmp, i);
			for(int j = 1; j < cols; j++)
				SetItem(i, j, data->GetItemText(i, j));
		}
		if(!hidden) Show(true);
	}
	else
	{
		int cnt = GetItemCount();
		long cols = data->GetColumnCount();
		for(int i = 0; i < cnt; i++)
		{
			long idata = GetItemData(i);
			if((idata >= start) && (idata <= end))
			{
				SetItem(i, 0, data->GetItemText(idata, 0), data->GetImage(idata));
				for(int j = 1; j < cols; j++)
					SetItem(i, j, data->GetItemText(idata, j));
			}
		}
			
	}
	return true;
}

bool wxxReportCtrl::StoreColumnWidths()
{
	// if there's no config group, the class is configured not to store any data. 
	if(configGroup.IsEmpty())
		return true;
	// try to store now
	wxConfigBase* cfg = wxConfig::Get();
	if(!cfg)
		return false;
	wxString oldPath = cfg->GetPath();
	cfg->SetPath(configGroup);

	bool ret = true;
	for(int i = 0; i < data->GetColumnCount(); i++)
	{
		int w = GetColumnWidth(i);
		// on some occasions, GetColumnWidth() returns odd values, so we've got to be careful
		if((w >= wxREPORT_AUTOSIZE_USEHEADER) && (w != 0) && (w < 500))
			ret &= cfg->Write(wxT("W") + data->GetColumnTitle(i), (long)w);
	}

	cfg->SetPath(oldPath);
	return ret;
}

bool wxxReportCtrl::ReadColumnWidths()
{
	// if there's no config group, the class is configured not to store any data. 
	if(configGroup.IsEmpty())
		return true;
	// try to read now
	wxConfigBase* cfg = wxConfig::Get();
	if(!cfg)
		return false;
	wxString oldPath = cfg->GetPath();
	cfg->SetPath(configGroup);

	long w = 0;
	for(int i = 0; i < data->GetColumnCount(); i++)
	{
		if(cfg->Read(wxT("W") + data->GetColumnTitle(i), &w))
			// there's sometimes odd data in the config files
			if((w >= wxREPORT_AUTOSIZE_USEHEADER) && (w != 0) && (w < 500))
				SetColumnWidth(i, (int)w);
	}

	cfg->SetPath(oldPath);
	return true;
}

///////////////// event handlers ////////////

int wxCALLBACK wxReportCtrlCompareFunction(long v1, long v2, long data)
{
	wxxReportData* ptr = ((wxxReportCtrl*) data)->data;
	return ptr->CompareItems(v1, v2);
}

void wxxReportCtrl::OnColumnClicked(wxListEvent& evt)
{
	int clicked = evt.GetColumn();

	if(clicked == data->GetSortColumn()) // reverse order
		data->SetSortOrder((wxxReportSortOrder)(data->GetSortOrder() * -1));
	else
		data->SetSortOrder(wxREPORT_ASCENDING);
	data->SetSortColumn(clicked);
	data->SortItems();
}
