/*
 * robhtml.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// Implements HTML functions (creating tournament HTMLs) 

#include "robhtml.h"

#include "robutils.h"
#include "robtrans.h"

#include <rtsystem.h>
#include <rtini.h>
#include <rtfile.h>
#include <rtmath.h>
#include <rtchar.h>
#include <rtresource.h>
#include <rtstreams.h>

using namespace lrt;

namespace rt {

String HtmlCreator::defaultChartsIniName = "charts.xxx";
String HtmlCreator::defaultChartsHtmlName = "charts.html";
String HtmlCreator::cssName = "charts.css";

HtmlCreator::HtmlCreator(const String& competitionName, const File& competitionFolder,
	const File& outputFolder, bool createLinks, bool createDownloadLinks) : 
	competitionName(competitionName), competitionFolder(competitionFolder),
	outputFolder(outputFolder), createLinks(createLinks), createDownloadLinks(
	createDownloadLinks), myManager(false, Math::MAX_INT, competitionFolder.getName()),
	chartsHtmlName(defaultChartsHtmlName)
{
}

File HtmlCreator::getDetailedHtml(const String& botFileName)
{
	return File(outputFolder, botFileName + ".html");
}

bool HtmlCreator::createChartsHtml(bool recalc, const String& iniName, const String& htmlName)
{
	chartsHtmlName = htmlName;

	// first check if this can succeed (charts ini must be present of course)
	String chartsFileName(File(competitionFolder, iniName).getName());
	if(!File(chartsFileName).exists())
		return false; // no charts ini found

	if(!cssExists())
		if(!writeCss())
			System::println(_("warning: couldn't write CSS file"));

	Vector<TopTourResult> store(0);

	// read the charts
	if(!recalc) 
	{
		IniFile charts;
		charts.read(chartsFileName);
		myManager.readCharts(charts, store);
		store.sort(TopTourResult::compareByPoints);
	}
	else // calc the charts
	{
		IniFile charts;
		charts.read(chartsFileName);
		myManager.calcCharts(charts, store);
		// is already sorted now
		NoBreakSection nbs; // protect new charts ini
			myManager.checkChartsLength(charts, store);
			myManager.updateCharts(charts, store);
			charts.write(chartsFileName);
		nbs.leave();
		// now finished updating, lets go html 
		store.clear();
		myManager.readCharts(charts, store);
		store.sort(TopTourResult::compareByPoints);
	}

	if(store.length() == 0)
		System::println(_("warning: empty charts or read error?"));

	// create the HTML
	NoBreakSection nbsProtectHtml; // create complete HTML always

	OutputStream* html = new FileOutputStream(File(outputFolder, htmlName));
	if(html->fail())
	{ delete html; return false; }

	html->write("<html>\n<head>\n <title>" + toHtml(_("RoboTour Charts")) + "</title>\n");
	html->write(" <link rel=\"stylesheet\" href=\"" + cssName + "\">\n");
	html->write("</head>\n\n");
	html->write("<body class=\"bodybg\">\n");
	html->write("<h1>" + toHtml(_("RoboTour Charts")) + "</h1>\n");
	html->write("<h3>" + toHtml(competitionName) + "</h3>\n");
	html->write("<table border=0 cellspacing=2 cellpadding=2 width=\"100%\">\n");
	html->write("<tr class=\"title\">\n");

	html->write("  <td>\n" );
	html->write("    &nbsp;\n" );
	html->write("  </td>\n" );
	html->write("  <td>\n" );
	html->write("   " + toHtml(_("Name")) + "<br>\n" );
	html->write("   <small>" + toHtml(_("Last Updated")) + "</small>\n" );
	html->write("  </td>\n" );
	html->write("  <td>\n" );
	html->write("   " + toHtml(_("Author")) + ",<br>" + toHtml(_("Country")) + "\n" );
	html->write("  </td>\n" );
	html->write("  <td align=center width=50>\n" );
	html->write("   " + toHtml(_("Wins")) + "\n" );
	html->write("   </td>\n" );
	html->write("  <td align=center width=50>\n" );
	html->write("   " + toHtml(_("Losses")) + "\n" );
	html->write("  </td>\n" );
	html->write("  <td align=center width=50>\n" );
	html->write("   " + toHtml(_("Ties")) + "\n" );
	html->write("  </td>\n" );
	html->write("  <td align=center width=100>\n" );
	html->write("   " + toHtml(_("Score")) + "\n" );
	html->write("  </td>\n" );
	html->write(" </tr>\n" );

	for(int b = 0; b < store.length(); b++)
	{
		TopTourResult& res = store[b];
		String filename    = File(competitionFolder, res.fileName).
			getLocalName(outputFolder).replace(File::separatorChar, '/');
		String programname = toHtml(res.headers["name"].value);
		String author      = toHtml(res.headers["author"].value);
		String country     = toHtml(res.headers["country"].value);
		String wins        = toHtml(String(res.wins));
		String looses      = toHtml(String(res.looses));
		String ties        = toHtml(String(res.ties));
		String points      = toHtml(String(res.points));
		String date        = toHtml(res.date.toString());
		String dethtml	   = getDetailedHtml(res.fileName).
			getLocalName(outputFolder).replace(File::separatorChar, '/');

		html->write(" <tr class=\"topcur\">\n" );
		html->write("  <td>\n" );
		html->write(String("    ") + (b+1) + ".\n");
		html->write("  </td>\n" );
		html->write("  <td>\n" );
		if(createLinks)
		  html->write("   <a href=\"" + dethtml + "\">" + programname + "</a>\n");
		else
		  html->write("   " + programname + "\n");

		if(createDownloadLinks)
		  html->write("   <br><small>" + date + "&nbsp;&nbsp;<a href=\"" + filename + "\">" 
		  	  + toHtml(_("Download")) + "</a></small>\n");
		else
		  html->write("   <br><small>" + date + "&nbsp;&nbsp;</small>\n");
		html->write("  </td>\n" );
		html->write("  <td>\n" );
		html->write("   " +author+ ",<br>" +country+ "\n");
		html->write("  </td>\n" );
		html->write("  <td align=center width=50>\n" );
		html->write("   " +wins+ "\n");
		html->write("   </td>\n" );
		html->write("  <td align=center width=50>\n" );
		html->write("   " +looses+ "\n");
		html->write("  </td>\n" );
		html->write("  <td align=center width=50>\n" );
		html->write("   " +ties+ "\n");
		html->write("  </td>\n" );
		html->write("  <td align=center width=100>\n" );
		html->write("   " +points+ "\n");
		html->write("  </td>\n" );
		html->write(" </tr>\n" );
	}

	html->write("</table>\n");
	html->write("</body>\n");
	html->write("</html>");

	html->close();
	delete html;

	return true;
}

bool HtmlCreator::createDetailedHtml(const String& botFileName)
{
	if(!cssExists())
		if(!writeCss())
			System::println(_("warning: couldn't write CSS file"));

	String infoFile = myManager.getInfoFileName(botFileName);
	TopInfo info(infoFile);
	info.read();

	// link to the bot file
	String botLink = File(competitionFolder, botFileName).
		getLocalName(outputFolder).replace(File::separatorChar, '/');
	// link to the main charts html page
	String chartsLink = File(outputFolder, chartsHtmlName).
		getLocalName(outputFolder).replace(File::separatorChar, '/');

	// create the HTML
	NoBreakSection nbsProtectHtml; // create complete HTML always

	OutputStream* html = new FileOutputStream(getDetailedHtml(botFileName));
	if(html->fail())
	{ delete html; return false; }

	html->write("<html>\n<head>\n <title>" + toHtml(_("RoboTour Detailed Simulation Results")) + "</title>\n");
	html->write(" <link rel=\"stylesheet\" href=\"" + cssName + "\">\n");
	html->write("</head>\n\n");
	html->write("<body class=\"bodybg\">\n");

	html->write("<H1>" + toHtml(_("Detailed Simulation Results for ")) + toHtml(info.headers["name"].value) + "</H1>\n\n");

	html->write("<H3>" + toHtml(competitionName) + "</H3>\n\n" );

	html->write("<table WIDTH=\"100%\">\n");
	html->write(" <tr>" );
	html->write("  <td class=\"detleft\" WIDTH=\"70%\">\n");
	html->write("   <BIG><B>" + toHtml(info.headers["name"].value) + "</B></BIG><br>\n");
	html->write("   " + toHtml(info.headers["author"].value) + "<br>\n");
	html->write("   " + toHtml(info.headers["country"].value) + "<br>\n");
	html->write("  </td>\n" );
	html->write("  <td class=\"detright\">" );
	html->write("   <small>" + toHtml(_("last changed:")) + "<br>\n" );
	html->write("   " + toHtml(info.date.toString()) + "<br>&nbsp;<br>\n");
	if(createDownloadLinks)
	  html->write("   <a href=\"" + botLink + "\">" + toHtml(_("Download")) + " " + toHtml(botLink) + "</a></small>\n");
	else
	  html->write("   " + toHtml(_("File name:")) + " " + toHtml(botLink) + "</small>\n");

	html->write("  </td>" );
	html->write(" </tr>" );
	html->write("</table>\n\n" );

	// big table
	html->write("<table border=\"0\" cellspacing=\"2\" cellpadding=\"2\" width=\"100%\">\n");
	html->write(" <tr class=\"title\">\n" );
	html->write("  <td>\n" );
	html->write("   " + toHtml(_("Opponent")) + "<br>\n" );
	html->write("   <small>" + toHtml(_("Last updated")) + "</small>\n" );
	html->write("  </td>\n" );
	html->write("  <td>\n" );
	html->write("   " + toHtml(_("Author")) + ",<br>" + toHtml(_("Country")) + "\n" );
	html->write("  <td align=center width=50>\n" );
	html->write("   " + toHtml(_("Wins")) + "\n" );
	html->write("   </td>\n" );
	html->write("  <td align=center width=50>\n" );
	html->write("   " + toHtml(_("Losses")) + "\n" );
	html->write("  </td>\n" );
	html->write("  <td align=center width=50>\n" );
	html->write("   " + toHtml(_("Ties")) + "\n" );
	html->write("  </td>\n" );
	html->write(" </tr>\n" );

	Vector<TopTourResult> store(0);
	readDetailed(botFileName, store);
	store.sort(TopTourResult::compareByDate);

	for(int o = 0; o < store.length(); o++)
	{
		TopTourResult& res = store[o];
		String filename    = File(competitionFolder, res.fileName).
			getLocalName(outputFolder).replace(File::separatorChar, '/');
		String programname = toHtml(res.headers["name"].value);
		String author      = toHtml(res.headers["author"].value);
		String country     = toHtml(res.headers["country"].value);
		String wins        = toHtml(String(res.wins));
		String looses      = toHtml(String(res.looses));
		String ties        = toHtml(String(res.ties));
		String points      = toHtml(String(res.points));
		String date        = toHtml(res.date.toString());
		String dethtml	   = getDetailedHtml(res.fileName).
			getLocalName(outputFolder).replace(File::separatorChar, '/');

		String cssClass;
		if(res.top == "yes")
		{
			if(res.id == res.infoId) // is bot played version (id) == bot current version (infoId)?
				cssClass = "topcur";
			else
				cssClass = "topold";
		}
		else
		{
			if(res.id == res.infoId)
				cssClass = "badcur";
			else
				cssClass = "badold";
		}

		html->write(" <tr class=\"" + cssClass + "\">\n");
		html->write("  <td>\n" );
		if(createLinks)
			html->write("   <a href=\"" + dethtml + "\">" + programname + "</a><br>\n");
		else
			html->write("   " + programname + "<br>\n");
		if(createDownloadLinks)
			html->write("   <small>"+ date + "&nbsp;&nbsp;<a href=\""+ filename +"\">" 
			    + toHtml(_("Download")) + "</a></small>\n");
		else
			html->write("   <small>"+ date + "</small>\n");
		html->write("  </td>\n" );
		html->write("  <td>\n" );
		html->write("   " + author + ",<br>" + country + "\n");
		html->write("  </td>\n" );
		html->write("  <td align=center width=50>\n" );
		html->write("   " + wins + "\n");
		html->write("   </td>\n" );
		html->write("  <td align=center width=50>\n" );
		html->write("   " + looses + "\n");
		html->write("  </td>\n" );
		html->write("  <td align=center width=50>\n" );
		html->write("   " + ties + "\n");
		html->write("  </td>\n" );
		html->write(" </tr>\n" );
	}

	// print legend
	html->write("</table>\n");
	html->write("<a href=\"" + chartsLink + "\">" + toHtml(_("Back to the charts")) + "</a><br>\n\n");
	html->write("<h3>" + toHtml(_("Legend:")) + "</h3>\n");
	html->write("<table cellpadding=2 cellspacing=0 border=0>\n");
	html->write("<tr><td class=\"topcur\">\n");
	html->write("  " + toHtml(_("Opponent is in the charts")) + "\n");
	html->write("</td></tr>\n");
	html->write("<tr><td class=\"topold\">\n");
	html->write("  " + toHtml(_("Former version of a charts bot")) + "\n");
	html->write("</td></tr>\n");
	html->write("<tr><td class=\"badcur\">\n");
	html->write("  " + toHtml(_("Opponent isn't good enough for the charts")) + "\n");
	html->write("</td></tr>\n");
	html->write("<tr><td class=\"badold\">\n");
	html->write("  " + toHtml(_("Former version of a bot, which is not in the charts")) + "\n");
	html->write("</td></tr>\n");
	html->write("</table>\n");
	html->write("</body>\n");
	html->write("</html>\n");

	html->close();
	delete html;

	return true;
}

bool HtmlCreator::readDetailed(const String& botFileName, Vector<TopTourResult>& store)
{
	store.clear();

	String botFile = File(competitionFolder, botFileName).getName();
	String resFile = myManager.getResultFileName(botFile);
	IniFile results;
	if(!results.read(resFile))
		System::println(String::format(_("warning: result file %s not found"), resFile.cStr()));


	IniFile::Iterator iter = results.begin();
	while(iter.hasElement())
	{
		const IniSection& sec = iter.get().getValue();
		String infoFile = myManager.getInfoFileName(iter.get().getKey());
		TopInfo info(infoFile);
		info.read();
		TopTourResult res(info.headers, iter.get().getKey());
		res.wins   = sec["wins"].intValue(0);
		res.looses = sec["looses"].intValue(0);
		res.ties   = sec["ties"].intValue(0);
		res.id     = sec["id"].intValue(0);
		res.date = info.date;
		res.top = info.top;
		res.infoId = info.id.intValue(0);
		store += res;
		++iter;
	}

	return true;
}

String HtmlCreator::toHtml(const String& text)
{
	String ret;

	int tlen = text.length();
	const char* data = text.cStr();
	for(int t = 0; t < tlen; t++)
	{
		char c = data[t];
		if(Char::isLetter(c) || Char::isDigit(c) ||
			(c==' ') || (c=='.') || (c==',')) // char is safe
			ret += c;
		else 
		{
			ret += "&#";
			ret += String(((int)((unsigned char)c)));
			ret += ';';
		}
	}

	return ret;
}

bool HtmlCreator::writeCss()
{
	InputStream* cssRes = Resource::getResource(cssName);
	if((cssRes == 0) || (cssRes->fail())) // not found
		return false;

	NoBreakSection nbsProtectCss; // write complete CSS always

	OutputStream* cssOut = new FileOutputStream(File(outputFolder, cssName));
	if(cssOut->fail()) // can't write
		return false;
	// copy stream
	int c;
	while((c = cssRes->read()) >= 0)
		cssOut->write(c);
	delete cssRes;
	delete cssOut;

	return true;
}

bool HtmlCreator::cssExists()
{
	return File(outputFolder, cssName).exists();
}


} // namespace

