/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/

#if !defined (__DVD_H_INSIDE__) && !defined (LIBDVD_COMPILATION)
#error "Only <dvd.h> can be included directly."
#endif

#ifndef __DVD_DECODER_H__
#define __DVD_DECODER_H__


G_BEGIN_DECLS


/* --- type macros --- */
#define DVD_DECODER_TYPE		(dvd_decoder_get_type())
#define DVD_DECODER(o)			(G_TYPE_CHECK_INSTANCE_CAST (o, DVD_DECODER_TYPE, DvdDecoder))
#define DVD_DECODER_CLASS(k)		(G_TYPE_CHECK_CLASS_CAST    (k, DVD_DECODER_TYPE, DvdDecoderClass))
#define DVD_IS_DVD_DECODER(o)		(G_TYPE_CHECK_INSTANCE_TYPE (o, DVD_DECODER_TYPE))
#define DVD_IS_DVD_DECODER_CLASS(k)	(G_TYPE_CHECK_CLASS_TYPE    (k, DVD_DECODER_TYPE))

/* --- typedefs & structures --- */
typedef struct	_DvdDecoder		DvdDecoder;
typedef struct	_DvdDecoderClass	DvdDecoderClass;
typedef enum	_DvdDecoderState	DvdDecoderState;

enum _DvdDecoderState {
	DVD_DECODER_STATE_STARTCODE,
	DVD_DECODER_STATE_HEADER,
	DVD_DECODER_STATE_FRAME
};

struct _DvdDecoder {
	GObject	 parent;
	
	DvdStream	 stream_type;
	guint8		 track;
	DvdDecoderState	 state;
	guint32		 current_pts;	/* current presentation time stamp (90kHz) */
	gboolean	 have_start_pts;
	guint32		 start_pts;	/* presentation time stamp (90kHz) of first output */
	guint64		 frame_clocks;
	guint32		 frames;
	guint32		 buffer_size;
	guint8		*buffer;
	guint32		 buffer_pos;
	
	void	(*read_startcode)	(DvdDecoder	 *decoder,
					 guint8		**start,
					 guint8		 *end,
					 guint32	  pts);
	void	(*read_header)		(DvdDecoder	 *decoder,
					 guint8		**start,
					 guint8		 *end,
					 guint32	  pts);
	void	(*read_frame)		(DvdDecoder	 *decoder,
					 guint8		**start,
					 guint8		 *end,
					 guint32	  pts);
};


struct _DvdDecoderClass {
	GObjectClass	parent;
	/**
	 * output_data Signal outputs full frames.
	 * Bytes = -1 signals end of data stream.
	 * @param decoder Pointer to #DvdDecoder emitting signal
	 * @param stream_type #DvdStream data type
	 * @param track Track data belongs to starting at 0
	 * @param bytes Frame size in bytes
	 * @param frame_buffer Full frame buffer (including headers)
	 * @param pts Current presentation time stamp (90kHz pts)
	 * @returns void
	 */
	void	(*output_frame)		(DvdDecoder	*decoder,
					 DvdStream	 stream_type,
					 gint		 track,
					 gint		 bytes,
					 guint8		*frame_buffer,
					 guint32	 pts,
					 guint64	 frame_clocks);
};

/* --- prototypes --- */
GType		dvd_decoder_get_type		(void);
DvdDecoder*	dvd_decoder_new			(DvdStream	  stream_type);
DvdStream	dvd_decoder_get_stream_type	(DvdDecoder	 *decoder);
void		dvd_decoder_set_track		(DvdDecoder	 *decoder,
						 guint8		  track);
guint8		dvd_decoder_get_track		(DvdDecoder	 *decoder);
guint32		dvd_decoder_get_start_pts	(DvdDecoder	 *decoder);
guint32		dvd_decoder_get_current_pts	(DvdDecoder	 *decoder);
guint16		dvd_decoder_get_frames		(DvdDecoder	 *decoder);
DvdDecoderState dvd_decoder_get_state		(DvdDecoder	 *decoder);
void		dvd_decoder_write		(DvdDecoder	 *decoder,
						 guint8		 *data,
						 guint16	  length,
						 guint32	  pts);
/* The following fuctions used only by dvd-decoder-* objects */
void		dvd_decoder_set_start_pts	(DvdDecoder	 *decoder,
						 guint32	  start_pts);
void		dvd_decoder_set_current_pts	(DvdDecoder	 *decoder,
						 guint32	  current_pts);
void		dvd_decoder_add_to_current_pts	(DvdDecoder	 *decoder,
						 guint32	  add_pts);
void		dvd_decoder_set_frame_clocks	(DvdDecoder	 *decoder,
						 guint64	  frame_clocks);
void		dvd_decoder_set_state		(DvdDecoder	 *decoder,
						 DvdDecoderState  state);
void		dvd_decoder_output_frame	(DvdDecoder	 *decoder);
void		dvd_decoder_discard_frame	(DvdDecoder	 *decoder);
void		dvd_decoder_discard_bytes	(DvdDecoder	 *decoder,
						 guint16	  length);
void		dvd_decoder_ensure_buffer_size	(DvdDecoder	 *decoder,
						 guint32	  buffer_size);
void		dvd_decoder_write_frame_byte	(DvdDecoder	 *decoder,
						 guint8		  byte);
void		dvd_decoder_write_frame_bytes	(DvdDecoder	 *decoder,
						 const guint8	 *bytes,
						 guint32	  length);
G_END_DECLS

#endif /*__DVD_DECODER_H__*/
