/*  smplayer, GUI front-end for mplayer.
    Copyright (C) 2007 Ricardo Villalba <rvm@escomposlinux.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "defaultgui.h"
#include "helper.h"
#include "core.h"
#include "global.h"
#include "timeslider.h"
#include "playlist.h"
#include "mplayerwindow.h"
#include "floatingcontrol.h"
#include "myaction.h"
#include <qsettings.h>

#include <qmainwindow.h>
#include <qlabel.h>
#include <qstatusbar.h>
#include <qmenubar.h>
#include "myaction.h"
#include <qtooltip.h>
#include <qpushbutton.h>
#include <qtoolbutton.h>
#include <qlcdnumber.h>
#include <qtextstream.h>
#include <qdockarea.h>
#include "desktopinfo.h"
#include "qt3_4_compat.h"

DefaultGui::DefaultGui( QStringList files, QWidget * parent, const char* name, 
                        WFlags fl )
	: BaseGuiPlus( files, parent, name, fl )
{
	floating_control_width = 100; //%

	createStatusBar();

	connect( this, SIGNAL(timeChanged(double, int, QString)),
             this, SLOT(displayTime(double, int, QString)) );
    connect( this, SIGNAL(frameChanged(int)),
             this, SLOT(displayFrame(int)) );

	connect( this, SIGNAL(cursorNearBottom(QPoint)), 
             this, SLOT(showFloatingControl(QPoint)) );
	connect( this, SIGNAL(cursorNearTop(QPoint)), 
             this, SLOT(showFloatingMenu(QPoint)) );
	connect( this, SIGNAL(cursorFarEdges()), 
             this, SLOT(hideFloatingControls()) );

	createMainToolBars();
	createActions();
    createControlWidget();
    createControlWidgetMini();
	createFloatingControl();
	createMenus();

#if DOCK_PLAYLIST
	// Playlistdock
	playlistdock = new QDockWindow(QDockWindow::InDock, this, "playlist" );
	playlist->reparent(playlistdock, QPoint(0,0), FALSE);
	playlistdock->setWidget( playlist );
	playlistdock->setResizeEnabled(true);
	playlistdock->setMovingEnabled(false);
	playlistdock->setCloseMode(QDockWindow::Always);
	playlistdock->setNewLine(true);
	playlistdock->setVerticallyStretchable(TRUE);
	playlistdock->setHorizontallyStretchable(TRUE);
	playlistdock->setOffset(0);

	playlistdock->setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Preferred );

	leftDock()->setAcceptDockWindow( playlistdock, false);
	rightDock()->setAcceptDockWindow( playlistdock, false);
	topDock()->setAcceptDockWindow( playlistdock, false);

	addDockWindow(playlistdock, Qt::DockBottom );
	playlistdock->hide();

	connect( playlistdock, SIGNAL(visibilityChanged(bool)),
             this, SLOT(playlistdockVisibilityChanged(bool)) );
#endif

	languageChange();

	loadConfig();

	if (pref->compact_mode) {
		controlwidget->hide();
		toolbar1->hide();
		toolbar2->hide();
	}
}

DefaultGui::~DefaultGui() {
	saveConfig();

#if DOCK_PLAYLIST
	playlist->reparent(this, QPoint(0,0), FALSE);
#endif
}

/*
void DefaultGui::closeEvent( QCloseEvent * e ) {
	BaseGuiPlus::closeEvent(e);
}
*/

void DefaultGui::createActions() {
	showMainToolbarAct = new MyAction(Qt::Key_F5, this, "show_main_toolbar" );
	showMainToolbarAct->setToggleAction(true);
	connect( showMainToolbarAct, SIGNAL(toggled(bool)),
             this, SLOT(showMainToolbar(bool)) );

	showLanguageToolbarAct = new MyAction(Qt::Key_F6, this, "show_language_toolbar" );
	showLanguageToolbarAct->setToggleAction(true);
	connect( showLanguageToolbarAct, SIGNAL(toggled(bool)),
             this, SLOT(showLanguageToolbar(bool)) );

	/*
	toolbarAct = new QActionGroup(this, "toolbars");
	toolbarAct->setUsesDropDown(true);
	toolbarAct->setExclusive(false);
	toolbarAct->add( showMainToolbarAct );
	toolbarAct->add( showLanguageToolbarAct );
	*/
}

void DefaultGui::createMenus() {
	toolbar_menu = new QPopupMenu(this, "toolbar_menu");
	showMainToolbarAct->addTo( toolbar_menu );
	showLanguageToolbarAct->addTo( toolbar_menu );
	
	optionsMenu->insertSeparator();
	toolbar_menu_id = optionsMenu->insertItem( "Toolbars", toolbar_menu );
}

void DefaultGui::createMainToolBars() {
	toolbar1 = new QToolBar( this, "toolbar1" );
	toolbar1->setMovingEnabled(false);
	openFileAct->addTo( toolbar1 );
	openDVDAct->addTo( toolbar1 );
	openURLAct->addTo( toolbar1 );
	toolbar1->addSeparator();
	compactAct->addTo( toolbar1 );
	fullscreenAct->addTo( toolbar1 );
	toolbar1->addSeparator();
	screenshotAct->addTo( toolbar1 );
	toolbar1->addSeparator();
	showPropertiesAct->addTo( toolbar1 );
	showPlaylistAct->addTo( toolbar1 );
	showPreferencesAct->addTo( toolbar1 );
	toolbar1->addSeparator();
	playPrevAct->addTo( toolbar1 );
	playNextAct->addTo( toolbar1 );

	toolbar2 = new QToolBar( this, "toolbar2" );
	toolbar2->setMovingEnabled(false);
	select_audio = new QPushButton( toolbar2, "select_audio" );
	select_audio->setPopup( (QPopupMenu *) audiotrack_menu );

	select_subtitle = new QPushButton( toolbar2, "select_subtitle" );
	select_subtitle->setPopup( (QPopupMenu *) subtitlestrack_menu );

	/*
	toolbar1->show();
	toolbar2->show();
	*/
}


void DefaultGui::createControlWidgetMini() {
	controlwidget_mini = new QToolBar( this, "controlwidget" );
	controlwidget_mini->setResizeEnabled(false);
	controlwidget_mini->setMovingEnabled(false);
	addDockWindow(controlwidget_mini, Qt::DockBottom );

	playOrPauseAct->addTo( controlwidget_mini );
	stopAct->addTo( controlwidget_mini );
	controlwidget_mini->addSeparator();

	rewind1Act->addTo( controlwidget_mini );

	timeslider_mini = new TimeSlider( controlwidget_mini, "timeslider_mini");
	connect( timeslider_mini, SIGNAL( posChanged(int) ), 
             core, SLOT(goToPos(int)) );
	connect( timeslider_mini, SIGNAL( draggingPos(int) ), 
             this, SLOT(displayGotoTime(int)) );
	controlwidget_mini->setStretchableWidget( timeslider_mini );

	forward1Act->addTo( controlwidget_mini );

	controlwidget_mini->addSeparator();

	muteAct->addTo( controlwidget_mini );
	volumeslider_mini = new MySlider( controlwidget_mini, "volumeslider_mini" );
	volumeslider_mini->setOrientation( Qt::Horizontal );
	volumeslider_mini->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
	volumeslider_mini->setFocusPolicy( NOFOCUS );
	volumeslider_mini->setTickmarks( QSlider::Below );
	volumeslider_mini->setTickInterval( 10 );
	volumeslider_mini->setLineStep( 1 );
	volumeslider_mini->setPageStep( 10 );
	connect( volumeslider_mini, SIGNAL( valueChanged(int) ), 
             core, SLOT( setVolume(int) ) );
	connect( core, SIGNAL(volumeChanged(int)),
             volumeslider_mini, SLOT(setValue(int)) );

	controlwidget_mini->hide();
}

void DefaultGui::createControlWidget() {
	controlwidget = new QToolBar( this, "controlwidget" );
	controlwidget->setResizeEnabled(false);
	controlwidget->setMovingEnabled(false);
	addDockWindow(controlwidget, Qt::DockBottom );

	playAct->addTo( controlwidget );
	pauseAndStepAct->addTo( controlwidget );
	stopAct->addTo( controlwidget );

	controlwidget->addSeparator();

	rewind3Act->addTo( controlwidget );
	rewind2Act->addTo( controlwidget );
	rewind1Act->addTo( controlwidget );

	timeslider = new TimeSlider( controlwidget, "timeslider");
	connect( timeslider, SIGNAL( posChanged(int) ), 
             core, SLOT(goToPos(int)) );
	connect( timeslider, SIGNAL( draggingPos(int) ), 
             this, SLOT(displayGotoTime(int)) );
	controlwidget->setStretchableWidget( timeslider );

	forward1Act->addTo( controlwidget );
	forward2Act->addTo( controlwidget );
	forward3Act->addTo( controlwidget );

	controlwidget->addSeparator();

	fullscreenAct->addTo( controlwidget );
	muteAct->addTo( controlwidget );

	volumeslider = new MySlider( controlwidget, "volumeslider" );
	volumeslider->setOrientation( Qt::Horizontal );
	volumeslider->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
	volumeslider->setFocusPolicy( NOFOCUS );
	volumeslider->setTickmarks( QSlider::Below );
	volumeslider->setTickInterval( 10 );
	volumeslider->setLineStep( 1 );
	volumeslider->setPageStep( 10 );
	connect( volumeslider, SIGNAL( valueChanged(int) ), 
             core, SLOT( setVolume(int) ) );
	connect( core, SIGNAL(volumeChanged(int)),
             volumeslider, SLOT(setValue(int)) );

	/*
	controlwidget->show();
	*/
}

void DefaultGui::createFloatingControl() {
	floating_control = new FloatingControl(this, "floating_control" );

	connect( floating_control->rewind3, SIGNAL(clicked()),
             core, SLOT(fastrewind()) );
	connect( floating_control->rewind2, SIGNAL(clicked()),
             core, SLOT(rewind()) );
	connect( floating_control->rewind1, SIGNAL(clicked()),
             core, SLOT(srewind()) );

	connect( floating_control->forward1, SIGNAL(clicked()),
             core, SLOT(sforward()) );
	connect( floating_control->forward2, SIGNAL(clicked()),
             core, SLOT(forward()) );
	connect( floating_control->forward3, SIGNAL(clicked()),
             core, SLOT(fastforward()) );

	connect( floating_control->play, SIGNAL(clicked()),
             core, SLOT(play()) );
	connect( floating_control->pause, SIGNAL(clicked()),
             core, SLOT(pause_and_frame_step()) );
	connect( floating_control->stop, SIGNAL(clicked()),
             core, SLOT(stop()) );

	connect( floating_control->mute, SIGNAL(toggled(bool)),
             core, SLOT(mute(bool)) );

	connect( floating_control->fullscreen, SIGNAL(toggled(bool)),
             this, SLOT(toggleFullscreen(bool)) );

	connect( floating_control->volume, SIGNAL( valueChanged(int) ), 
             core, SLOT( setVolume(int) ) );
	connect( core, SIGNAL(volumeChanged(int)),
             floating_control->volume, SLOT(setValue(int)) );

	connect( floating_control->time, SIGNAL( posChanged(int) ), 
             core, SLOT(goToPos(int)) );
	connect( floating_control->time, SIGNAL( draggingPos(int) ), 
             this, SLOT(displayGotoTime(int)) );
}

void DefaultGui::createStatusBar() {
	qDebug("DefaultGui::createStatusBar");

	time_display = new QLabel( statusBar() );
	time_display->setAlignment(Qt::AlignRight);

	frame_display = new QLabel( statusBar() );
	frame_display->setAlignment(Qt::AlignRight);
    #if QT_VERSION >= 0x040000
	statusBar()->setAutoFillBackground(TRUE);
	#endif
	statusBar()->setPaletteBackgroundColor( QColor(0,0,0) );
	statusBar()->setPaletteForegroundColor( QColor(255,255,255) );
	time_display->setPaletteBackgroundColor( QColor(0,0,0) );
	time_display->setPaletteForegroundColor( QColor(255,255,255) );
	frame_display->setPaletteBackgroundColor( QColor(0,0,0) );
	frame_display->setPaletteForegroundColor( QColor(255,255,255) );
	statusBar()->setSizeGripEnabled(FALSE);

    statusBar()->message( tr("Welcome to SMPlayer") );
	statusBar()->addWidget( frame_display, 0, TRUE );
	frame_display->setText( " 0 " );

    statusBar()->addWidget( time_display, 0, TRUE );
	time_display->setText("00:00:00 / 00:00:00");

	time_display->show();
	frame_display->hide();
}

void DefaultGui::languageChange() {
	BaseGuiPlus::languageChange();

	showMainToolbarAct->change( tr("&Main toolbar") );
	showLanguageToolbarAct->change( tr("&Language toolbar") );
	//toolbarAct->setMenuText( tr("&Toolbars") );

	optionsMenu->changeItem( toolbar_menu_id, tr("&Toolbars") );

	QToolTip::add( volumeslider, tr("Volume") );
	QToolTip::add( volumeslider_mini, tr("Volume") );

	select_audio->setText( tr("Audio") );
	select_subtitle->setText( tr("Subtitle") );

#if DOCK_PLAYLIST
    playlistdock->setCaption( tr("Playlist") );
#endif
}

void DefaultGui::showPlaylist(bool b) {
	qDebug("DefaultGui::showPlaylist");
#if !DOCK_PLAYLIST
	BaseGuiPlus::showPlaylist(b);
#else
    if ( !b ) {
        playlistdock->hide();
		resize( width(), height() - playlistdock->height() );
    } else {
        exitFullscreenIfNeeded();
        playlistdock->show();
		//playlistdock->setFixedExtentWidth( width() );
		int new_height = height() + playlistdock->height();
		if (new_height > desktop_size().height()) 
			new_height = desktop_size().height() - 20;
		resize( width(), new_height );
    }
    updateWidgets();
	#if QT_VERSION >= 0x040000
	// It doesn't show until the window is resized!
	resize( width()-1, height()-1 );
	resize( width()+1, height()+1 );
	#endif
#endif
}


void DefaultGui::displayTime(double sec, int perc, QString text) {
	time_display->setText( text );
	timeslider->setPos(perc);
	timeslider_mini->setPos(perc);

	//if (floating_control->isVisible()) {
		floating_control->time->setPos(perc);
#if NEW_CONTROLWIDGET
		floating_control->time_label->setText( Helper::formatTime(sec) );
#else
		floating_control->lcd->display( Helper::formatTime(sec) );
#endif
	//}
}

void DefaultGui::displayFrame(int frame) {
	if (frame_display->isVisible()) {
		frame_display->setNum( frame );
	}
}

void DefaultGui::updateWidgets() {
	qDebug("DefaultGui::updateWidgets");

	BaseGuiPlus::updateWidgets();

	// Frame counter
	frame_display->setShown( pref->show_frame_counter );

	floating_control->fullscreen->setOn(pref->fullscreen);
	floating_control->mute->setOn(core->mset.mute);

	/*
	showMainToolbarAct->setOn( show_main_toolbar );
	showLanguageToolbarAct->setOn( show_language_toolbar );
	*/

	panel->setFocus();
}

void DefaultGui::aboutToEnterFullscreen() {
	qDebug("DefaultGui::aboutToEnterFullscreen");

	if (!pref->compact_mode) {
		menuBar()->hide();
		statusBar()->hide();
		controlwidget->hide();
		controlwidget_mini->hide();
		toolbar1->hide();
		toolbar2->hide();
	}

	#if DOCK_PLAYLIST
	playlist_was_visible = playlistdock->isVisible();
	playlistdock->hide();
	#endif
}

void DefaultGui::aboutToExitFullscreen() {
	qDebug("DefaultGui::aboutToExitFullscreen");

	floating_control->hide();

	if (!pref->compact_mode) {
		menuBar()->show();
		statusBar()->show();
		controlwidget->show();

		showMainToolbar( show_main_toolbar );
		showLanguageToolbar( show_language_toolbar );
	}

	#if DOCK_PLAYLIST
	if (playlist_was_visible)
		playlistdock->show();
	#endif
}

void DefaultGui::aboutToEnterCompactMode() {
	menuBar()->hide();
	statusBar()->hide();
	controlwidget->hide();
	controlwidget_mini->hide();
	toolbar1->hide();
	toolbar2->hide();
}

void DefaultGui::aboutToExitCompactMode() {
	menuBar()->show();
	statusBar()->show();
	controlwidget->show();

	showMainToolbar( show_main_toolbar );
	showLanguageToolbar( show_language_toolbar );

	// Recheck size of controlwidget
	resizeEvent( new QResizeEvent( size(), size() ) );
}

void DefaultGui::showFloatingControl(QPoint p) {
	qDebug("DefaultGui::showFloatingControl");

#if CONTROLWIDGET_OVER_VIDEO
	//int w = mplayerwindow->width() / 2;
	int w = mplayerwindow->width() * floating_control_width / 100;
	int h = floating_control->height();
	floating_control->resize( w, h );

	//int x = ( mplayerwindow->width() - floating_control->width() ) / 2;
	//int y = mplayerwindow->height() - floating_control->height();

	int x = ( panel->x() + panel->width() - floating_control->width() ) / 2;
	int y = panel->y() + panel->height() - floating_control->height();
	floating_control->move( mapToGlobal(QPoint(x, y)) );

	floating_control->show();
#else
	if (!controlwidget->isVisible()) {
		controlwidget->show();
	}
#endif
}

void DefaultGui::showFloatingMenu(QPoint p) {
#if !CONTROLWIDGET_OVER_VIDEO
	qDebug("DefaultGui::showFloatingMenu");

	if (!menuBar()->isVisible())
		menuBar()->show();
#endif
}

void DefaultGui::hideFloatingControls() {
	qDebug("DefaultGui::hideFloatingControls");

#if CONTROLWIDGET_OVER_VIDEO
	floating_control->hide();
#else
	if (controlwidget->isVisible())	
		controlwidget->hide();

	if (menuBar()->isVisible())	
		menuBar()->hide();
#endif
}

void DefaultGui::resizeEvent( QResizeEvent * ) {
	/*
	qDebug("defaultGui::resizeEvent");
	qDebug(" controlwidget width: %d", controlwidget->width() );
	qDebug(" controlwidget_mini width: %d", controlwidget_mini->width() );
	*/

#if QT_VERSION < 0x040000
#define LIMIT 470
#else
#define LIMIT 570
#endif

	if ( (controlwidget->isVisible()) && (width() < LIMIT) ) {
		controlwidget->hide();
		controlwidget_mini->show();
	}
	else
	if ( (controlwidget_mini->isVisible()) && (width() > LIMIT) ) {
		controlwidget_mini->hide();
		controlwidget->show();
	}
}

void DefaultGui::showMainToolbar(bool b) {
	qDebug("DefaultGui::showMainToolBar: %d", b);

	show_main_toolbar = b;
	if (b) {
		toolbar1->show();
	}
	else {
		toolbar1->hide();
	}
}

void DefaultGui::showLanguageToolbar(bool b) {
	qDebug("DefaultGui::showLanguageToolBar: %d", b);

	show_language_toolbar = b;
	if (b) {
		toolbar2->show();
	}
	else {
		toolbar2->hide();
	}
}

void DefaultGui::saveConfig() {
	qDebug("DefaultGui::saveConfig");

	QSettings * set = settings;

	set->beginGroup( "default_gui");

	/*
	QString str;
	QTextOStream out(&str);
	out << *this;
	set->writeEntry( "data", str);
	*/

	set->writeEntry( "show_main_toolbar", show_main_toolbar );
	set->writeEntry( "show_language_toolbar", show_language_toolbar );
	set->writeEntry( "floating_control_width", floating_control_width );

	set->writeEntry( "x", x() );
	set->writeEntry( "y", y() );
	set->writeEntry( "width", width() );
	set->writeEntry( "height", height() );

	set->endGroup();
}

void DefaultGui::loadConfig() {
	qDebug("DefaultGui::loadConfig");

	QSettings * set = settings;

	set->beginGroup( "default_gui");

	/*
	QString str = set->readEntry("data");
	QTextIStream in(&str);
	in >> *this;
	*/

	show_main_toolbar = set->readBoolEntry( "show_main_toolbar", true );
	show_language_toolbar = set->readBoolEntry( "show_language_toolbar", true );
	floating_control_width = set->readNumEntry( "floating_control_width", floating_control_width );

	int x = set->readNumEntry( "x", this->x() );
	int y = set->readNumEntry( "y", this->y() );
	int width = set->readNumEntry( "width", this->width() );
	int height = set->readNumEntry( "height", this->height() );

	if (height < 200) {
		width = 580;
		height = 440;
	}

	move(x,y);
	resize(width,height);

	set->endGroup();

	showMainToolbarAct->setOn( show_main_toolbar );
	showLanguageToolbarAct->setOn( show_language_toolbar );

	showMainToolbar( show_main_toolbar );
	showLanguageToolbar( show_language_toolbar );

	updateWidgets();
}

