/* Stars -- Displays a Map of the Night Sky
    Copyright (C) September 22, 2002  Walter Brisken

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

#include <math.h>
#include "configurelocation.h"

/* Protos */

void siteentrychanged(GtkWidget *widget, gpointer *d);

struct locationconfig *newlocationconfig(GladeXML *configxml)
{
	struct locationconfig *lc;

	lc = g_new(struct locationconfig, 1);

	lc->citybutton = glade_xml_get_widget(configxml, "citybutton");
	lc->citycombo = glade_xml_get_widget(configxml, "citycombo");
	lc->cityentry = glade_xml_get_widget(configxml, "cityentry");
	lc->citylabel = glade_xml_get_widget(configxml, "citylabel");
	lc->obsbutton = glade_xml_get_widget(configxml, "obsbutton");
	lc->obscombo = glade_xml_get_widget(configxml, "obscombo");
	lc->obsentry = glade_xml_get_widget(configxml, "obsentry");
	lc->obslabel = glade_xml_get_widget(configxml, "obslabel");
	lc->otherbutton = glade_xml_get_widget(configxml, "otherbutton");
	lc->latentry = glade_xml_get_widget(configxml, "latentry");
	lc->NScombo = glade_xml_get_widget(configxml, "NScombo");
	lc->NSentry = glade_xml_get_widget(configxml, "NSentry");
	lc->longentry = glade_xml_get_widget(configxml, "longentry");
	lc->WEcombo = glade_xml_get_widget(configxml, "WEcombo");
	lc->WEentry = glade_xml_get_widget(configxml, "WEentry");
	lc->eleventry = glade_xml_get_widget(configxml, "eleventry");
	lc->elevunitscombo = glade_xml_get_widget(configxml, "elevunitscombo");
	lc->elevunitsentry = glade_xml_get_widget(configxml, "elevunitsentry");
	lc->addcitybutton = glade_xml_get_widget(configxml, "addcitybutton");
	lc->addobsbutton = glade_xml_get_widget(configxml, "addobsbutton");

	lc->oldloc = 0;
	lc->curloc = 0;

	lc->citylist = GTK_COMBO(lc->citycombo)->list;
	lc->obslist = GTK_COMBO(lc->obscombo)->list;
	lc->NSlist = GTK_COMBO(lc->NScombo)->list;
	lc->WElist = GTK_COMBO(lc->WEcombo)->list;
	lc->elevunitslist = GTK_COMBO(lc->elevunitscombo)->list;
	
	sitecombofill(lc->citycombo, 0);
	sitecombofill(lc->obscombo, 1);
	gtk_signal_connect(GTK_OBJECT(lc->cityentry), "changed",
		GTK_SIGNAL_FUNC(siteentrychanged), lc);
	gtk_signal_connect(GTK_OBJECT(lc->obsentry), "changed",
		GTK_SIGNAL_FUNC(siteentrychanged), lc);

	return lc;
}

void deletelocationconfig(struct locationconfig *lc)
{
	g_free(lc);
}

/* distribute config from curloc to widgets */
void locationconfigupdate(struct locationconfig *lc)
{
	gchar *tmp;
	
	switch(lc->curloc->type)
	{
		case 0:	
			gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(lc->citybutton), TRUE);
			break;
		case 1:	
			gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(lc->obsbutton), TRUE);
			break;
		case 2:	
			gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(lc->otherbutton), TRUE);
			break;
		default:
			fprintf(stderr, "location type not in [012]\n");
			break;
	}
	gtk_list_select_item(GTK_LIST(lc->citylist), lc->curloc->cityid);
	gtk_label_set_text(GTK_LABEL(lc->citylabel), 
		site2text(getsite(CITY_SITES, lc->curloc->cityid)));
	gtk_list_select_item(GTK_LIST(lc->obslist), lc->curloc->obsid);
	gtk_label_set_text(GTK_LABEL(lc->obslabel), 
		site2text(getsite(OBS_SITES, lc->curloc->obsid)));
	
	tmp = g_double2text(fabs(lc->curloc->otherpos->latitude), ' ');
	gtk_entry_set_text(GTK_ENTRY(lc->latentry), tmp);
	g_free(tmp);
	gtk_list_select_item(GTK_LIST(lc->NSlist), 
		lc->curloc->otherpos->latitude >= 0.0 ? 0 : 1);
		
	tmp = g_double2text(fabs(lc->curloc->otherpos->longitude), ' ');
	gtk_entry_set_text(GTK_ENTRY(lc->longentry), tmp);
	g_free(tmp);
	gtk_list_select_item(GTK_LIST(lc->WElist), 
		lc->curloc->otherpos->longitude >= 0.0 ? 0 : 1);
	
	tmp = g_strdup_printf("%1.5f", lc->curloc->otherpos->elevation);
	gtk_entry_set_text(GTK_ENTRY(lc->eleventry), tmp);
	g_free(tmp);
	gtk_list_select_item(GTK_LIST(lc->elevunitslist), 0);
}

/* collect info from widgets to curloc */
void locationconfigcollect(struct locationconfig *lc)
{
	gchar *cityname, *obsname, *tmp;
	gint cityid, obsid;

	cityname = gtk_editable_get_chars(GTK_EDITABLE(lc->cityentry), 0, -1);
	obsname  = gtk_editable_get_chars(GTK_EDITABLE(lc->obsentry), 0, -1);

	cityid = getsitenum(CITY_SITES, cityname);
	obsid  = getsitenum(OBS_SITES,  obsname);

	if(cityid >= 0) lc->curloc->cityid = cityid;
	if(obsid >= 0) lc->curloc->obsid = obsid;

	g_free(cityname);
	g_free(obsname);

	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lc->citybutton)))
		lc->curloc->type = 0;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lc->obsbutton)))
		lc->curloc->type = 1;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lc->otherbutton)))
		lc->curloc->type = 2;
	
	tmp = gtk_editable_get_chars(GTK_EDITABLE(lc->latentry), 0, -1);
	lc->curloc->otherpos->latitude = text2latitude(tmp);
	g_free(tmp);
	tmp = gtk_editable_get_chars(GTK_EDITABLE(lc->NSentry), 0, -1);
	if(strcmp(tmp, "South") == 0) 
		lc->curloc->otherpos->latitude = 
			-lc->curloc->otherpos->latitude;
	g_free(tmp);
	
	tmp = gtk_editable_get_chars(GTK_EDITABLE(lc->longentry), 0, -1);
	lc->curloc->otherpos->longitude = text2longitude(tmp);
	g_free(tmp);
	tmp = gtk_editable_get_chars(GTK_EDITABLE(lc->WEentry), 0, -1);
	if(strcmp(tmp, "East") == 0) 
		lc->curloc->otherpos->longitude = 
			-lc->curloc->otherpos->longitude;
	g_free(tmp);
}

void locationconfigtry(struct configurator *cfg)
{
	locationconfigcollect(cfg->locinfo);
	locationconfigupdate(cfg->locinfo);
	viewersetlocation(cfg->v, cfg->locinfo->curloc);
}

void locationconfigrevert(struct configurator *cfg)
{
	if(cfg->locinfo->curloc) deletelocation(cfg->locinfo->curloc);
	cfg->locinfo->curloc = copylocation(cfg->locinfo->oldloc);
	locationconfigupdate(cfg->locinfo);
	viewersetlocation(cfg->v, cfg->locinfo->curloc);
}

void locationconfigsetup(struct configurator *cfg)
{
	static struct viewer *v = 0;

	v = cfg->v;
	if(cfg->locinfo->oldloc) deletelocation(cfg->locinfo->oldloc);
	cfg->locinfo->oldloc = copylocation(v->loc);

	if(cfg->locinfo->curloc) deletelocation(cfg->locinfo->curloc);
	cfg->locinfo->curloc = copylocation(v->loc);
	
	locationconfigupdate(cfg->locinfo);
}

/* ------------- Callbacks ----------- */

void siteentrychanged(GtkWidget *widget, gpointer *d)
{
	gchar *text;
	gint index;
	gint sitelistnum;
	struct locationconfig *lc;
	
	lc = (struct locationconfig *)d;

	if(widget == lc->cityentry) sitelistnum = CITY_SITES;
	else if(widget == lc->obsentry) sitelistnum = OBS_SITES;
	else
	{
		fprintf(stderr, "siteentrychanged called with bad widget\n");
		return;
	}
	
	text = gtk_editable_get_chars(GTK_EDITABLE(widget), 0, -1);
	index = getsitenum(sitelistnum, text);
	g_free(text);

	if(index >= 0) switch(sitelistnum)
	{
		case 0:
			if(index < 0) 
				gtk_label_set_text(GTK_LABEL(lc->citylabel),"");
			else gtk_label_set_text(GTK_LABEL(lc->citylabel),
				site2text(getsite(CITY_SITES, index)));
			break;
		case 1:
			if(index < 0) 
				gtk_label_set_text(GTK_LABEL(lc->citylabel),"");
			else gtk_label_set_text(GTK_LABEL(lc->obslabel),
				site2text(getsite(OBS_SITES, index)));
			break;
		default:
			fprintf(stderr, "Weird -- siteentrychanged: sitelistnum !in [01]\n");
			break;
	}
}
