//============================================================================
//
//   SSSS    tt          lll  lll       
//  SS  SS   tt           ll   ll        
//  SS     tttttt  eeee   ll   ll   aaaa 
//   SSSS    tt   ee  ee  ll   ll      aa
//      SS   tt   eeeeee  ll   ll   aaaaa  --  "An Atari 2600 VCS Emulator"
//  SS  SS   tt   ee      ll   ll  aa  aa
//   SSSS     ttt  eeeee llll llll  aaaaa
//
// Copyright (c) 1995-2005 by Bradford W. Mott and the Stella team
//
// See the file "license" for information on usage and redistribution of
// this file, and for a DISCLAIMER OF ALL WARRANTIES.
//
// $Id: OSystemGP2X.cxx,v 1.9 2006/03/17 19:44:18 stephena Exp $
// Modified on 2006/01/06 by Alex Zaballa for use on GP2X
//============================================================================

#include <cstdlib>
#include <sstream>
#include <fstream>

#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "bspf.hxx"
#include "OSystem.hxx"
#include "OSystemGP2X.hxx"

#ifdef HAVE_GETTIMEOFDAY
  #include <time.h>
  #include <sys/time.h>
#endif

/**
  Each derived class is responsible for calling the following methods
  in its constructor:

  setBaseDir()
  setStateDir()
  setPropertiesDir()
  setConfigFile()
  setCacheFile()

  See OSystem.hxx for a further explanation
*/

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
OSystemGP2X::OSystemGP2X()
{
  // GP2X needs all config files in exec directory
  
  char *currdir = getcwd(NULL, 0);
  string basedir = currdir;
  free(currdir);
  setBaseDir(basedir);
  
  setStateDir(basedir + "/state");
  
  setPropertiesDir(basedir);
  setConfigFile(basedir + "/stellarc");

  setCacheFile(basedir + "/stella.cache");

  // Set event arrays to a known state
  myPreviousEvents = new uInt8[8];  memset(myPreviousEvents, 0, 8);
  myCurrentEvents  = new uInt8[8];  memset(myCurrentEvents, 0, 8);
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
OSystemGP2X::~OSystemGP2X()
{
  delete[] myPreviousEvents;
  delete[] myCurrentEvents;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void OSystemGP2X::mainLoop()
{
  // These variables are common to both timing options
  // and are needed to calculate the overall frames per second.
  uInt32 frameTime = 0, numberOfFrames = 0;

  // Set up less accurate timing stuff
  uInt32 startTime, virtualTime, currentTime;

  // Set the base for the timers
  virtualTime = getTicks();
  frameTime = 0;

  // Main game loop
  for(;;)
  {
    // Exit if the user wants to quit
    if(myEventHandler->doQuit())
      break;

    startTime = getTicks();
    myEventHandler->poll(startTime);
    myFrameBuffer->update();

    currentTime = getTicks();
    virtualTime += myTimePerFrame;
    if(currentTime < virtualTime)
      SDL_Delay((virtualTime - currentTime)/1000);

    currentTime = getTicks() - startTime;
    frameTime += currentTime;
    ++numberOfFrames;
  }
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
uInt32 OSystemGP2X::getTicks()
{
#ifdef HAVE_GETTIMEOFDAY
  timeval now;
  gettimeofday(&now, 0);

  return (uInt32) (now.tv_sec * 1000000 + now.tv_usec);
#else
  return (uInt32) SDL_GetTicks() * 1000;
#endif
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void OSystemGP2X::getScreenDimensions(int& width, int& height)
{
  width  = 320;
  height = 240;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void OSystemGP2X::setDefaultJoymap()
{
  myEventHandler->setDefaultJoyMapping(Event::LauncherMode, 0, 8);      // Start
  myEventHandler->setDefaultJoyMapping(Event::CmdMenuMode, 0, 9);       // Select
  myEventHandler->setDefaultJoyMapping(Event::ConsoleReset, 0, 10);     // L
  myEventHandler->setDefaultJoyMapping(Event::ConsoleSelect, 0, 11);    // R
  myEventHandler->setDefaultJoyMapping(Event::TakeSnapshot, 0, 12);	    // A
  myEventHandler->setDefaultJoyMapping(Event::JoystickZeroFire, 0, 13); // B
  myEventHandler->setDefaultJoyMapping(Event::Pause, 0, 14);            // X
  myEventHandler->setDefaultJoyMapping(Event::MenuMode, 0, 15);         // Y
  myEventHandler->setDefaultJoyMapping(Event::VolumeIncrease, 0, 16);   // Vol+
  myEventHandler->setDefaultJoyMapping(Event::VolumeDecrease, 0, 17);   // Vol-
  myEventHandler->setDefaultJoyMapping(Event::NoType, 0, 18);           // Click
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void OSystemGP2X::pollEvent()
{
  // Translate joystick button events that act as directions into proper
  // SDL axis events.  This method will use 'case 2', as discussed on the
  // GP2X forums.  Technically, this code should be integrated directly
  // into the GP2X SDL port.

  // Swap event buffers
  uInt8* tmp = myCurrentEvents;
  myCurrentEvents = myPreviousEvents;
  myPreviousEvents = tmp;

  // Scan all the buttons, and detect if an event has occurred
  bool changeDetected = false;
  SDL_Joystick* stick = myEventHandler->getJoystick(0);
  for(int i = 0; i < 8; ++i)
  {
    myCurrentEvents[i] = SDL_JoystickGetButton(stick, i);
    myActiveEvents[i] = myCurrentEvents[i] != myPreviousEvents[i];
    changeDetected = changeDetected || myActiveEvents[i];
  }

  if(changeDetected)
  {
    SDL_JoyAxisEvent eventA0, eventA1;
    eventA0.type  = eventA1.type  = SDL_JOYAXISMOTION;
    eventA0.which = eventA1.which = 0;
    eventA0.value = 0;eventA1.value = 0;
    eventA0.axis  = 0;
    eventA1.axis  = 1;

    bool axisZeroChanged = false, axisOneChanged = false;

    axisOneChanged = axisOneChanged || myActiveEvents[kJDirUp];
    if(myCurrentEvents[kJDirUp])         // up
    {
      eventA1.value = -32768;
    }
    axisOneChanged = axisOneChanged || myActiveEvents[kJDirDown];
    if(myCurrentEvents[kJDirDown])       // down
    {
      eventA1.value =  32767;
    }
    axisZeroChanged = axisZeroChanged || myActiveEvents[kJDirLeft];
    if(myCurrentEvents[kJDirLeft])       // left
    {
      eventA0.value = -32768;
    }
    axisZeroChanged = axisZeroChanged || myActiveEvents[kJDirRight];
    if(myCurrentEvents[kJDirRight])      // right
    {
      eventA0.value =  32767;
    }

    axisOneChanged  = axisOneChanged || myActiveEvents[kJDirUpLeft];
    axisZeroChanged = axisZeroChanged || myActiveEvents[kJDirUpLeft];
    if(myCurrentEvents[kJDirUpLeft])     // up-left
    {
      eventA1.value = -16834;
      eventA0.value = -16834;
    }
    axisOneChanged  = axisOneChanged || myActiveEvents[kJDirUpRight];
    axisZeroChanged = axisZeroChanged || myActiveEvents[kJDirUpRight];
    if(myCurrentEvents[kJDirUpRight])    // up-right
    {
      eventA1.value = -16834;
      eventA0.value =  16834;
    }
    axisOneChanged  = axisOneChanged || myActiveEvents[kJDirDownLeft];
    axisZeroChanged = axisZeroChanged || myActiveEvents[kJDirDownLeft];
    if(myCurrentEvents[kJDirDownLeft])   // down-left
    {
      eventA1.value =  16834;
      eventA0.value = -16834;
    }
    axisOneChanged  = axisOneChanged || myActiveEvents[kJDirDownRight];
    axisZeroChanged = axisZeroChanged || myActiveEvents[kJDirDownRight];
    if(myCurrentEvents[kJDirDownRight])  // down-right
    {
      eventA1.value =  16834;
      eventA0.value =  16834;
    }

    if(axisZeroChanged) SDL_PushEvent((SDL_Event*)&eventA0);
    if(axisOneChanged)  SDL_PushEvent((SDL_Event*)&eventA1);
  }
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
bool OSystemGP2X::joyButtonHandled(int button)
{
  return (button < 8);
}
