//============================================================================
//
//   SSSS    tt          lll  lll       
//  SS  SS   tt           ll   ll        
//  SS     tttttt  eeee   ll   ll   aaaa 
//   SSSS    tt   ee  ee  ll   ll      aa
//      SS   tt   eeeeee  ll   ll   aaaaa  --  "An Atari 2600 VCS Emulator"
//  SS  SS   tt   ee      ll   ll  aa  aa
//   SSSS     ttt  eeeee llll llll  aaaaa
//
// Copyright (c) 1995-2008 by Bradford W. Mott and the Stella team
//
// See the file "license" for information on usage and redistribution of
// this file, and for a DISCLAIMER OF ALL WARRANTIES.
//
// $Id: EquateList.hxx,v 1.22 2008/05/18 20:04:30 stephena Exp $
//============================================================================

#ifndef EQUATELIST_HXX
#define EQUATELIST_HXX

#include <map>

#include "bspf.hxx"

class EquateList
{
  public:
    EquateList();
    ~EquateList();

    /**
      Add an equate consisting of the given label and address
    */
    void addEquate(const string& label, uInt16 address);

    /**
      Remove the equate with the given label
    */
    bool removeEquate(const string& label);

    /**
      Accessor methods for labels and addresses

      The mapping from address to label can be one-to-many (ie, an
      address can have different labels depending on its context, and
      whether its being read or written; if isRead is true, the context
      is a read, else it's a write
      If places is not -1 and a label hasn't been defined, return a
      formatted hexidecimal address
    */
    const string& getLabel(uInt16 addr, bool isRead, int places = -1);
    int getAddress(const string& label) const;

    /**
      Load user equates from the given symbol file (generated by DASM)
    */
    string loadFile(const string& file);

    /**
      Save user equates into a symbol file similar to that generated by DASM
    */
    bool saveFile(const string& file);

    /**
      Methods used by the command parser for tab-completion
    */
    int countCompletions(const char *in);
    const string& getCompletions() const      { return myCompletions; }
    const string& getCompletionPrefix() const { return myCompPrefix;  }

  private:
    enum equate_t {
      EQF_READ  = 1 << 0,               // address can be read from
      EQF_WRITE = 1 << 1,               // address can be written to
      EQF_RW    = EQF_READ | EQF_WRITE  // address can be both read and written
    };
    enum address_t {
      ADDR_TIA  = 1 << 0,
      ADDR_RAM  = 1 << 1,
      ADDR_RIOT = 1 << 2,
      ADDR_ROM  = 1 << 3
    };
    struct Equate {
      string label;
      uInt16 address;
      equate_t flags;
    };

    typedef map<uInt16, Equate> AddrToLabel;
    typedef map<string, Equate> LabelToAddr;

  private:
    // Extract labels and values from the given character stream
    string extractLabel(char *c);
    int extractValue(char *c);

    // Count completions for the given mapping
    int countCompletions(const char *in, LabelToAddr& addresses);

    // Determine what type address we're dealing with
    inline void addressType(uInt16 addr);

  private:
    enum { kSystemEquateSize = 158 };
    static const Equate ourSystemEquates[kSystemEquateSize];

    string myCompletions;
    string myCompPrefix;

    LabelToAddr mySystemAddresses;
    AddrToLabel mySystemReadLabels;   // labels used in a read context
    AddrToLabel mySystemWriteLabels;  // labels used in a write context

    LabelToAddr myUserAddresses;
    AddrToLabel myUserLabels;

    address_t myAddressType;
    string myCurrentLabel;
};

#endif
