/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package docrobot;

import java.awt.Graphics;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

import javax.imageio.ImageIO;
import javax.swing.*;

import org.jvnet.substance.*;
import org.jvnet.substance.skin.SubstanceSkin;

import test.check.SampleFrame;

/**
 * The base class for taking screenshots of skins for Substance documentation.
 * 
 * @author Kirill Grouchnikov
 */
public abstract class SkinRobot {
	/**
	 * The associated Substance skin.
	 */
	protected SubstanceSkin skin;

	/**
	 * The screenshot filename.
	 */
	protected String screenshotFilename;

	/**
	 * The frame instance.
	 */
	protected SampleFrame sf;

	/**
	 * Indicates whether the screenshot process is complete.
	 */
	protected boolean done = false;

	/**
	 * Creates the new screenshot robot.
	 * 
	 * @param skin
	 *            Substance skin.
	 * @param screenshotFilename
	 *            The screenshot filename.
	 */
	public SkinRobot(SubstanceSkin skin, String screenshotFilename) {
		this.skin = skin;
		this.screenshotFilename = screenshotFilename;
	}

	/**
	 * Runs the screenshot process.
	 */
	public void run() {
		long start = System.currentTimeMillis();
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				try {
					UIManager.setLookAndFeel(new SubstanceDefaultLookAndFeel());
					SubstanceLookAndFeel.setSkin(skin);
				} catch (Exception exc) {
					exc.printStackTrace();
				}
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						JFrame.setDefaultLookAndFeelDecorated(true);
						sf = new SampleFrame(false);
						sf
								.setIconImage(SubstanceImageCreator
										.getThemeImage(
												null,
												new ImageIcon(
														SkinRobot.class
																.getClassLoader()
																.getResource(
																		"test/resource/image-x-generic.png")),
												SubstanceLookAndFeel.getTheme(),
												false));
						sf.addComponentListener(new ComponentAdapter() {
							@Override
							public void componentShown(ComponentEvent e) {
								SwingUtilities.invokeLater(new Runnable() {
									public void run() {
										makeScreenshot(1);
										sf.switchToLastTab();
										try {
											Thread.sleep(2000);
										} catch (Exception exc) {
										}
										makeScreenshot(2);
										SwingUtilities
												.invokeLater(new Runnable() {
													public void run() {
														sf.dispose();
														done = true;
													}
												});
									}
								});
							}
						});
						sf.setSize(300, 225);
						sf.setLocationRelativeTo(null);
						sf.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
						sf.setVisible(true);
					}
				});
			}
		});
		while (!done) {
			try {
				Thread.sleep(100);
			} catch (InterruptedException ie) {
			}
		}
		long end = System.currentTimeMillis();
		System.out.println(this.getClass().getSimpleName() + " : "
				+ (end - start) + "ms");
	}

	/**
	 * Creates the screenshot and saves it on the disk.
	 * 
	 * @param count
	 *            Sequence number for the screenshot.
	 */
	public void makeScreenshot(int count) {
		BufferedImage bi = new BufferedImage(sf.getWidth(), sf.getHeight(),
				BufferedImage.TYPE_INT_ARGB);
		Graphics g = bi.getGraphics();
		sf.paint(g);
		try {
			ImageIO.write(bi, "png", new File(this.screenshotFilename + count
					+ ".png"));
		} catch (IOException ioe) {
			ioe.printStackTrace();
		}
	}
}
