/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance;

import java.awt.*;

import javax.swing.JComponent;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.plaf.UIResource;

import org.jvnet.lafwidget.animation.FadeKind;
import org.jvnet.lafwidget.animation.FadeTracker;
import org.jvnet.lafwidget.layout.TransitionLayout;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.SubstanceCoreUtilities;
import org.jvnet.substance.utils.SubstanceSizeUtils;

/**
 * Gradient border for the <b>Substance</b> look and feel. This class is <b>for
 * internal use only</b>.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceBorder implements Border, UIResource {
	/**
	 * Insets of <code>this</code> border.
	 */
	protected Insets myInsets;

	/**
	 * Background delegate.
	 */
	private static SubstanceFillBackgroundDelegate bgDelegate = new SubstanceFillBackgroundDelegate(
			0.6f);

	/**
	 * Border opacity.
	 */
	protected float alpha;

	/**
	 * When the border is painted, the default radius is multiplied by this
	 * factor.
	 */
	protected float radiusFactor;

	/**
	 * Creates a new border with dynamic insets (computed at the invocation time
	 * of {@link #getBorderInsets(Component)} call).
	 */
	public SubstanceBorder() {
		super();
		this.alpha = 1.0f;
		this.radiusFactor = 0.5f;
	}

	/**
	 * Creates a new border with dynamic insets (computed at the invocation time
	 * of {@link #getBorderInsets(Component)} call).
	 * 
	 * @param radiusFactor
	 *            Radius factor.
	 */
	public SubstanceBorder(float radiusFactor) {
		this();
		this.radiusFactor = radiusFactor;
	}

	/**
	 * Creates a new border with the specified insets.
	 * 
	 * @param insets
	 *            Insets.
	 */
	public SubstanceBorder(Insets insets) {
		this();
		this.myInsets = new Insets(insets.top, insets.left, insets.bottom,
				insets.right);
	}

	/**
	 * Sets the alpha for this border.
	 * 
	 * @param alpha
	 *            Alpha factor.
	 */
	public void setAlpha(float alpha) {
		this.alpha = alpha;
	}

	/**
	 * Paints border instance for the specified component.
	 * 
	 * @param c
	 *            The component.
	 * @param g
	 *            Graphics context.
	 * @param x
	 *            Component left X (in graphics context).
	 * @param y
	 *            Component top Y (in graphics context).
	 * @param width
	 *            Component width.
	 * @param height
	 *            Component height.
	 * @param isEnabled
	 *            Component enabled status.
	 * @param hasFocus
	 *            Component focus ownership status.
	 * @param alpha
	 *            Alpha value.
	 */
	private void paintBorder(Component c, Graphics g, int x, int y, int width,
			int height, boolean isEnabled, boolean hasFocus, float alpha) {
		// failsafe for LAF change
		if (!(UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel)) {
			return;
		}

		Graphics2D graphics = (Graphics2D) g.create();
		graphics.setComposite(TransitionLayout.getAlphaComposite(c, alpha, g));

		if (SubstanceCoreUtilities.toBleedWatermark(c)) {
			if (c instanceof JComponent) {
				Border b = ((JComponent) c).getBorder();
				if (b != null) {
					Graphics2D graphics2 = (Graphics2D) graphics.create();
					Shape clip = graphics2.getClip();
					Insets ins = b.getBorderInsets(c);
					graphics2.clipRect(x, y, ins.left, height);
					bgDelegate.updateIfOpaque(graphics2, c);
					graphics2.setClip(clip);
					graphics2.clipRect(x, y, width, ins.top);
					bgDelegate.updateIfOpaque(graphics2, c);
					graphics2.setClip(clip);
					graphics2.clipRect(x + width - ins.right, y, ins.right,
							height);
					bgDelegate.updateIfOpaque(graphics2, c);
					graphics2.setClip(clip);
					graphics2.clipRect(x, y + height - ins.bottom, width,
							ins.bottom);
					bgDelegate.updateIfOpaque(graphics2, c);

					graphics2.dispose();
				}
			}
		}

		// boolean isFocused = FocusBorderListener.isFocused(c);
		//
		SubstanceTheme theme1 = null;
		SubstanceTheme theme2 = null;
		float cyclePos = 1.0f;

		FadeTracker fadeTracker = FadeTracker.getInstance();
		boolean isFocusAnimated = fadeTracker.isTracked(c, FadeKind.FOCUS);
		boolean isBorderAnimated = fadeTracker.isTracked(c,
				SubstanceLookAndFeel.BORDER_ANIMATION_KIND);

		float radius = this.radiusFactor
				* SubstanceSizeUtils
						.getClassicButtonCornerRadius(SubstanceSizeUtils
								.getComponentFontSize(c));
		if (isFocusAnimated || isBorderAnimated || c.hasFocus()) {
			SubstanceTheme controlActiveTheme = SubstanceCoreUtilities
					.getActiveTheme(c, true);
			SubstanceTheme controlDefaultTheme = SubstanceCoreUtilities
					.getDefaultTheme(c, true);
			theme1 = controlDefaultTheme;
			theme2 = controlActiveTheme;
			if (isFocusAnimated) {
				cyclePos = fadeTracker.getFade10(c, FadeKind.FOCUS) / 10.f;
			} else {
				if (isBorderAnimated) {
					// theme1 = controlActiveTheme;
					// theme2 = controlDefaultTheme;
					cyclePos = /* Math.abs(1.0f - 2.0f * */fadeTracker
							.getFade10(c,
									SubstanceLookAndFeel.BORDER_ANIMATION_KIND) / 10.f;
					// System.out.println(controlDefaultTheme);
					// System.out.println(cyclePos);
					// System.out.println(controlActiveTheme);
				} else {
					cyclePos = 1.0f;
				}
			}

			SubstanceImageCreator.paintBorder(c, graphics, x, y, width, height,
					radius, theme1);
			// System.out.println(graphics.getClip());
			graphics.setComposite(TransitionLayout.getAlphaComposite(c,
					cyclePos * alpha, g));
			SubstanceImageCreator.paintBorder(c, graphics, x, y, width, height,
					radius, theme2);
		} else {
			if (isEnabled) {
				// theme1 = controlDefaultTheme;
				// theme2 = controlDefaultTheme;
				// if (isFocusAnimated) {
				// theme2 = controlActiveTheme;
				// cyclePos = fadeTracker.getFade10(c, FadeKind.FOCUS) / 10.f;
				// }
				SubstanceImageCreator.paintBorder(c, graphics, x, y, width,
						height, radius, SubstanceCoreUtilities.getDefaultTheme(
								c, true));
				// graphics.setComposite(AlphaComposite.getInstance(
				// AlphaComposite.SRC_OVER, cyclePos));
				// SubstanceImageCreator.paintBorder(c, graphics, x, y, width,
				// height, theme2);
			} else {
				SubstanceTheme controlDisabledTheme = SubstanceCoreUtilities
						.getDisabledTheme(c, true);
				SubstanceImageCreator.paintBorder(c, graphics, x, y, width,
						height, radius, controlDisabledTheme);
			}
		}
		if (fadeTracker.isTracked(c, FadeKind.FOCUS_LOOP_ANIMATION)) {
			float coef = 0.4f;
			if (fadeTracker.isTracked(c, FadeKind.FOCUS))
				coef = 0.4f * fadeTracker.getFade10(c, FadeKind.FOCUS) / 10.0f;
			graphics.setComposite(TransitionLayout.getAlphaComposite(c, coef
					* alpha, g));
			int extraPadding = 2 + SubstanceSizeUtils.getExtraPadding(c
					.getFont().getSize());
			SubstanceCoreUtilities.paintFocus(graphics, c, c, null,
					new Rectangle(1, 1, c.getWidth() - 1, c.getHeight() - 1),
					coef, extraPadding);
		}
		graphics.dispose();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.border.Border#paintBorder(java.awt.Component,
	 *      java.awt.Graphics, int, int, int, int)
	 */
	public void paintBorder(Component c, Graphics g, int x, int y, int width,
			int height) {
		// if (c instanceof JComponent) {
		// ((JComponent)c).setOpaque(false);
		// }
		paintBorder(c, g, x, y, width, height, c.isEnabled(), c.hasFocus(),
				this.alpha);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.border.Border#getBorderInsets(java.awt.Component)
	 */
	public Insets getBorderInsets(Component c) {
		if (this.myInsets == null) {
			return SubstanceSizeUtils.getDefaultBorderInsets(SubstanceSizeUtils
					.getComponentFontSize(c));
		}
		return this.myInsets;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.border.Border#isBorderOpaque()
	 */
	public boolean isBorderOpaque() {
		return false;
	}
}
