/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.HashSet;
import java.util.Set;

import javax.swing.*;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.basic.BasicMenuUI;

import org.jvnet.lafwidget.animation.FadeKind;
import org.jvnet.lafwidget.animation.FadeStateListener;
import org.jvnet.substance.utils.SubstanceCoreUtilities;
import org.jvnet.substance.utils.icon.MenuArrowIcon;
import org.jvnet.substance.utils.menu.MenuUtilities;
import org.jvnet.substance.utils.menu.MenuUtilities.MenuPropertyListener;

/**
 * UI for menus in <b>Substance</b> look and feel.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceMenuUI extends BasicMenuUI implements SubstanceMenu {
	/**
	 * Delegate for painting the background.
	 */
	private static SubstanceMenuBackgroundDelegate backgroundDelegate = new SubstanceMenuBackgroundDelegate(
			1.0f);

	/**
	 * For rollover effects - enhancement 93.
	 */
	protected MouseListener substanceMouseListener;

	/**
	 * Listener for fade animations.
	 */
	protected FadeStateListener substanceFadeStateListener;

	/**
	 * Listens on all changes to the underlying menu item.
	 */
	protected MenuPropertyListener substanceMenuPropertyListener;

	/**
	 * Property change listener. Listens on changes to
	 * {@link AbstractButton#MODEL_CHANGED_PROPERTY} property.
	 */
	protected PropertyChangeListener substancePropertyListener;

	/**
	 * For rollover effects - enhancement 93.
	 */
	protected FocusListener substanceFocusListener;

	/**
	 * Graphics context for laying out the menu items.
	 */
	protected static Graphics graphics;

	static {
		BufferedImage dummy = new BufferedImage(1, 1,
				BufferedImage.TYPE_INT_ARGB);
		SubstanceMenuUI.graphics = dummy.getGraphics();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#createUI(javax.swing.JComponent)
	 */
	public static ComponentUI createUI(JComponent c) {
		return new SubstanceMenuUI();
	}

	@Override
	protected void installDefaults() {
		super.installDefaults();
		this.menuItem.setRolloverEnabled(true);

		// SubstanceTheme theme = SubstanceLookAndFeel.getTheme();
		// SubstanceTheme titleTheme = SubstanceLookAndFeel.getTheme()
		// .getActiveTitlePaneTheme();
		// Color foregroundColor = new ColorUIResource(SubstanceColorUtilities
		// .getForegroundColor(theme));
		// Color headerForegroundColor = new ColorUIResource(
		// SubstanceColorUtilities.getForegroundColor(titleTheme));
		// Color backgroundColor = new ColorUIResource(SubstanceColorUtilities
		// .getBackgroundColor(theme));
		// Color headerBackgroundColor = new ColorUIResource(
		// SubstanceColorUtilities.getBackgroundColor(titleTheme));

		// System.out.println("fore : " + foregroundColor);
		// System.out.println("hfore : " + headerForegroundColor);
		// System.out.println("back : " + backgroundColor);
		// System.out.println("hback : " + headerBackgroundColor);

		// Color disabledForegroundColor = new ColorUIResource(
		// SubstanceColorUtilities.getDisabledForegroundColor(theme));
		// Color menuDisabledForegroundColor = new ColorUIResource(
		// SubstanceColorUtilities.getDisabledForegroundColor(titleTheme));

		// Color bg = this.menuItem.getBackground();
		// if (bg instanceof UIResource) {
		// this.menuItem.setBackground(new MenuColorDelegate(this.menuItem,
		// headerBackgroundColor, backgroundColor));
		// }
		// Color fg = this.menuItem.getForeground();
		// if (fg instanceof UIResource) {
		// this.menuItem.setForeground(new MenuColorDelegate(this.menuItem,
		// headerForegroundColor, foregroundColor));
		// }
		//
		// // MenuItem specific defaults
		// if (selectionBackground == null
		// || selectionBackground instanceof UIResource) {
		// selectionBackground = new MenuColorDelegate(this.menuItem,
		// headerBackgroundColor, backgroundColor);
		// }
		// if (selectionForeground == null
		// || selectionForeground instanceof UIResource) {
		// selectionForeground = new MenuColorDelegate(this.menuItem,
		// headerForegroundColor, foregroundColor);
		// }
		// if (disabledForeground == null
		// || disabledForeground instanceof UIResource) {
		// disabledForeground = new MenuColorDelegate(this.menuItem,
		// headerForegroundColor, foregroundColor);
		// }
		// if (acceleratorForeground == null
		// || acceleratorForeground instanceof UIResource) {
		// acceleratorForeground = new MenuColorDelegate(this.menuItem,
		// headerForegroundColor, foregroundColor);
		// }
		// if (acceleratorSelectionForeground == null
		// || acceleratorSelectionForeground instanceof UIResource) {
		// acceleratorSelectionForeground = new MenuColorDelegate(
		// this.menuItem, headerForegroundColor, foregroundColor);
		// }

		this.arrowIcon = new MenuArrowIcon(this.menuItem);
	}

	@Override
	protected void installListeners() {
		super.installListeners();

		// Improving performance on big menus.
		this.substanceMenuPropertyListener = new MenuPropertyListener(
				this.menuItem);
		this.substanceMenuPropertyListener.install();

		// fix for enhancement 93 - rollover effects on menu items
		this.substanceMouseListener = new MouseAdapter() {
			// fix for defect 93 - no rollover effects on menu
			// items that are not in the selected path
			private boolean toRepaint() {
				MenuElement[] selectedMenuPath = MenuSelectionManager
						.defaultManager().getSelectedPath();
				for (MenuElement elem : selectedMenuPath) {
					if (elem == SubstanceMenuUI.this.menuItem) {
						return true;
					}
				}
				return (selectedMenuPath.length == 0);
			}

			@Override
			public void mouseEntered(MouseEvent e) {
				if (this.toRepaint()) {
					SubstanceMenuUI.this.menuItem.getModel().setRollover(true);
					SubstanceMenuUI.this.menuItem.repaint();
				}
			}

			@Override
			public void mouseExited(MouseEvent e) {
				if (this.toRepaint()) {
					SubstanceMenuUI.this.menuItem.getModel().setRollover(false);
					SubstanceMenuUI.this.menuItem.repaint();
				}
			}
		};
		this.menuItem.addMouseListener(this.substanceMouseListener);
		this.substanceFocusListener = new FocusAdapter() {
			// fix for defect 93 - no rollover effects on menu
			// items that are not in the selected path
			private boolean toRepaint() {
				MenuElement[] selectedMenuPath = MenuSelectionManager
						.defaultManager().getSelectedPath();
				for (MenuElement elem : selectedMenuPath) {
					if (elem == SubstanceMenuUI.this.menuItem) {
						return true;
					}
				}
				return (selectedMenuPath.length == 0);
			}

			@Override
			public void focusLost(FocusEvent e) {
				if (toRepaint()) {
					SubstanceMenuUI.this.menuItem.getModel().setRollover(false);
					SubstanceMenuUI.this.menuItem.repaint();
				}
			}
		};
		this.menuItem.addFocusListener(this.substanceFocusListener);

		final Set<FadeKind> toIgnore = new HashSet<FadeKind>();
		// toIgnore.add(FadeKind.ROLLOVER);
		this.substanceFadeStateListener = new FadeStateListener(this.menuItem,
				this.menuItem.getModel(), SubstanceCoreUtilities
						.getFadeCallback(this.menuItem, this.menuItem
								.getModel(), false), toIgnore);
		// new FadeTracker.FadeTrackerCallback() {
		// private boolean toIgnore(FadeKind fadeKind) {
		// if (fadeKind != FadeKind.SELECTION)
		// return false;
		// return true;
		// }
		//
		// public void fadeEnded(FadeKind fadeKind) {
		// if (toIgnore(fadeKind))
		// return;
		// menuItem.repaint();
		// }
		//
		// public void fadePerformed(FadeKind fadeKind,
		// float fadeCycle10) {
		// if (toIgnore(fadeKind))
		// return;
		// menuItem.repaint();
		// }
		// });
		this.substanceFadeStateListener.registerListeners();

		// this.menuItem.getModel().addChangeListener(new ChangeListener() {
		// public void stateChanged(ChangeEvent e) {
		// ButtonModel bm = menuItem.getModel();
		// System.out.println(menuItem.getText() + " e:" + bm.isEnabled()
		// + ":a:" + bm.isArmed() + ":r:" + bm.isRollover()
		// + ":p:" + bm.isPressed() + ":s:" + bm.isSelected());
		// }
		// });

		this.substancePropertyListener = new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				if (AbstractButton.MODEL_CHANGED_PROPERTY.equals(evt
						.getPropertyName())) {
					if (substanceFadeStateListener != null)
						substanceFadeStateListener.unregisterListeners();
					substanceFadeStateListener = new FadeStateListener(
							menuItem, menuItem.getModel(),
							SubstanceCoreUtilities.getFadeCallback(menuItem,
									menuItem.getModel(), false), toIgnore);
					substanceFadeStateListener.registerListeners();
				}
				if ("font".equals(evt.getPropertyName())) {
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							menuItem.updateUI();
						}
					});
				}
			}
		};
		this.menuItem.addPropertyChangeListener(this.substancePropertyListener);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicMenuUI#uninstallListeners()
	 */
	@Override
	protected void uninstallListeners() {
		super.uninstallListeners();

		// Improving performance on big menus.
		this.substanceMenuPropertyListener.uninstall();
		this.substanceMenuPropertyListener = null;

		// fix for enhancement 93 - rollover effects on menu items
		this.menuItem.removeMouseListener(this.substanceMouseListener);
		this.substanceMouseListener = null;
		this.menuItem.removeFocusListener(this.substanceFocusListener);
		this.substanceFocusListener = null;

		this.menuItem
				.removePropertyChangeListener(this.substancePropertyListener);
		this.substancePropertyListener = null;

		this.substanceFadeStateListener.unregisterListeners();
		this.substanceFadeStateListener = null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicMenuItemUI#paintBackground(java.awt.Graphics,
	 *      javax.swing.JMenuItem, java.awt.Color)
	 */
	@Override
	protected void paintBackground(Graphics g, JMenuItem menuItem, Color bgColor) {
		// try {
		// throw new IOException();
		// } catch (IOException ioe) {
		// System.err.println(new Date());
		// ioe.printStackTrace();
		// }
		int textOffset = MenuUtilities.getTextOffset(g, menuItem);
		if (menuItem.getParent() instanceof JMenuBar)
			textOffset = 0;
		SubstanceMenuUI.backgroundDelegate.paintBackground(g, menuItem,
				bgColor, 0.5f, textOffset);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicMenuItemUI#paintText(java.awt.Graphics,
	 *      javax.swing.JMenuItem, java.awt.Rectangle, java.lang.String)
	 */
	@Override
	protected void paintText(Graphics g, JMenuItem menuItem,
			Rectangle textRect, String text) {
		// workaround bug 233 - the font of the graphics doesn't always
		// match with the font of the menu item
		g.setFont(menuItem.getFont());
		// fix for defect 125 - support of RTL menus
		MenuUtilities.lineupTextRectangle(g, menuItem, textRect,
				this.defaultTextIconGap);
		SubstanceCoreUtilities.paintMenuText(menuItem, g, text, textRect);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.SubstanceMenu#getAssociatedMenuItem()
	 */
	public JMenuItem getAssociatedMenuItem() {
		return this.menuItem;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.SubstanceMenu#getAcceleratorFont()
	 */
	public Font getAcceleratorFont() {
		return this.acceleratorFont;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.SubstanceMenu#getArrowIcon()
	 */
	public Icon getArrowIcon() {
		return this.arrowIcon;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.SubstanceMenu#getCheckIcon()
	 */
	public Icon getCheckIcon() {
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.SubstanceMenu#getDefaultTextIconGap()
	 */
	public int getDefaultTextIconGap() {
		return this.defaultTextIconGap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicMenuItemUI#getPreferredMenuItemSize(javax.swing.JComponent,
	 *      javax.swing.Icon, javax.swing.Icon, int)
	 */
	@Override
	protected Dimension getPreferredMenuItemSize(JComponent c, Icon checkIcon,
			Icon arrowIcon, int defaultTextIconGap) {
		Dimension superDim = super.getPreferredMenuItemSize(c, checkIcon,
				arrowIcon, defaultTextIconGap);
		SubstanceMenuUI ui = (SubstanceMenuUI) this.menuItem.getUI();
		int textOffset = MenuUtilities.getTextOffset(SubstanceMenuUI.graphics,
				this.menuItem)
				- MenuUtilities.getTextOffset(SubstanceMenuUI.graphics,
						this.menuItem, ui.getAcceleratorFont(), checkIcon,
						arrowIcon, defaultTextIconGap);
		return new Dimension(superDim.width + textOffset, superDim.height);
	}

	// /*
	// * (non-Javadoc)
	// *
	// * @see javax.swing.plaf.basic.BasicMenuItemUI#update(java.awt.Graphics,
	// * javax.swing.JComponent)
	// */
	// @Override
	// public void update(Graphics g, JComponent c) {
	// super.update(g, c);
	//
	// if (c.getParent() instanceof JMenuBar) {
	// GhostPaintingUtils.paintGhostImages(c, g);
	// }
	// }
}
