/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.theme;

import org.jvnet.substance.color.*;

/**
 * Hue-shifted theme. A hue-shifted theme is based on some original theme that
 * is hue-shifted in HSB space. This class is part of officially supported API.
 * 
 * @author Kirill Grouchnikov
 * @since version 3.1
 */
public class SubstanceHueShiftTheme extends SubstanceWrapperTheme {
	/**
	 * The hue-shift factor of <code>this</code> theme.
	 */
	private double hueShiftFactor;

	/**
	 * Creates a new hue-shift theme.<b>Do not</b> use this constructor
	 * directly, use {@link SubstanceTheme#hueShift(double)} instead.
	 * 
	 * @param substanceTheme
	 *            The original theme.
	 * @param hueShiftFactor
	 *            Hue-shift factor.
	 */
	SubstanceHueShiftTheme(SubstanceTheme substanceTheme, double hueShiftFactor) {
		super(substanceTheme, new HueShiftColorScheme(substanceTheme
				.getColorScheme(), hueShiftFactor), "Hue-shift "
				+ substanceTheme.getDisplayName() + " "
				+ (int) (100 * hueShiftFactor) + "%", substanceTheme.getKind());
		this.hueShiftFactor = hueShiftFactor;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getDefaultTheme()
	 */
	@Override
	public SubstanceTheme getDefaultTheme() {
		if (this.defaultTheme == null)
			this.defaultTheme = this.originalTheme.getDefaultTheme().hueShift(
					this.hueShiftFactor);
		return this.defaultTheme;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getDisabledTheme()
	 */
	@Override
	public SubstanceTheme getDisabledTheme() {
		if (this.disabledTheme == null)
			this.disabledTheme = this.originalTheme.getDisabledTheme()
					.hueShift(this.hueShiftFactor);
		return this.disabledTheme;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#hueShift(double)
	 */
	@Override
	public SubstanceTheme hueShift(double hueShiftFactor) {
		hueShiftFactor += this.hueShiftFactor;
		if (hueShiftFactor != 0.0)
			return this.getOriginalTheme().hueShift(hueShiftFactor);
		else
			return this.getOriginalTheme();
	}
}
