/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.theme;

import java.awt.*;
import java.awt.event.KeyEvent;

import javax.swing.*;
import javax.swing.border.EmptyBorder;
import javax.swing.border.LineBorder;
import javax.swing.plaf.*;
import javax.swing.plaf.basic.BasicBorders;
import javax.swing.plaf.basic.BasicBorders.MarginBorder;
import javax.swing.text.DefaultEditorKit;

import org.jvnet.lafwidget.animation.FadeKind;
import org.jvnet.substance.*;
import org.jvnet.substance.color.*;
import org.jvnet.substance.painter.StandardGradientPainter;
import org.jvnet.substance.painter.SubstanceGradientPainter;
import org.jvnet.substance.scroll.SubstanceScrollPaneBorder;
import org.jvnet.substance.title.SubstanceHeaderPainter;
import org.jvnet.substance.title.SubstanceTitlePainter;
import org.jvnet.substance.utils.*;
import org.jvnet.substance.utils.icon.CheckBoxMenuItemIcon;
import org.jvnet.substance.utils.icon.RadioButtonMenuItemIcon;

/**
 * Theme for <code>Substance</code> look and feel. This class is part of
 * officially supported API.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceTheme implements SubstanceTrait {
	/**
	 * Color scheme for <code>this</code> theme.
	 */
	private ColorScheme colorScheme;

	/**
	 * Theme kind for <code>this</code> theme.
	 */
	private ThemeKind kind;

	/**
	 * Display name for <code>this</code> theme.
	 */
	private String displayName;

	/**
	 * Theme kind enumeration.
	 * 
	 * @author Kirill Grouchnikov.
	 */
	public enum ThemeKind {
		/**
		 * Bright themes.
		 */
		BRIGHT,

		/**
		 * Cold themes.
		 */
		COLD,

		/**
		 * Dark themes.
		 */
		DARK,

		/**
		 * Inverted themes.
		 */
		INVERTED,

		/**
		 * Mixed themes.
		 */
		MIXED,

		/**
		 * Negated themes.
		 */
		NEGATED
	}

	// /**
	// * Contains all themes.
	// */
	// private static Map<String, SubstanceTheme> allThemes = new
	// HashMap<String, SubstanceTheme>();

	/**
	 * Used as reference to the default color scheme. This field is <b>for
	 * internal use only</b>.
	 */
	public static ColorScheme AQUA = new AquaColorScheme();

	/**
	 * Used as reference to the green color scheme. This field is <b>for
	 * internal use only</b>.
	 */
	public static ColorScheme GREEN = new BottleGreenColorScheme();

	/**
	 * Used as reference in attention-drawing animations. This field is <b>for
	 * internal use only</b>.
	 */
	public static ColorScheme YELLOW = new SunGlareColorScheme();

	/**
	 * Used as reference in attention-drawing animations. This field is <b>for
	 * internal use only</b>.
	 */
	public static ColorScheme ORANGE = new SunfireRedColorScheme();

	/**
	 * Used as reference to default scheme of non-dark themes. This field is
	 * <b>for internal use only</b>.
	 */
	private static ColorScheme METALLIC_SCHEME = new MetallicColorScheme();

	/**
	 * Used as reference to default scheme of inverted themes. This field is
	 * <b>for internal use only</b>.
	 */
	private static ColorScheme INVERTED_METALLIC_SCHEME = new InvertedColorScheme(
			METALLIC_SCHEME);

	/**
	 * Used as reference to default scheme of negated themes. This field is
	 * <b>for internal use only</b>.
	 */
	private static ColorScheme NEGATED_METALLIC_SCHEME = new NegatedColorScheme(
			METALLIC_SCHEME);

	/**
	 * Used as reference to default scheme of dark themes. This field is <b>for
	 * internal use only</b>.
	 */
	private static ColorScheme DARK_METALLIC_SCHEME = new DarkMetallicColorScheme();

	/**
	 * Used as reference to disabled scheme of dark themes. This field is <b>for
	 * internal use only</b>.
	 */
	private static ColorScheme DARK_GRAY_SCHEME = new DarkGrayColorScheme();

	/**
	 * Used as reference to disabled scheme of non-dark themes. This field is
	 * <b>for internal use only</b>.
	 */
	private static ColorScheme LIGHT_GRAY_SCHEME = new LightGrayColorScheme();

	/**
	 * Used as reference to disabled scheme of inverted themes. This field is
	 * <b>for internal use only</b>.
	 */
	private static ColorScheme INVERTED_LIGHT_GRAY_SCHEME = new InvertedColorScheme(
			LIGHT_GRAY_SCHEME);

	/**
	 * Used as reference to disabled scheme of negated themes. This field is
	 * <b>for internal use only</b>.
	 */
	private static ColorScheme NEGATED_LIGHT_GRAY_SCHEME = new NegatedColorScheme(
			LIGHT_GRAY_SCHEME);

	/**
	 * Used as reference to default theme of non-dark themes. This field is
	 * <b>for internal use only</b>.
	 */
	private static SubstanceTheme METALLIC = new SubstanceTheme(
			METALLIC_SCHEME, "Metallic", ThemeKind.COLD);

	/**
	 * Used as reference to default theme of inverted themes. This field is
	 * <b>for internal use only</b>.
	 */
	private static SubstanceTheme INVERTED_METALLIC = new SubstanceTheme(
			INVERTED_METALLIC_SCHEME, "Inverted Metallic", ThemeKind.INVERTED);

	/**
	 * Used as reference to default theme of negated themes. This field is
	 * <b>for internal use only</b>.
	 */
	private static SubstanceTheme NEGATED_METALLIC = new SubstanceTheme(
			NEGATED_METALLIC_SCHEME, "Negated Metallic", ThemeKind.NEGATED);

	/**
	 * Used as reference to default theme of dark themes. This field is <b>for
	 * internal use only</b>.
	 */
	private static SubstanceTheme DARK_METALLIC = new SubstanceTheme(
			DARK_METALLIC_SCHEME, "Dark Metallic", ThemeKind.DARK);

	/**
	 * Used as reference to disabled theme of non-dark themes. This field is
	 * <b>for internal use only</b>.
	 */
	private static SubstanceTheme LIGHT_GRAY = new SubstanceTheme(
			LIGHT_GRAY_SCHEME, "Light Gray", ThemeKind.COLD);

	/**
	 * Used as reference to disabled theme of dark themes. This field is <b>for
	 * internal use only</b>.
	 */
	private static SubstanceTheme DARK_GRAY = LIGHT_GRAY.negate();// new

	// SubstanceTheme(

	// DARK_GRAY_SCHEME, "Dark Gray", ThemeKind.DARK);

	/**
	 * Used as reference to disabled theme of inverted themes. This field is
	 * <b>for internal use only</b>.
	 */
	private static SubstanceTheme INVERTED_LIGHT_GRAY = new SubstanceTheme(
			INVERTED_LIGHT_GRAY_SCHEME, "Inverted Light Gray",
			ThemeKind.INVERTED);

	/**
	 * Used as reference to disabled theme of negated themes. This field is
	 * <b>for internal use only</b>.
	 */
	private static SubstanceTheme NEGATED_LIGHT_GRAY = new SubstanceTheme(
			NEGATED_LIGHT_GRAY_SCHEME, "Negated Light Gray", ThemeKind.NEGATED);

	/**
	 * Theme for pressed states.
	 */
	private SubstanceTheme pressedTheme;

	// /**
	// * Contains classes of all controls that should be painted as active.
	// */
	// protected Collection<Class> activeControls;
	//
	/**
	 * Simple constructor. This constructor is part of officially supported API.
	 * 
	 * @param colorScheme
	 *            Color scheme for <code>this</code> theme.
	 * @param displayName
	 *            Display name for <code>this</code> theme.
	 * @param themeKind
	 *            Kind of <code>this</code> theme.
	 */
	public SubstanceTheme(ColorScheme colorScheme, String displayName,
			ThemeKind themeKind) {
		this.kind = themeKind;
		this.colorScheme = colorScheme;
		this.displayName = displayName;

		if (colorScheme instanceof AquaColorScheme)
			AQUA = colorScheme;

		// allThemes.put(displayName, this);
		// this.activeControls = new HashSet<Class>();
	}

	/**
	 * Returns the color scheme of <code>this</code> theme. This method is
	 * part of officially supported API.
	 * 
	 * @return Color scheme of <code>this</code> theme.
	 */
	public ColorScheme getColorScheme() {
		return this.colorScheme;
	}

	/**
	 * Returns the name of <code>this</code> theme. This method is part of
	 * officially supported API.
	 * 
	 * @return The name of <code>this</code> theme.
	 */
	public String getThemeName() {
		return this.getDisplayName();
	}

	/**
	 * Adds theme-specific entries to the UI defaults table.
	 * 
	 * @param table
	 *            UI defaults table.
	 */
	public void addCustomEntriesToTable(UIDefaults table) {
		// Apparently this function is called with null table
		// when the application is run with -Dswing.defaultlaf
		// setting. In this case, this function will be called
		// second time with correct table.
		if (table == null) {
			return;
		}

		Object menuArrowIcon = new UIDefaults.LazyValue() {
			public Object createValue(UIDefaults table) {
				// fix for defect 125 - support of RTL menus
				return new Icon() {
					Icon ltrIcon = SubstanceImageCreator.getArrowIcon(
							SubstanceSizeUtils
									.getArrowIconWidth(SubstanceSizeUtils
											.getControlFontSize()) - 2,
							SubstanceSizeUtils
									.getArrowIconHeight(SubstanceSizeUtils
											.getControlFontSize()) - 1,
							SubstanceSizeUtils
									.getArrowStrokeWidth(SubstanceSizeUtils
											.getControlFontSize()),
							SwingConstants.EAST, SubstanceCoreUtilities
									.getActiveTheme(null, true));

					Icon rtlIcon = SubstanceImageCreator.getArrowIcon(
							SubstanceSizeUtils
									.getArrowIconWidth(SubstanceSizeUtils
											.getControlFontSize()) - 2,
							SubstanceSizeUtils
									.getArrowIconHeight(SubstanceSizeUtils
											.getControlFontSize()) - 1,
							SubstanceSizeUtils
									.getArrowStrokeWidth(SubstanceSizeUtils
											.getControlFontSize()),
							SwingConstants.WEST, SubstanceCoreUtilities
									.getActiveTheme(null, true));

					Icon ltrIconDisabled = SubstanceImageCreator.getArrowIcon(
							SubstanceSizeUtils
									.getArrowIconWidth(SubstanceSizeUtils
											.getControlFontSize()) - 2,
							SubstanceSizeUtils
									.getArrowIconHeight(SubstanceSizeUtils
											.getControlFontSize()) - 1,
							SubstanceSizeUtils
									.getArrowStrokeWidth(SubstanceSizeUtils
											.getControlFontSize()),
							SwingConstants.EAST, SubstanceCoreUtilities
									.getDisabledTheme(null, true));

					Icon rtlIconDisabled = SubstanceImageCreator.getArrowIcon(
							SubstanceSizeUtils
									.getArrowIconWidth(SubstanceSizeUtils
											.getControlFontSize()) - 2,
							SubstanceSizeUtils
									.getArrowIconHeight(SubstanceSizeUtils
											.getControlFontSize()) - 1,
							SubstanceSizeUtils
									.getArrowStrokeWidth(SubstanceSizeUtils
											.getControlFontSize()),
							SwingConstants.WEST, SubstanceCoreUtilities
									.getDisabledTheme(null, true));

					public int getIconHeight() {
						return 7;
					}

					public int getIconWidth() {
						return 5;
					}

					public void paintIcon(Component c, Graphics g, int x, int y) {
						if (c.getComponentOrientation().isLeftToRight()) {
							if (c.isEnabled())
								this.ltrIcon.paintIcon(c, g, x, y);
							else
								this.ltrIconDisabled.paintIcon(c, g, x, y);
						} else {
							if (c.isEnabled())
								this.rtlIcon.paintIcon(c, g, x, y);
							else
								this.rtlIconDisabled.paintIcon(c, g, x, y);
						}
					}
				};
			}
		};

		Object listCellRendererActiveValue = new UIDefaults.ActiveValue() {
			public Object createValue(UIDefaults table) {
				return new SubstanceDefaultListCellRenderer.SubstanceUIResource();
			}
		};

		Color controlText = new ColorUIResource(this.colorScheme
				.getLightColor());
		final Color foregroundColor = this.getForegroundColor();
		// final Color menuForegroundColor = new ColorUIResource(
		// SubstanceColorUtilities.getMenuForegroundColor(this));
		Color backgroundColor = new ColorUIResource(this.getBackgroundColor());
		// Color menuBackgroundColor = new
		// ColorUIResource(SubstanceColorUtilities
		// .getBackgroundColor(this.getActiveTitlePaneTheme()));
		Color backgroundLightColor = new ColorUIResource(this
				.getLightBackgroundColor());

		Color disabledForegroundColor = new ColorUIResource(this
				.getDisabledForegroundColor());
		// Color menuDisabledForegroundColor = new ColorUIResource(
		// SubstanceColorUtilities.getDisabledForegroundColor(this
		// .getActiveTitlePaneTheme()));
		Color disabledBackgroundColor = new ColorUIResource(this
				.getDisabledBackgroundColor());

		Color lineColor = new ColorUIResource(this.getLineColor());

		Color lineColorDefault = new ColorUIResource(this.getLineColorDefault());

		int lcb = SubstanceColorUtilities
				.getColorBrightness(lineColor.getRGB());
		Color lineBwColor = new ColorUIResource(new Color(lcb, lcb, lcb));

		Color selectionBackgroundColor = new ColorUIResource(this
				.getSelectionBackgroundColor());

		Color selectionForegroundColor = new ColorUIResource(this
				.getSelectionForegroundColor());

		// Color selectionCellBackgroundColor = new ColorUIResource(
		// SubstanceColorUtilities.getSelectionBackgroundColor(this.getHighlightBackgroundTheme()));
		//
		Color selectionCellForegroundColor = new ColorUIResource(this
				.getSelectionCellForegroundColor());

		Object regularMarginBorder = new BorderUIResource.CompoundBorderUIResource(
				new SubstanceBorder(), new BasicBorders.MarginBorder());

		Object regularBorder = new SubstanceBorder();

		Object textBorder = new BorderUIResource.CompoundBorderUIResource(
				new SubstanceBorder(SubstanceSizeUtils
						.getTextBorderInsets(SubstanceSizeUtils
								.getControlFontSize())),
				new BasicBorders.MarginBorder());

		// Object comboBorder = new BorderUIResource.CompoundBorderUIResource(
		// new SubstanceBorder(new Insets(1, 2, 1, 2)),
		// new BasicBorders.MarginBorder());
		//
		Object comboBorder = new SubstanceBorder(SubstanceSizeUtils
				.getComboBorderInsets(SubstanceSizeUtils.getControlFontSize()));

		Object spinnerBorder = new BorderUIResource.CompoundBorderUIResource(
				new SubstanceBorder(SubstanceSizeUtils
						.getSpinnerBorderInsets(SubstanceSizeUtils
								.getControlFontSize())),
				new BasicBorders.MarginBorder());

		ColorScheme activeScheme = this.getActiveTheme().getColorScheme();

		ColorScheme defaultScheme = this.getDefaultColorScheme();

		// special handling of mixed dark themes
		final SubstanceTheme topDecorationTheme = this
				.getActiveTitlePaneTheme();
		// (this.getKind() == ThemeKind.DARK_MIXED) ? ((SubstanceMixDarkBiTheme)
		// this)
		// .getOriginalDarkTheme() : this;

		SubstanceTitlePainter titlePainter = SubstanceLookAndFeel
				.getCurrentTitlePainter();
		boolean shouldUseHeaderPainter = titlePainter instanceof SubstanceHeaderPainter;

		Object fieldInputMap = new UIDefaults.LazyInputMap(new Object[] {
				"ctrl C", DefaultEditorKit.copyAction,

				"ctrl V", DefaultEditorKit.pasteAction,

				"ctrl X", DefaultEditorKit.cutAction,

				"COPY", DefaultEditorKit.copyAction,

				"PASTE", DefaultEditorKit.pasteAction,

				"CUT", DefaultEditorKit.cutAction,

				"shift LEFT", DefaultEditorKit.selectionBackwardAction,

				"shift KP_LEFT", DefaultEditorKit.selectionBackwardAction,

				"shift RIGHT", DefaultEditorKit.selectionForwardAction,

				"shift KP_RIGHT", DefaultEditorKit.selectionForwardAction,

				"ctrl LEFT", DefaultEditorKit.previousWordAction,

				"ctrl KP_LEFT", DefaultEditorKit.previousWordAction,

				"ctrl RIGHT", DefaultEditorKit.nextWordAction,

				"ctrl KP_RIGHT", DefaultEditorKit.nextWordAction,

				"ctrl shift LEFT",
				DefaultEditorKit.selectionPreviousWordAction,

				"ctrl shift KP_LEFT",
				DefaultEditorKit.selectionPreviousWordAction,

				"ctrl shift RIGHT", DefaultEditorKit.selectionNextWordAction,

				"ctrl shift KP_RIGHT",
				DefaultEditorKit.selectionNextWordAction,

				"ctrl A", DefaultEditorKit.selectAllAction,

				"HOME", DefaultEditorKit.beginLineAction,

				"END", DefaultEditorKit.endLineAction,

				"shift HOME", DefaultEditorKit.selectionBeginLineAction,

				"shift END", DefaultEditorKit.selectionEndLineAction,

				"BACK_SPACE", DefaultEditorKit.deletePrevCharAction,

				"ctrl H", DefaultEditorKit.deletePrevCharAction,

				"DELETE", DefaultEditorKit.deleteNextCharAction,

				"RIGHT", DefaultEditorKit.forwardAction,

				"LEFT", DefaultEditorKit.backwardAction,

				"KP_RIGHT", DefaultEditorKit.forwardAction,

				"KP_LEFT", DefaultEditorKit.backwardAction,

				"ENTER", JTextField.notifyAction,

				"ctrl BACK_SLASH",
				"unselect"/* DefaultEditorKit.unselectAction */,

				"control shift O", "toggle-componentOrientation"/* DefaultEditorKit.toggleComponentOrientation */
		});

		Object passwordInputMap = new UIDefaults.LazyInputMap(new Object[] {
				"ctrl C", DefaultEditorKit.copyAction,

				"ctrl V", DefaultEditorKit.pasteAction,

				"ctrl X", DefaultEditorKit.cutAction,

				"COPY", DefaultEditorKit.copyAction,

				"PASTE", DefaultEditorKit.pasteAction,

				"CUT", DefaultEditorKit.cutAction,

				"shift LEFT", DefaultEditorKit.selectionBackwardAction,

				"shift KP_LEFT", DefaultEditorKit.selectionBackwardAction,

				"shift RIGHT", DefaultEditorKit.selectionForwardAction,

				"shift KP_RIGHT", DefaultEditorKit.selectionForwardAction,

				"ctrl LEFT", DefaultEditorKit.beginLineAction,

				"ctrl KP_LEFT", DefaultEditorKit.beginLineAction,

				"ctrl RIGHT", DefaultEditorKit.endLineAction,

				"ctrl KP_RIGHT", DefaultEditorKit.endLineAction,

				"ctrl shift LEFT", DefaultEditorKit.selectionBeginLineAction,

				"ctrl shift KP_LEFT",
				DefaultEditorKit.selectionBeginLineAction,

				"ctrl shift RIGHT", DefaultEditorKit.selectionEndLineAction,

				"ctrl shift KP_RIGHT", DefaultEditorKit.selectionEndLineAction,

				"ctrl A", DefaultEditorKit.selectAllAction,

				"HOME", DefaultEditorKit.beginLineAction,

				"END", DefaultEditorKit.endLineAction,

				"shift HOME", DefaultEditorKit.selectionBeginLineAction,

				"shift END", DefaultEditorKit.selectionEndLineAction,

				"BACK_SPACE", DefaultEditorKit.deletePrevCharAction,

				"ctrl H", DefaultEditorKit.deletePrevCharAction,

				"DELETE", DefaultEditorKit.deleteNextCharAction,

				"RIGHT", DefaultEditorKit.forwardAction,

				"LEFT", DefaultEditorKit.backwardAction,

				"KP_RIGHT", DefaultEditorKit.forwardAction,

				"KP_LEFT", DefaultEditorKit.backwardAction,

				"ENTER", JTextField.notifyAction,

				"ctrl BACK_SLASH",
				"unselect"/* DefaultEditorKit.unselectAction */,

				"control shift O", "toggle-componentOrientation"/* DefaultEditorKit.toggleComponentOrientation */
		});

		Object multilineInputMap = new UIDefaults.LazyInputMap(new Object[] {
				"ctrl C", DefaultEditorKit.copyAction,

				"ctrl V", DefaultEditorKit.pasteAction,

				"ctrl X", DefaultEditorKit.cutAction,

				"COPY", DefaultEditorKit.copyAction,

				"PASTE", DefaultEditorKit.pasteAction,

				"CUT", DefaultEditorKit.cutAction,

				"shift LEFT", DefaultEditorKit.selectionBackwardAction,

				"shift KP_LEFT", DefaultEditorKit.selectionBackwardAction,

				"shift RIGHT", DefaultEditorKit.selectionForwardAction,

				"shift KP_RIGHT", DefaultEditorKit.selectionForwardAction,

				"ctrl LEFT", DefaultEditorKit.previousWordAction,

				"ctrl KP_LEFT", DefaultEditorKit.previousWordAction,

				"ctrl RIGHT", DefaultEditorKit.nextWordAction,

				"ctrl KP_RIGHT", DefaultEditorKit.nextWordAction,

				"ctrl shift LEFT",
				DefaultEditorKit.selectionPreviousWordAction,

				"ctrl shift KP_LEFT",
				DefaultEditorKit.selectionPreviousWordAction,

				"ctrl shift RIGHT", DefaultEditorKit.selectionNextWordAction,

				"ctrl shift KP_RIGHT",
				DefaultEditorKit.selectionNextWordAction,

				"ctrl A", DefaultEditorKit.selectAllAction,

				"HOME", DefaultEditorKit.beginLineAction,

				"END", DefaultEditorKit.endLineAction,

				"shift HOME", DefaultEditorKit.selectionBeginLineAction,

				"shift END", DefaultEditorKit.selectionEndLineAction,

				"UP", DefaultEditorKit.upAction,

				"KP_UP", DefaultEditorKit.upAction,

				"DOWN", DefaultEditorKit.downAction,

				"KP_DOWN", DefaultEditorKit.downAction,

				"PAGE_UP", DefaultEditorKit.pageUpAction,

				"PAGE_DOWN", DefaultEditorKit.pageDownAction,

				"shift PAGE_UP", "selection-page-up",

				"shift PAGE_DOWN", "selection-page-down",

				"ctrl shift PAGE_UP", "selection-page-left",

				"ctrl shift PAGE_DOWN", "selection-page-right",

				"shift UP", DefaultEditorKit.selectionUpAction,

				"shift KP_UP", DefaultEditorKit.selectionUpAction,

				"shift DOWN", DefaultEditorKit.selectionDownAction,

				"shift KP_DOWN", DefaultEditorKit.selectionDownAction,

				"ENTER", DefaultEditorKit.insertBreakAction,

				"BACK_SPACE", DefaultEditorKit.deletePrevCharAction,

				"ctrl H", DefaultEditorKit.deletePrevCharAction,

				"DELETE", DefaultEditorKit.deleteNextCharAction,

				"RIGHT", DefaultEditorKit.forwardAction,

				"LEFT", DefaultEditorKit.backwardAction,

				"KP_RIGHT", DefaultEditorKit.forwardAction,

				"KP_LEFT", DefaultEditorKit.backwardAction,

				"TAB", DefaultEditorKit.insertTabAction,

				"ctrl BACK_SLASH",
				"unselect"/* DefaultEditorKit.unselectAction */,

				"ctrl HOME", DefaultEditorKit.beginAction,

				"ctrl END", DefaultEditorKit.endAction,

				"ctrl shift HOME", DefaultEditorKit.selectionBeginAction,

				"ctrl shift END", DefaultEditorKit.selectionEndAction,

				"ctrl T", "next-link-action",

				"ctrl shift T", "previous-link-action",

				"ctrl SPACE", "activate-link-action",

				"control shift O", "toggle-componentOrientation"/* DefaultEditorKit.toggleComponentOrientation */
		});

		Icon emptyIcon = new IconUIResource(new Icon() {
			public int getIconHeight() {
				return 2;
			}

			public int getIconWidth() {
				return 2;
			}

			public void paintIcon(Component c, Graphics g, int x, int y) {
			}
		});

		Object[] defaults = new Object[] {
				"control",
				controlText,

				"TextField.focusInputMap",
				fieldInputMap,

				"PasswordField.focusInputMap",
				passwordInputMap,

				"TextArea.focusInputMap",
				multilineInputMap,

				"TextPane.focusInputMap",
				multilineInputMap,

				"EditorPane.focusInputMap",
				multilineInputMap,

				// "MenuBarUI",
				// "org.jvnet.substance.SubstanceMenuBarUI",

				"Button.defaultButtonFollowsFocus",
				Boolean.FALSE,

				"Button.disabledText",
				disabledForegroundColor,

				"Button.foreground",
				foregroundColor,

				"Button.margin",
				new InsetsUIResource(0, 0, 0, 0),

				"CheckBox.background",
				backgroundColor,

				"CheckBox.border",
				new BorderUIResource.CompoundBorderUIResource(
						SubstanceSizeUtils.getCheckBoxBorder(SubstanceSizeUtils
								.getControlFontSize()), new MarginBorder()),

				"CheckBox.disabledText",
				disabledForegroundColor,

				"CheckBox.foreground",
				foregroundColor,

				"CheckBoxMenuItem.acceleratorForeground",
				foregroundColor,

				"CheckBoxMenuItem.acceleratorSelectionForeground",
				foregroundColor,

				"CheckBoxMenuItem.background",
				backgroundColor,

				"CheckBoxMenuItem.borderPainted",
				Boolean.FALSE,

				"CheckBoxMenuItem.checkIcon",
				new CheckBoxMenuItemIcon(null, 1 + SubstanceSizeUtils
						.getMenuCheckMarkSize(SubstanceSizeUtils
								.getControlFontSize())),
				// new UIDefaults.LazyValue() {
				// public Object createValue(UIDefaults table) {
				// return new IconUIResource(SubstanceIconFactory
				// .getCheckBoxMenuItemIcon(1 + SubstanceSizeUtils
				// .getMenuCheckMarkSize()));
				// }
				// },

				"CheckBoxMenuItem.disabledForeground",
				disabledForegroundColor,

				"CheckBoxMenuItem.foreground",
				foregroundColor,

				"CheckBoxMenuItem.selectionForeground",
				selectionCellForegroundColor,

				"ColorChooser.background",
				backgroundColor,

				"ColorChooser.foreground",
				foregroundColor,

				"ComboBox.ancestorInputMap",
				new UIDefaults.LazyInputMap(new Object[] { "ESCAPE",
						"hidePopup",

						"PAGE_UP", "pageUpPassThrough",

						"PAGE_DOWN", "pageDownPassThrough",

						"HOME", "homePassThrough",

						"END", "endPassThrough",

						"DOWN", "selectNext",

						"KP_DOWN", "selectNext",

						"alt DOWN", "togglePopup",

						"alt KP_DOWN", "togglePopup",

						"alt UP", "togglePopup",

						"alt KP_UP", "togglePopup",

						"SPACE", "spacePopup",

						"ENTER", "enterPressed",

						"UP", "selectPrevious",

						"KP_UP", "selectPrevious" }),

				"ComboBox.background",
				backgroundLightColor,

				"ComboBox.border",
				comboBorder,

				"ComboBox.disabledBackground",
				backgroundLightColor,

				"ComboBox.disabledForeground",
				disabledForegroundColor,

				"ComboBox.foreground",
				foregroundColor,

				"ComboBox.selectionBackground",
				backgroundLightColor,

				"ComboBox.selectionForeground",
				foregroundColor,

				"DesktopIcon.border",
				regularMarginBorder,

				"DesktopIcon.width",
				new Integer(140),

				"Desktop.background",
				new ColorUIResource(new Color(0x0, true)),

				"Desktop.foreground",
				foregroundColor,

				"Dialog.background",
				backgroundColor,

				"EditorPane.background",
				backgroundLightColor,

				"EditorPane.border",
				textBorder,

				"EditorPane.foreground",
				foregroundColor,

				"EditorPane.caretForeground",
				foregroundColor,

				"EditorPane.disabledBackground",
				disabledBackgroundColor,

				"EditorPane.inactiveBackground",
				disabledBackgroundColor,

				"EditorPane.inactiveForeground",
				disabledForegroundColor,

				"EditorPane.selectionBackground",
				selectionBackgroundColor,

				"EditorPane.selectionForeground",
				selectionForegroundColor,

				"FileChooser.upFolderIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/go-up.png");
					}
				},

				"FileChooser.newFolderIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/folder-new.png");
					}
				},

				"FileChooser.homeFolderIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/user-home.png");
					}
				},

				"FileChooser.listViewIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/application_view_list.png");
					}
				},

				"FileChooser.detailsViewIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/application_view_detail.png");
					}
				},

				"FileChooser.viewMenuLabelText",
				"View",

				"FileChooser.refreshActionLabelText",
				"Refresh",

				"FileChooser.newFolderActionLabelText",
				"New Folder",

				"FileChooser.listViewActionLabelText",
				"List",

				"FileChooser.detailsViewActionLabelText",
				"Details",

				"FileChooser.lookInLabelMnemonic",
				new Integer(KeyEvent.VK_I),

				"FileChooser.fileNameLabelMnemonic",
				new Integer(KeyEvent.VK_N),

				"FileChooser.filesOfTypeLabelMnemonic",
				new Integer(KeyEvent.VK_T),

				"FileChooser.usesSingleFilePane",
				Boolean.TRUE,

				"FileChooser.ancestorInputMap",
				new UIDefaults.LazyInputMap(new Object[] { "ESCAPE",
						"cancelSelection", "F2", "editFileName", "F5",
						"refresh", "BACK_SPACE", "Go Up", "ENTER",
						"approveSelection" }),

				"FileView.computerIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/computer.png");
					}
				},

				"FileView.directoryIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/folder.png");
					}
				},

				"FileView.fileIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/text-x-generic.png");
					}
				},

				"FileView.floppyDriveIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/media-floppy.png");
					}
				},

				"FileView.hardDriveIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/drive-harddisk.png");
					}
				},

				"FormattedTextField.background",
				backgroundLightColor,

				"FormattedTextField.border",
				textBorder,

				"FormattedTextField.caretForeground",
				foregroundColor,

				"FormattedTextField.disabledBackground",
				disabledBackgroundColor,

				"FormattedTextField.foreground",
				foregroundColor,

				"FormattedTextField.inactiveBackground",
				disabledBackgroundColor,

				"FormattedTextField.inactiveForeground",
				disabledForegroundColor,

				"FormattedTextField.selectionBackground",
				selectionBackgroundColor,

				"FormattedTextField.selectionForeground",
				selectionForegroundColor,

				"InternalFrame.activeTitleBackground",
				selectionForegroundColor,

				"InternalFrame.inactiveTitleBackground",
				foregroundColor,

				// "InternalFrame.activeTitleGradient",
				// selectionBackgroundColor,
				//
				// "InternalFrame.inactiveTitleGradient",
				// backgroundColor,

				"InternalFrame.border",
				new BorderUIResource(new SubstancePaneBorder()),

				"InternalFrame.closeIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceImageCreator
								.getCloseIcon(topDecorationTheme);
					}
				},

				"InternalFrame.iconifyIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceImageCreator
								.getMinimizeIcon(topDecorationTheme);
					}
				},

				// maximize2
				"InternalFrame.maximizeIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceImageCreator
								.getMaximizeIcon(topDecorationTheme);
					}
				},

				// restore
				"InternalFrame.minimizeIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceImageCreator
								.getRestoreIcon(topDecorationTheme);
					}
				},

				"InternalFrame.paletteCloseIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceImageCreator
								.getCloseIcon(topDecorationTheme);
					}
				},

				"Label.background",
				backgroundColor,

				"Label.foreground",
				foregroundColor,

				"Label.disabledText",
				disabledForegroundColor,

				"Label.disabledForeground",
				disabledForegroundColor,

				"List.background",
				backgroundColor,

				"List.cellRenderer",
				listCellRendererActiveValue,

				"List.focusCellHighlightBorder",
				regularBorder,

				"List.foreground",
				foregroundColor,

				"List.selectionBackground",
				new ColorUIResource(activeScheme.getLightColor()),

				"List.selectionForeground",
				selectionCellForegroundColor,

				"Menu.arrowIcon",
				menuArrowIcon,

				"Menu.background",
				backgroundColor,

				"Menu.borderPainted",
				Boolean.FALSE,

				"Menu.checkIcon",
				null,

				"Menu.disabledForeground",
				disabledForegroundColor,

				"Menu.foreground",
				foregroundColor,

				"Menu.selectionForeground",
				selectionCellForegroundColor,

				"MenuBar.background",
				shouldUseHeaderPainter ? new ColorUIResource(this
						.getActiveTitlePaneTheme().getColorScheme()
						.getMidColor()) : backgroundColor,

				"MenuBar.foreground",
				shouldUseHeaderPainter ? new ColorUIResource(this
						.getActiveTitlePaneTheme().getForegroundColor())
						: foregroundColor,

				"MenuBar.border",
				null,

				"MenuItem.acceleratorForeground",
				foregroundColor,

				"MenuItem.acceleratorSelectionForeground",
				foregroundColor,

				"MenuItem.background",
				backgroundColor,

				"MenuItem.borderPainted",
				Boolean.FALSE,

				"MenuItem.checkIcon",
				null,

				"MenuItem.disabledForeground",
				disabledForegroundColor,

				"MenuItem.foreground",
				foregroundColor,

				"MenuItem.selectionForeground",
				selectionCellForegroundColor,

				"OptionPane.background",
				backgroundColor,

				"OptionPane.errorIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/32/dialog-error.png");
					}
				},

				"OptionPane.foreground",
				foregroundColor,

				"OptionPane.informationIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/32/dialog-information.png");
					}
				},

				"OptionPane.messageForeground",
				foregroundColor,

				"OptionPane.questionIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/32/help-browser.png");
					}
				},

				"OptionPane.warningIcon",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return SubstanceCoreUtilities
								.getIcon("resource/32/dialog-warning.png");
					}
				},

				"Panel.background",
				backgroundColor,

				"Panel.foreground",
				foregroundColor,

				"PasswordField.background",
				backgroundLightColor,

				"PasswordField.border",
				textBorder,

				"PasswordField.caretForeground",
				foregroundColor,

				"PasswordField.disabledBackground",
				disabledBackgroundColor,

				"PasswordField.foreground",
				foregroundColor,

				"PasswordField.inactiveBackground",
				disabledBackgroundColor,

				"PasswordField.inactiveForeground",
				disabledForegroundColor,

				"PasswordField.selectionBackground",
				selectionBackgroundColor,

				"PasswordField.selectionForeground",
				selectionForegroundColor,

				"PopupMenu.background",
				backgroundColor,

				"PopupMenu.border",
				regularMarginBorder,

				"ProgressBar.border",
				new BorderUIResource(new SubstanceBorder()),

				"ProgressBar.cycleTime",
				new Integer(1000),

				"ProgressBar.repaintInterval",
				new Integer(50),

				"ProgressBar.horizontalSize",
				new DimensionUIResource(146, SubstanceSizeUtils
						.getControlFontSize()),

				"ProgressBar.verticalSize",
				new DimensionUIResource(
						SubstanceSizeUtils.getControlFontSize(), 146),

				"ProgressBar.selectionBackground",
				foregroundColor,

				"ProgressBar.selectionForeground",
				foregroundColor,

				"RadioButton.background",
				backgroundColor,

				"RadioButton.border",
				new BorderUIResource.CompoundBorderUIResource(
						SubstanceSizeUtils
								.getRadioButtonBorder(SubstanceSizeUtils
										.getControlFontSize()),
						new MarginBorder()),

				"RadioButton.foreground",
				foregroundColor,

				"RadioButton.disabledText",
				disabledForegroundColor,

				"RadioButtonMenuItem.acceleratorForeground",
				foregroundColor,

				"RadioButtonMenuItem.acceleratorSelectionForeground",
				foregroundColor,

				"RadioButtonMenuItem.background",
				backgroundColor,

				"RadioButtonMenuItem.borderPainted",
				Boolean.FALSE,

				"RadioButtonMenuItem.checkIcon",
				new RadioButtonMenuItemIcon(null, SubstanceSizeUtils
						.getMenuCheckMarkSize(SubstanceSizeUtils
								.getControlFontSize())),

				"RadioButtonMenuItem.disabledForeground",
				disabledForegroundColor,

				"RadioButtonMenuItem.foreground",
				foregroundColor,

				"RadioButtonMenuItem.selectionForeground",
				selectionCellForegroundColor,

				"RootPane.background",
				backgroundColor,

				"RootPane.border",
				new SubstancePaneBorder(),

				"ScrollBar.background",
				backgroundColor,

				"ScrollBar.width",
				new Integer(SubstanceSizeUtils
						.getScrollBarWidth(SubstanceSizeUtils
								.getControlFontSize())),

				"ScrollBar.minimumThumbSize",
				new DimensionUIResource(SubstanceSizeUtils
						.getScrollBarWidth(SubstanceSizeUtils
								.getControlFontSize()) - 2, SubstanceSizeUtils
						.getScrollBarWidth(SubstanceSizeUtils
								.getControlFontSize()) - 2),

				"ScrollPane.background",
				backgroundColor,

				"ScrollPane.foreground",
				foregroundColor,

				"ScrollPane.border",
				new SubstanceScrollPaneBorder(),

				"Separator.background",
				backgroundColor,

				"Separator.foreground",
				lineBwColor,

				"Slider.altTrackColor",
				lineColor,

				"Slider.background",
				backgroundColor,

				"Slider.darkShadow",
				lineColor,

				"Slider.focus",
				lineColor,

				"Slider.focusInsets",
				new InsetsUIResource(2, 2, 0, 2),

				"Slider.foreground",
				lineColor,

				"Slider.highlight",
				backgroundLightColor,

				// "Slider.horizontalThumbIcon",
				// new UIDefaults.LazyValue() {
				// public Object createValue(UIDefaults table) {
				// return SubstanceIconFactory.getSliderHorizontalIcon(16,
				// false);
				// }
				// },
				//
				"Slider.shadow",
				lineColor,

				"Slider.tickColor",
				foregroundColor,

				// "Slider.verticalThumbIcon",
				// new UIDefaults.LazyValue() {
				// public Object createValue(UIDefaults table) {
				// return SubstanceIconFactory.getSliderVerticalIcon(16,
				// false);
				// }
				// },

				"Spinner.arrowButtonInsets",
				null,

				"Spinner.background",
				backgroundLightColor,

				"Spinner.border",
				spinnerBorder,

				"Spinner.foreground",
				foregroundColor,

				"Spinner.editorBorderPainted",
				Boolean.TRUE,

				// "Spinner.arrowButtonInsets",
				// SubstanceSizeUtils.getSpinnerArrowButtonInsets(),

				"SplitPane.background",
				backgroundColor,

				"SplitPane.foreground",
				foregroundColor,

				"SplitPane.dividerFocusColor",
				backgroundColor,

				"SplitPaneDivider.draggingColor",
				backgroundColor,

				"SplitPane.border",
				new BorderUIResource(new EmptyBorder(0, 0, 0, 0)),

				"SplitPane.dividerSize",
				new Integer(SubstanceSizeUtils
						.getArrowIconWidth(SubstanceSizeUtils
								.getControlFontSize())
						+ SubstanceSizeUtils.getAdjustedSize(SubstanceSizeUtils
								.getControlFontSize(), -1, 6, -1, true)),

				"SplitPaneDivider.border",
				new BorderUIResource(new EmptyBorder(1, 1, 1, 1)),

				"TabbedPane.tabAreaBackground",
				backgroundColor,

				"TabbedPane.unselectedBackground",
				backgroundColor,

				"TabbedPane.background",
				backgroundColor,

				"TabbedPane.borderHightlightColor",
				new ColorUIResource(activeScheme.getMidColor()),

				"TabbedPane.contentAreaColor",
				new ColorUIResource(this.colorScheme.getUltraLightColor()),

				"TabbedPane.contentBorderInsets",
				new InsetsUIResource(4, 4, 4, 4),

				"TabbedPane.contentOpaque",
				Boolean.FALSE,

				"TabbedPane.darkShadow",
				(SubstanceCoreUtilities.isThemeDark(this) ? new ColorUIResource(
						activeScheme.getDarkColor())
						: new ColorUIResource(activeScheme.getMidColor())),

				"TabbedPane.focus",
				foregroundColor,

				"TabbedPane.foreground",
				foregroundColor,

				"TabbedPane.highlight",
				new ColorUIResource(activeScheme.getLightColor()),

				"TabbedPane.light",
				(SubstanceCoreUtilities.isThemeDark(this) ? new ColorUIResource(
						SubstanceColorUtilities.getAlphaColor(defaultScheme
								.getUltraDarkColor(), 100))
						: new ColorUIResource(defaultScheme.getLightColor())),

				"TabbedPane.selected",
				new ColorUIResource(activeScheme.getExtraLightColor()),

				"TabbedPane.selectedForeground",
				foregroundColor,

				"TabbedPane.selectHighlight",
				new ColorUIResource(activeScheme.getMidColor()),

				"TabbedPane.shadow",
				new ColorUIResource(SubstanceColorUtilities
						.getInterpolatedColor(defaultScheme
								.getExtraLightColor(), defaultScheme
								.getLightColor(), 0.5)),

				"TabbedPane.tabRunOverlay",
				new Integer(0),

				"Table.background",
				backgroundColor,

				"Table.focusCellBackground",
				selectionBackgroundColor,

				"Table.focusCellForeground",
				foregroundColor,

				"Table.focusCellHighlightBorder",
				new SubstanceBorder(),
				// new UIDefaults.LazyValue() {
				// public Object createValue(UIDefaults table) {
				// return new BorderUIResource(new SubstanceBorder());
				// }
				// },

				"Table.foreground",
				foregroundColor,

				"Table.gridColor",
				lineColorDefault,

				"Table.scrollPaneBorder",
				regularMarginBorder,

				"Table.selectionBackground",
				selectionBackgroundColor,

				"Table.selectionForeground",
				selectionCellForegroundColor,

				"TableHeader.cellBorder",
				null,

				"TableHeader.foreground",
				foregroundColor,

				"TableHeader.background",
				backgroundColor,

				"TextArea.background",
				backgroundLightColor,

				"TextArea.border",
				textBorder,

				"TextArea.caretForeground",
				foregroundColor,

				"TextArea.disabledBackground",
				disabledBackgroundColor,

				"TextArea.foreground",
				foregroundColor,

				"TextArea.inactiveBackground",
				disabledBackgroundColor,

				"TextArea.inactiveForeground",
				disabledForegroundColor,

				"TextArea.selectionBackground",
				selectionBackgroundColor,

				"TextArea.selectionForeground",
				selectionForegroundColor,

				"TextField.background",
				backgroundLightColor,

				"TextField.border",
				textBorder,

				"TextField.caretForeground",
				foregroundColor,

				"TextField.disabledBackground",
				disabledBackgroundColor,

				"TextField.foreground",
				foregroundColor,

				"TextField.inactiveBackground",
				disabledBackgroundColor,

				"TextField.inactiveForeground",
				disabledForegroundColor,

				"TextField.selectionBackground",
				selectionBackgroundColor,

				"TextField.selectionForeground",
				selectionForegroundColor,

				"TextPane.background",
				backgroundLightColor,

				"TextPane.border",
				textBorder,

				"TextPane.disabledBackground",
				disabledBackgroundColor,

				"TextPane.foreground",
				foregroundColor,

				"TextPane.caretForeground",
				foregroundColor,

				"TextPane.inactiveBackground",
				disabledBackgroundColor,

				"TextPane.inactiveForeground",
				disabledForegroundColor,

				"TextPane.selectionBackground",
				selectionBackgroundColor,

				"TextPane.selectionForeground",
				selectionForegroundColor,

				"TitledBorder.titleColor",
				foregroundColor,

				"TitledBorder.border",
				new SubstanceEtchedBorder(),

				"ToggleButton.foreground",
				foregroundColor,

				"ToggleButton.disabledText",
				disabledForegroundColor,

				"ToggleButton.margin",
				new InsetsUIResource(0, 0, 0, 0),

				"ToolBar.background",
				backgroundColor,

				"ToolBar.border",
				new BorderUIResource(new SubstanceToolBarBorder()),

				"ToolBar.isRollover",
				Boolean.TRUE,

				"ToolBar.foreground",
				foregroundColor,

				"ToolBarSeparator.background",
				backgroundColor,

				"ToolBarSeparator.foreground",
				lineBwColor,

				"ToolBar.separatorSize",
				null,

				"ToolTip.border",
				new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return new LineBorder(foregroundColor);
					}
				},

				"ToolTip.borderInactive",
				regularMarginBorder,

				"ToolTip.background",
				backgroundColor,

				"ToolTip.backgroundInactive",
				new ColorUIResource(getDefaultColorScheme(this.kind)
						.getExtraLightColor()),

				"ToolTip.foreground", foregroundColor,

				"ToolTip.foregroundInactive", disabledForegroundColor,

				"Tree.closedIcon", emptyIcon,
				// new UIDefaults.LazyValue() {
				// public Object createValue(UIDefaults table) {
				// return new IconUIResource(
				// new ImageIcon(
				// SubstanceImageCreator
				// .getThemeImage(
				// SubstanceCoreUtilities
				// .getIcon("resource/folder_go.png"),
				// SubstanceLookAndFeel
				// .getTheme(),
				// true)));
				// }
				// },

				"Tree.collapsedIcon", new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return new IconUIResource(
								new ImageIcon(
										SubstanceImageCreator
												.getTreeIcon(
														SubstanceSizeUtils
																.getControlFontSize(),
														null,
														SubstanceTheme.this
																.getDefaultTheme()
																.getColorScheme(),
														SubstanceCoreUtilities
																.isThemeDark(SubstanceTheme.this),
														true)));
					}
				},

				"Tree.expandedIcon", new UIDefaults.LazyValue() {
					public Object createValue(UIDefaults table) {
						return new IconUIResource(
								new ImageIcon(
										SubstanceImageCreator
												.getTreeIcon(
														SubstanceSizeUtils
																.getControlFontSize(),
														null,
														SubstanceTheme.this
																.getDefaultTheme()
																.getColorScheme(),
														SubstanceCoreUtilities
																.isThemeDark(SubstanceTheme.this),
														false)));
					}
				},

				"Tree.leafIcon", emptyIcon,
				// new UIDefaults.LazyValue() {
				// public Object createValue(UIDefaults table) {
				// return new IconUIResource(
				// new ImageIcon(
				// SubstanceImageCreator
				// .getThemeImage(
				// SubstanceCoreUtilities
				// .getIcon("resource/page.png"),
				// SubstanceLookAndFeel
				// .getTheme(),
				// true)));
				// }
				// },

				"Tree.openIcon", emptyIcon,
				// new UIDefaults.LazyValue() {
				// public Object createValue(UIDefaults table) {
				// return new IconUIResource(
				// new ImageIcon(
				// SubstanceImageCreator
				// .getThemeImage(
				// SubstanceCoreUtilities
				// .getIcon("resource/folder.png"),
				// SubstanceLookAndFeel
				// .getTheme(),
				// true)));
				// // new ImageIcon(
				// // SubstanceImageCreator
				// // .getTreeNodeIcon(
				// // SubstanceCoreUtilities
				// // .isThemeDark(SubstanceTheme.this),
				// // SubstanceConstants.TreeIcon.OPENED)));
				// }
				// },

				"Tree.background", backgroundColor,

				"Tree.selectionBackground", backgroundLightColor,

				"Tree.foreground", foregroundColor,

				"Tree.hash", lineColorDefault,

				"Tree.rowHeight", new Integer(0),

				"Tree.selectionBorderColor", lineColor,

				"Tree.selectionForeground", selectionCellForegroundColor,

				"Tree.textBackground", backgroundColor,

				"Tree.textForeground", foregroundColor,

				"Viewport.background", backgroundColor,

				"Viewport.foreground", foregroundColor,

		};
		table.putDefaults(defaults);
	}

	/**
	 * Returns the foreground color for this theme.
	 * 
	 * @return The foreground color for this theme.
	 * 
	 * @since version 4.0
	 */
	public ColorUIResource getForegroundColor() {
		return new ColorUIResource(SubstanceColorUtilities
				.getForegroundColor(this));
	}

	/**
	 * Returns the foreground color for selected cells for this theme.
	 * 
	 * @return The foreground color for selected cells for this theme.
	 * 
	 * @since version 4.0
	 */
	public Color getSelectionCellForegroundColor() {
		return SubstanceColorUtilities.getForegroundColor(this
				.getHighlightTheme(null, ComponentState.SELECTED));
	}

	/**
	 * Returns the selection foreground color for this theme.
	 * 
	 * @return The selection foreground color for this theme.
	 * 
	 * @since version 4.0
	 */
	public Color getSelectionForegroundColor() {
		return SubstanceColorUtilities.getSelectionForegroundColor(this);
	}

	/**
	 * Returns the selection background color for this theme.
	 * 
	 * @return The selection background color for this theme.
	 * 
	 * @since version 4.0
	 */
	public Color getSelectionBackgroundColor() {
		return SubstanceColorUtilities.getSelectionBackgroundColor(this);
	}

	/**
	 * Returns the line color for this theme.
	 * 
	 * @return The line color for this theme.
	 * 
	 * @since version 4.0
	 */
	public Color getLineColor() {
		return SubstanceColorUtilities.getLineColor(this);
	}

	/**
	 * Returns the line color for this theme.
	 * 
	 * @return The line color for this theme.
	 * 
	 * @since version 4.0
	 */
	public Color getLineColorDefault() {
		return SubstanceColorUtilities.getLineColor(this.getDefaultTheme());
	}

	/**
	 * Returns the disabled background color for this theme.
	 * 
	 * @return The disabled background color for this theme.
	 * 
	 * @since version 4.0
	 */
	public Color getDisabledBackgroundColor() {
		return SubstanceColorUtilities.getInterpolatedColor(this
				.getBackgroundColor(), this.getLightBackgroundColor(), 0.5);
	}

	/**
	 * Returns the disabled foreground color for this theme.
	 * 
	 * @return The disabled foreground color for this theme.
	 * 
	 * @since version 4.0
	 */
	public Color getDisabledForegroundColor() {
		return SubstanceColorUtilities.getDisabledForegroundColor(this);
	}

	/**
	 * Returns the light background color for this theme.
	 * 
	 * @return The light background color for this theme.
	 * 
	 * @since version 4.0
	 */
	public Color getLightBackgroundColor() {
		return SubstanceColorUtilities.getInterpolatedColor(
				getDefaultColorScheme().getUltraLightColor(),
				getDefaultColorScheme().getExtraLightColor(), 0.8);
	}

	/**
	 * Returns the background color for this theme.
	 * 
	 * @return The background color for this theme.
	 * 
	 * @since version 4.0
	 */
	public Color getBackgroundColor() {
		return SubstanceColorUtilities.getBackgroundColor(this);
	}

	/**
	 * Returns the kind of <code>this</code> theme. This method is part of
	 * officially supported API.
	 * 
	 * @return The kind of <code>this</code> theme.
	 */
	public ThemeKind getKind() {
		return this.kind;
	}

	/**
	 * Returns the default scheme that corresponds to the specified theme kind.
	 * This method is <b>for internal use only</b>.
	 * 
	 * @param themeKind
	 *            Theme kind.
	 * @return Default scheme that corresponds to the specified theme kind.
	 */
	public static ColorScheme getDefaultColorScheme(ThemeKind themeKind) {
		if (themeKind == ThemeKind.DARK)
			return DARK_METALLIC_SCHEME;
		if (themeKind == ThemeKind.INVERTED)
			return INVERTED_LIGHT_GRAY_SCHEME;
		if (themeKind == ThemeKind.NEGATED)
			return NEGATED_LIGHT_GRAY_SCHEME;
		return METALLIC_SCHEME;
	}

	/**
	 * Returns the default theme that corresponds to the specified theme kind.
	 * This method is <b>for internal use only</b>.
	 * 
	 * @param themeKind
	 *            Theme kind.
	 * @return Default theme that corresponds to the specified theme kind.
	 */
	private static SubstanceTheme getDefaultTheme(ThemeKind themeKind) {
		if (themeKind == ThemeKind.DARK)
			return DARK_METALLIC;
		if (themeKind == ThemeKind.INVERTED)
			return INVERTED_LIGHT_GRAY;
		if (themeKind == ThemeKind.NEGATED)
			return NEGATED_LIGHT_GRAY;
		return METALLIC;
	}

	/**
	 * Returns the disabled scheme that corresponds to the specified theme kind.
	 * This method is <b>for internal use only</b>.
	 * 
	 * @param themeKind
	 *            Theme kind.
	 * @return Disabled scheme that corresponds to the specified theme kind.
	 */
	public static ColorScheme getDisabledColorScheme(ThemeKind themeKind) {
		if (themeKind == ThemeKind.DARK)
			return DARK_GRAY_SCHEME;
		if (themeKind == ThemeKind.INVERTED)
			return INVERTED_METALLIC_SCHEME;
		if (themeKind == ThemeKind.NEGATED)
			return NEGATED_METALLIC_SCHEME;
		return LIGHT_GRAY_SCHEME;
	}

	/**
	 * Returns the disabled theme that corresponds to the specified theme kind.
	 * This method is <b>for internal use only</b>.
	 * 
	 * @param themeKind
	 *            Theme kind.
	 * @return Disabled theme that corresponds to the specified theme kind.
	 */
	private static SubstanceTheme getDisabledTheme(ThemeKind themeKind) {
		if (themeKind == ThemeKind.DARK)
			return DARK_GRAY;
		if (themeKind == ThemeKind.INVERTED)
			return INVERTED_METALLIC;
		if (themeKind == ThemeKind.NEGATED)
			return NEGATED_METALLIC;
		return LIGHT_GRAY;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.SubstanceTrait#getDisplayName()
	 */
	public String getDisplayName() {
		return this.displayName;
	}

	/**
	 * Returns the theme that matches the specified display name. This method is
	 * <b>for internal use only</b>.
	 * 
	 * @param displayName
	 *            Theme display name.
	 * @return Theme that matches the specified display name.
	 */
	public static SubstanceTheme getTheme(String displayName) {
		// SubstanceTheme result = allThemes.get(displayName);
		// if (result != null)
		// return result;
		//
		// try fetching from the available themes
		ThemeInfo themeInfo = SubstanceLookAndFeel.getAllThemes().get(
				displayName);
		if (themeInfo != null)
			return SubstanceTheme.createInstance(themeInfo);

		throw new IllegalStateException("Theme with display name '"
				+ displayName + "' can not be found");
	}

	/**
	 * Returns the default color scheme that corresponds to <code>this</code>
	 * theme. This method is <b>for internal use only</b>.
	 * 
	 * @return The default color scheme that corresponds to <code>this</code>
	 *         theme.
	 */
	public ColorScheme getDefaultColorScheme() {
		return this.getDefaultTheme().getColorScheme();
	}

	/**
	 * Returns the disabled color scheme that corresponds to <code>this</code>
	 * theme. This method is <b>for internal use only</b>.
	 * 
	 * @return The disabled color scheme that corresponds to <code>this</code>
	 *         theme.
	 */
	public ColorScheme getDisabledColorScheme() {
		return this.getDisabledTheme().getColorScheme();
	}

	/**
	 * Returns the theme that corresponds to the specified component state.
	 * 
	 * @param comp
	 *            Component.
	 * @param componentState
	 *            Component state.
	 * @return The theme that corresponds to the specified component state.
	 * 
	 * @since version 4.0
	 */
	public SubstanceTheme getTheme(Component comp, ComponentState componentState) {
		if (componentState.isKindActive(FadeKind.PRESS)) {
			if (this.pressedTheme == null) {
				this.pressedTheme = this.getActiveTheme().shade(0.2).saturate(
						0.1);
			}
			return this.pressedTheme;
		}
		switch (componentState.getColorSchemeKind()) {
		case CURRENT:
			return this.getActiveTheme();
		case REGULAR:
			return this.getDefaultTheme();
		case DISABLED:
			return this.getDisabledTheme();
		}
		return this;
	}

	/**
	 * Returns the highlight theme that corresponds to the specified component
	 * state.
	 * 
	 * @param comp
	 *            Component.
	 * @param componentState
	 *            Component state.
	 * @return The highlight theme that corresponds to the specified component
	 *         state.
	 * 
	 * @since version 4.0
	 */
	public SubstanceTheme getHighlightTheme(Component comp,
			ComponentState componentState) {
		switch (componentState.getColorSchemeKind()) {
		case CURRENT:
			return this.getActiveTheme();
		case REGULAR:
			return this.getDefaultTheme();
		case DISABLED:
			return this.getDisabledTheme();
		}
		return this;
	}

	/**
	 * Returns the alpha channel that is applied on highlight theme when a
	 * component is in the specified state.
	 * 
	 * @param comp
	 *            Component.
	 * @param componentState
	 *            Component state.
	 * @return Alpha channel that is applied on highlight theme when a component
	 *         is in the specified state.
	 */
	public float getHighlightThemeAlpha(Component comp,
			ComponentState componentState) {
		boolean isRollover = componentState.isKindActive(FadeKind.ROLLOVER);
		boolean isSelected = componentState.isKindActive(FadeKind.SELECTION);
		boolean isArmed = componentState.isKindActive(FadeKind.ARM);

		if (isRollover && isSelected)
			return 0.9f;
		if (isRollover && isArmed)
			return 0.8f;
		if (isSelected)
			return 0.7f;
		if (isArmed)
			return 0.6f;
		if (isRollover)
			return 0.4f;
		return 0.0f;
	}

	/**
	 * Returns the alpha channel that is applied on theme when a component is in
	 * the specified state.
	 * 
	 * @param comp
	 *            Component.
	 * @param componentState
	 *            Component state.
	 * @return Alpha channel that is applied on theme when a component is in the
	 *         specified state.
	 */
	public float getThemeAlpha(Component comp, ComponentState componentState) {
		return 1.0f;
	}

	/**
	 * Returns the active theme that corresponds to <code>this</code> theme.
	 * This method is <b>for internal use only</b>.
	 * 
	 * @return The active theme that corresponds to <code>this</code> theme.
	 */
	public SubstanceTheme getActiveTheme() {
		return this;
	}

	/**
	 * Returns the default theme that corresponds to <code>this</code> theme.
	 * This method is <b>for internal use only</b>.
	 * 
	 * @return The default theme that corresponds to <code>this</code> theme.
	 */
	public SubstanceTheme getDefaultTheme() {
		// if (this.kind == ThemeKind.MIXED)
		// return getDefaultTheme(((SubstanceMixBiTheme) this)
		// .getOriginalFirstTheme().getKind());
		return getDefaultTheme(this.kind);
	}

	/**
	 * Returns the disabled theme that corresponds to <code>this</code> theme.
	 * This method is <b>for internal use only</b>.
	 * 
	 * @return The disabled theme that corresponds to <code>this</code> theme.
	 */
	public SubstanceTheme getDisabledTheme() {
		// if (this.kind == ThemeKind.MIXED)
		// return getDisabledTheme(((SubstanceMixBiTheme) this)
		// .getOriginalFirstTheme().getKind());
		return getDisabledTheme(this.kind);
	}

	/**
	 * Returns indication whether <code>this</code> theme is dark. This method
	 * is part of officially supported API.
	 * 
	 * @return <code>true</code> if <code>this</code> theme is dark,
	 *         <code>false</code> otherwise.
	 * @deprecated Use {@link #getKind()} instead.
	 */
	@Deprecated
	public boolean isDark() {
		return SubstanceCoreUtilities.isThemeDark(this);
	}

	/**
	 * Creates a new theme based on the specified parameters. Note that the
	 * preferred way is to create a plugin. This method is part of officially
	 * supported API.
	 * 
	 * @param scheme
	 *            Color scheme object.
	 * @param themeKind
	 *            Theme kind.
	 * @param displayName
	 *            Display name for the new theme.
	 * @return New theme object.
	 * @deprecated since 3.1 Use
	 *             {@link SubstanceTheme#SubstanceTheme(ColorScheme, String, org.jvnet.substance.theme.SubstanceTheme.ThemeKind)}
	 *             constructor instead.
	 */
	@Deprecated
	public static SubstanceTheme addUserDefined(ColorScheme scheme,
			ThemeKind themeKind, String displayName) {
		SubstanceTheme userTheme = new SubstanceTheme(scheme, displayName,
				themeKind);
		return userTheme;
	}

	// /**
	// * Returns the translucent dark color of <code>this</code> theme.
	// * This method is <b>for internal use only</b>.
	// *
	// * @param alpha
	// * Alpha channel value.
	// * @return Translucent dark color of <code>this</code> theme.
	// */
	// public Color getDarkColor(int alpha) {
	// return new Color(this.getColorScheme().getDarkColor().getRed(), this
	// .getColorScheme().getDarkColor().getGreen(), this
	// .getColorScheme().getDarkColor().getBlue(), alpha);
	//
	// }
	//
	// /**
	// * Returns the translucent ultra dark color of <code>this</code> theme.
	// * This method is <b>for internal use only</b>.
	// *
	// * @param alpha
	// * Alpha channel value.
	// * @return Translucent ultra dark color of <code>this</code> theme.
	// */
	// public Color getUltraDarkColor(int alpha) {
	// return new Color(this.getColorScheme().getUltraDarkColor().getRed(), this
	// .getColorScheme().getUltraDarkColor().getGreen(), this
	// .getColorScheme().getUltraDarkColor().getBlue(), alpha);
	//
	// }
	//
	// /**
	// * Returns the translucent mid color of <code>this</code> theme.
	// * This method is <b>for internal use only</b>.
	// *
	// * @param alpha
	// * Alpha channel value.
	// * @return Translucent mid color of <code>this</code> theme.
	// */
	// public Color getMidColor(int alpha) {
	// return new Color(this.getColorScheme().getMidColor().getRed(), this
	// .getColorScheme().getMidColor().getGreen(), this
	// .getColorScheme().getMidColor().getBlue(), alpha);
	//
	// }
	//
	// /**
	// * Returns the translucent ultra light color of <code>this</code> theme.
	// * This method is <b>for internal use only</b>.
	// *
	// * @param alpha
	// * Alpha channel value.
	// * @return Translucent ultra light color of <code>this</code> theme.
	// */
	// public Color getUltraLightColor(int alpha) {
	// return new Color(this.getColorScheme().getUltraLightColor().getRed(),
	// this.getColorScheme().getUltraLightColor().getGreen(), this
	// .getColorScheme().getUltraLightColor().getBlue(), alpha);
	//
	// }
	//
	// /**
	// * Returns the translucent light color of <code>this</code> theme.
	// * This method is <b>for internal use only</b>.
	// *
	// * @param alpha
	// * Alpha channel value.
	// * @return Translucent light color of <code>this</code> theme.
	// */
	// public Color getLightColor(int alpha) {
	// return new Color(this.getColorScheme().getLightColor().getRed(),
	// this.getColorScheme().getLightColor().getGreen(), this
	// .getColorScheme().getLightColor().getBlue(), alpha);
	//
	// }
	//
	/**
	 * Returns the translucent watermark stamp color of <code>this</code>
	 * theme. This method is <b>for internal use only</b>.
	 * 
	 * @return Translucent watermark stamp color of <code>this</code> theme.
	 */
	public Color getWatermarkStampColor() {
		return SubstanceCoreUtilities.isThemeDark(this) ? SubstanceColorUtilities
				.getAlphaColor(this.getColorScheme().getUltraLightColor(), 30)
				/*
				 * this .getUltraLightColor(30)
				 */: SubstanceColorUtilities.getAlphaColor(this.getColorScheme()
						.getMidColor(), 50);
		// this.getMidColor(50));
	}

	/**
	 * Creates a theme instance based on the specified theme info object. This
	 * method is <b>for internal use only</b>.
	 * 
	 * @param themeInfo
	 *            Theme info.
	 * @return Theme instance that matches the specified theme info object.
	 */
	public static SubstanceTheme createInstance(ThemeInfo themeInfo) {
		ThemeKind themeKind = themeInfo.getThemeKind();
		if (themeKind == ThemeKind.MIXED) {
			try {
				MixedThemeInfo mixedThemeInfo = (MixedThemeInfo) themeInfo;
				String[] themeClassNames = mixedThemeInfo.getThemeClassNames();
				SubstanceTheme[] themeInstances = new SubstanceTheme[themeClassNames.length];
				for (int i = 0; i < themeClassNames.length; i++) {
					Class<?> themeClass = Class.forName(themeClassNames[i]);
					if (themeClass == null) {
						return null;
					}
					Object obj = themeClass.newInstance();
					if (obj == null) {
						return null;
					}
					if (!(obj instanceof SubstanceTheme)) {
						return null;
					}
					themeInstances[i] = (SubstanceTheme) obj;
				}

				SubstanceTheme mixedTheme = new SubstanceMixTheme(
						themeInstances);

				return mixedTheme;
			} catch (Exception exc) {
				return null;
			}
		}
		if (themeKind == ThemeKind.INVERTED) {
			try {
				Class<?> themeClass = Class.forName(themeInfo.getClassName());
				if (themeClass == null) {
					return null;
				}
				Object obj = themeClass.newInstance();
				if (obj == null) {
					return null;
				}
				if (!(obj instanceof SubstanceTheme)) {
					return null;
				}
				SubstanceTheme origTheme = (SubstanceTheme) obj;
				SubstanceTheme invertedTheme = new SubstanceInvertedTheme(
						origTheme);
				return invertedTheme;
			} catch (Exception exc) {
				return null;
			}
		}

		if (themeKind == ThemeKind.NEGATED) {
			try {
				Class<?> themeClass = Class.forName(themeInfo.getClassName());
				if (themeClass == null) {
					return null;
				}
				Object obj = themeClass.newInstance();
				if (obj == null) {
					return null;
				}
				if (!(obj instanceof SubstanceTheme)) {
					return null;
				}
				SubstanceTheme origTheme = (SubstanceTheme) obj;
				SubstanceTheme negatedTheme = new SubstanceNegatedTheme(
						origTheme);
				return negatedTheme;
			} catch (Exception exc) {
				return null;
			}
		}

		try {
			Class<?> themeClass = Class.forName(themeInfo.getClassName());
			if (themeClass == null) {
				return null;
			}
			Object obj = themeClass.newInstance();
			if (obj == null) {
				return null;
			}
			if (!(obj instanceof SubstanceTheme)) {
				return null;
			}
			return (SubstanceTheme) obj;
		} catch (Exception exc) {
			return null;
		}
	}

	/**
	 * Creates a tinted (shifted towards white) version of <code>this</code>
	 * theme. This method is part of officially supported API.
	 * 
	 * @param tintFactor
	 *            Value in 0.0...1.0 range. Larger values shift more towards
	 *            white color.
	 * @return Tinted version of <code>this</code> theme.
	 */
	public SubstanceTheme tint(double tintFactor) {
		return new SubstanceTintTheme(this, tintFactor);
	}

	/**
	 * Creates a toned (shifted towards gray) version of <code>this</code>
	 * theme. This method is part of officially supported API.
	 * 
	 * @param toneFactor
	 *            Value in 0.0...1.0 range. Larger values shift more towards
	 *            gray color.
	 * @return Toned version of <code>this</code> theme.
	 */
	public SubstanceTheme tone(double toneFactor) {
		return new SubstanceToneTheme(this, toneFactor);
	}

	/**
	 * Creates a shaded (shifted towards black) version of <code>this</code>
	 * theme. This method is part of officially supported API.
	 * 
	 * @param shadeFactor
	 *            Value in 0.0...1.0 range. Larger values shift more towards
	 *            black color.
	 * @return Shaded version of <code>this</code> theme.
	 */
	public SubstanceTheme shade(double shadeFactor) {
		return new SubstanceShadeTheme(this, shadeFactor);
	}

	/**
	 * Creates a saturated or desaturated version of <code>this</code> theme.
	 * The value and brightness stay the same. This method is part of officially
	 * supported API.
	 * 
	 * @param saturateFactor
	 *            Value in -1.0...1.0 range. Positive values create more
	 *            saturated colors. Negative values create more desaturated
	 *            colors.
	 * @return Saturated version of <code>this</code> theme.
	 */
	public SubstanceTheme saturate(double saturateFactor) {
		return this.saturate(saturateFactor, true);
	}

	/**
	 * Creates a saturated or desaturated version of <code>this</code> theme.
	 * The value and brightness stay the same. This method is part of officially
	 * supported API.
	 * 
	 * @param saturateFactor
	 *            Value in -1.0...1.0 range. Positive values create more
	 *            saturated colors. Negative values create more desaturated
	 *            colors.
	 * @param toSaturateEverything
	 *            If <code>true</code>, all visual components of
	 *            <code>this</code> theme will be saturated. If
	 *            <code>false</code>, only the {@link #getActiveTheme()} will
	 *            be saturated.
	 * @return Saturated version of <code>this</code> theme.
	 */
	public SubstanceTheme saturate(double saturateFactor,
			boolean toSaturateEverything) {
		return new SubstanceSaturatedTheme(this, saturateFactor,
				toSaturateEverything);
	}

	/**
	 * Creates an inverted version of <code>this</code> theme. This method is
	 * part of officially supported API.
	 * 
	 * @return Inverted version of <code>this</code> theme.
	 */
	public SubstanceTheme invert() {
		return new SubstanceInvertedTheme(this);
	}

	/**
	 * Creates a negated version of <code>this</code> theme. This method is
	 * part of officially supported API.
	 * 
	 * @return Negated version of <code>this</code> theme.
	 */
	public SubstanceTheme negate() {
		return new SubstanceNegatedTheme(this);
	}

	/**
	 * Creates a hue-shifted (in HSB space) version of <code>this</code>
	 * theme. This method is part of officially supported API.
	 * 
	 * @param hueShiftFactor
	 *            Value in -1.0...1.0 range.
	 * @return Hue-shifted version of <code>this</code> theme.
	 */
	public SubstanceTheme hueShift(double hueShiftFactor) {
		return new SubstanceHueShiftTheme(this, hueShiftFactor);
	}

	/**
	 * Creates a new version of <code>this</code> theme that simulates the
	 * <code>protanopia</code> color blindness.
	 * 
	 * @return New version of <code>this</code> theme that simulates the
	 *         <code>protanopia</code> color blindness.
	 */
	public SubstanceTheme protanopia() {
		return new SubstanceProtanopiaTheme(this);
	}

	/**
	 * Creates a new version of <code>this</code> theme that simulates the
	 * <code>tritanopia</code> color blindness.
	 * 
	 * @return New version of <code>this</code> theme that simulates the
	 *         <code>tritanopia</code> color blindness.
	 */
	public SubstanceTheme tritanopia() {
		return new SubstanceTritanopiaTheme(this);
	}

	/**
	 * Creates a new version of <code>this</code> theme that simulates the
	 * <code>deuteranopia</code> color blindness.
	 * 
	 * @return New version of <code>this</code> theme that simulates the
	 *         <code>deuteranopia</code> color blindness.
	 */
	public SubstanceTheme deuteranopia() {
		return new SubstanceDeuteranopiaTheme(this);
	}

	/**
	 * Returns the theme for title panes of active frames. This method is <b>for
	 * internal use only</b>.
	 * 
	 * @return Theme for title panes of active frames
	 */
	public SubstanceTheme getActiveTitlePaneTheme() {
		return this;
	}

	/**
	 * Returns the theme for title panes of inactive frames. This method is
	 * <b>for internal use only</b>.
	 * 
	 * @return Theme for title panes of inactive frames
	 */
	public SubstanceTheme getDefaultTitlePaneTheme() {
		return this;
	}

	/**
	 * Returns the first theme. This method is <b>for internal use only</b>.
	 * 
	 * @return First theme.
	 */
	public SubstanceTheme getFirstTheme() {
		return this;
	}

	/**
	 * Returns the second theme. This method is <b>for internal use only</b>.
	 * 
	 * @return Second theme.
	 */
	public SubstanceTheme getSecondTheme() {
		return this;
	}

	/**
	 * Returns the theme for watermark painting.
	 * 
	 * @return Theme for watermark painting.
	 */
	public SubstanceTheme getWatermarkTheme() {
		if (SubstanceCoreUtilities.isThemeDark(this))
			return this.getActiveTheme();
		return this.getDefaultTheme();
	}

	/**
	 * Returns the theme for border painting.
	 * 
	 * @return Theme for border painting.
	 * @since version 4.0
	 */
	public SubstanceTheme getBorderTheme() {
		return this;
	}

	// /**
	// * Returns the theme for painting the highlight background of selected
	// * elements, such as tree / table / list renderers, selections in text
	// * controls and so on.
	// *
	// * @return Theme for painting the highlight background of selected
	// elements,
	// * such as tree / table / list renderers, selections in text
	// * controls and so on.
	// */
	// public SubstanceTheme getHighlightBackgroundTheme() {
	// return this.getActiveTheme();
	// }
	//
	/**
	 * Returns the painter for painting the non-active controls.
	 * 
	 * @return Painter for painting the non-active controls.
	 */
	public SubstanceGradientPainter getNonActivePainter() {
		return new StandardGradientPainter();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return this.getDisplayName();
	}

	//
	// public boolean toPaintActive(Component component) {
	// if (component == null)
	// return false;
	//
	// Class<?> clazz = component.getClass();
	// for (Class<?> registered : activeControls) {
	// if (registered.isAssignableFrom(clazz))
	// return true;
	// }
	// return false;
	// }
	//
	// public void addPaintAsActive(Class clazz) {
	// this.activeControls.add(clazz);
	// }

	/**
	 * Returns the start of fade effect on the selected tabs in
	 * {@link JTabbedPane}s. This value can be used to create XP-like "headers"
	 * on the selected tabs.
	 * 
	 * @return The start of fade effect on the selected tabs in
	 *         {@link JTabbedPane}s.
	 * @see #getSelectedTabFadeEnd()
	 */
	public double getSelectedTabFadeStart() {
		return 0.1;
	}

	/**
	 * Returns the end of fade effect on the selected tabs in
	 * {@link JTabbedPane}s. This value can be used to create XP-like "headers"
	 * on the selected tabs.
	 * 
	 * @return The end of fade effect on the selected tabs in
	 *         {@link JTabbedPane}s.
	 * @see #getSelectedTabFadeStart()
	 */
	public double getSelectedTabFadeEnd() {
		return 0.3;
	}
}
