/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.title;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.util.Map;

import org.jvnet.lafwidget.layout.TransitionLayout;
import org.jvnet.substance.SubstanceImageCreator;
import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.painter.ClassicGradientPainter;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.SoftHashMap;
import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * Title painter that paints a classic gradient. This class is part of
 * officially supported API.
 * 
 * @author Kirill Grouchnikov
 */
public class ClassicTitlePainter implements SubstanceTitlePainter {
	/**
	 * The display name for the title painters of this class.
	 */
	public static final String DISPLAY_NAME = "Classic";

	/**
	 * Cache for small objects.
	 */
	protected static Map<String, BufferedImage> smallImageCache = new SoftHashMap<String, BufferedImage>();

	/**
	 * Single gradient painter instance.
	 */
	protected ClassicGradientPainter painter;

	/**
	 * Creates new classic title painter.
	 */
	public ClassicTitlePainter() {
		this.painter = new ClassicGradientPainter();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.SubstanceGradientPainter#getDisplayName()
	 */
	public String getDisplayName() {
		return DISPLAY_NAME;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.title.SubstanceTitlePainter#paintTitleBackground(java.awt.Graphics2D,
	 *      java.awt.Component, int, int, int, int,
	 *      org.jvnet.substance.theme.SubstanceTheme, float)
	 */
	public void paintTitleBackground(Graphics2D graphics, Component comp,
			int width, int height, int leftTitleX, int rightTitleX,
			SubstanceTheme theme, float borderAlpha) {

		int size = Math.max(width, height);
		if (size < 150) {
			synchronized (this) {
				String key = width + "*" + height + "*" + leftTitleX + "*"
						+ rightTitleX + "*" + theme.getDisplayName() + "*"
						+ borderAlpha;
				BufferedImage result = smallImageCache.get(key);
				if (result == null) {
					result = SubstanceCoreUtilities
							.getBlankImage(width, height);
					this.internalPaint((Graphics2D) result.getGraphics(),
							width, height, leftTitleX, rightTitleX, theme,
							borderAlpha);
					smallImageCache.put(key, result);
				}
				graphics.drawImage(result, 0, 0, null);
				return;
			}
		}

		this.internalPaint(graphics, width, height, leftTitleX, rightTitleX,
				theme, borderAlpha);
	}

	/**
	 * Paints the title background.
	 * 
	 * @param graphics
	 *            Graphics context.
	 * @param width
	 *            Width.
	 * @param height
	 *            Height.
	 * @param leftTitleX
	 *            Title start poisition.
	 * @param rightTitleX
	 *            Title end position.
	 * @param theme
	 *            Substance theme for painting the title packground.
	 * @param borderAlpha
	 *            Border alpha.
	 */
	protected void internalPaint(Graphics2D graphics, int width, int height,
			int leftTitleX, int rightTitleX, SubstanceTheme theme,
			float borderAlpha) {
		graphics.setColor(theme.getActiveTitlePaneTheme().getColorScheme()
				.getMidColor());
		graphics.drawImage(this.painter.getContourBackground(width + 6,
				height + 6, new Rectangle(width + 6, height + 6), false, theme
						.getActiveTitlePaneTheme().getColorScheme(), theme
						.getActiveTitlePaneTheme().getColorScheme(), 0, false,
				false), -3, -3, null);
		if (borderAlpha > 0.0f) {
			Graphics2D g2d = (Graphics2D) graphics.create();
			g2d.setComposite(TransitionLayout.getAlphaComposite(null,
					borderAlpha, graphics));
			SubstanceImageCreator.paintBorder(null, g2d, 0, 0, width, height,
					0, theme.getActiveTitlePaneTheme());
			g2d.dispose();
		}
	}

	/**
	 * Resets image maps (used when setting new theme).
	 * 
	 * @see SubstanceLookAndFeel#setCurrentTheme(String)
	 * @see SubstanceLookAndFeel#setCurrentTheme(SubstanceTheme)
	 */
	public static synchronized void reset() {
		smallImageCache.clear();
	}
}
