/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.utils.icon;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.util.Map;

import javax.swing.*;

import org.jvnet.lafwidget.animation.FadeKind;
import org.jvnet.lafwidget.animation.FadeState;
import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.*;

/**
 * Icon with transition-aware capabilities. Has a delegate that does the actual
 * painting based on the transition themes. This class is used heavily on
 * Substance-provided icons, such as title pane button icons, arrow icons on
 * scroll bars and combos etc.
 * 
 * @author Kirill Grouchnikov
 */
public class TransitionAwareIcon implements Icon {
	/**
	 * The delegate needs to implement the method in this interface based on the
	 * provided theme. The theme is computed based on the transitions that are
	 * happening on the associated button.
	 * 
	 * @author Kirill Grouchnikov
	 */
	public static interface Delegate {
		/**
		 * Returns the icon that matches the specified theme.
		 * 
		 * @param theme
		 *            Theme.
		 * @return Icon that matches the specified theme.
		 */
		public Icon getThemeIcon(SubstanceTheme theme);
	}

	/**
	 * The associated button.
	 */
	private AbstractButton button;

	/**
	 * Delegate to compute the actual icons.
	 */
	private Delegate delegate;

	/**
	 * Icon cache to speed up the subsequent icon painting. The basic assumption
	 * is that the {@link #delegate} returns an icon that paints the same for
	 * the same parameters.
	 */
	private Map<String, Icon> iconMap;

	/**
	 * Creates a new transition-aware icon.
	 * 
	 * @param button
	 *            Associated button.
	 * @param delegate
	 *            Delegate to compute the actual icons.
	 */
	public TransitionAwareIcon(AbstractButton button, Delegate delegate) {
		this.button = button;
		this.delegate = delegate;
		this.iconMap = new SoftHashMap<String, Icon>();
	}

	/**
	 * Returns the current icon to paint.
	 * 
	 * @return Icon to paint.
	 */
	private Icon getIconToPaint() {
		ComponentState currState = ComponentState.getState(button.getModel(),
				button);
		ComponentState prevState = SubstanceCoreUtilities
				.getPrevComponentState(button);
		if (!currState.isKindActive(FadeKind.ENABLE))
			prevState = currState;
		float cyclePos = currState.getCycleCount();
		SubstanceTheme currTheme = SubstanceCoreUtilities.getTheme(button,
				currState, true, true);
		if ((button instanceof SubstanceTitleButton)
				&& (currState == ComponentState.DEFAULT)) {
			currTheme = SubstanceLookAndFeel.getTheme()
					.getActiveTitlePaneTheme();
		}

		SubstanceTheme prevTheme = currTheme;

		FadeState fadeState = SubstanceFadeUtilities.getFadeState(button,
				FadeKind.ROLLOVER, FadeKind.SELECTION, FadeKind.PRESS,
				FadeKind.ARM);
		if (fadeState != null) {
			prevTheme = SubstanceCoreUtilities.getTheme(button, prevState,
					true, true);
			if ((button instanceof SubstanceTitleButton)
					&& (prevState == ComponentState.DEFAULT)) {
				prevTheme = SubstanceLookAndFeel.getTheme()
						.getActiveTitlePaneTheme();
			}
			cyclePos = fadeState.getFadePosition();
			if (!fadeState.isFadingIn())
				cyclePos = 10 - cyclePos;
		}
		float currAlpha = currTheme.getThemeAlpha(this.button, currState);
		float prevAlpha = prevTheme.getThemeAlpha(this.button, prevState);

		String key = currTheme.getDisplayName() + ":"
				+ prevTheme.getDisplayName() + ":" + currAlpha + ":"
				+ prevAlpha + ":" + cyclePos;
		// System.out.println(key);
		if (!this.iconMap.containsKey(key)) {
			Icon icon = this.delegate.getThemeIcon(currTheme);
			Icon prevIcon = this.delegate.getThemeIcon(prevTheme);

			BufferedImage temp = SubstanceCoreUtilities.getBlankImage(icon
					.getIconWidth(), icon.getIconHeight());
			Graphics2D g2d = temp.createGraphics();

			if (currTheme == prevTheme) {
				// same theme - can paint just the current icon, no matter
				// what the cycle position is.
				g2d.setComposite(AlphaComposite.getInstance(
						AlphaComposite.SRC_OVER, currAlpha));
				icon.paintIcon(this.button, g2d, 0, 0);
			} else {
				// make optimizations for limit values of the cycle position.
				if (cyclePos < 10.0f) {
					g2d.setComposite(AlphaComposite.getInstance(
							AlphaComposite.SRC_OVER, prevAlpha));
					prevIcon.paintIcon(this.button, g2d, 0, 0);
				}
				if (cyclePos > 0.0f) {
					g2d.setComposite(AlphaComposite.getInstance(
							AlphaComposite.SRC_OVER, currAlpha * cyclePos
									/ 10.0f));
					icon.paintIcon(this.button, g2d, 0, 0);
				}
			}

			this.iconMap.put(key, new ImageIcon(temp));
			g2d.dispose();
		}

		return this.iconMap.get(key);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.Icon#getIconHeight()
	 */
	public int getIconHeight() {
		return this.getIconToPaint().getIconHeight();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.Icon#getIconWidth()
	 */
	public int getIconWidth() {
		return this.getIconToPaint().getIconWidth();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.Icon#paintIcon(java.awt.Component, java.awt.Graphics,
	 *      int, int)
	 */
	public void paintIcon(Component c, Graphics g, int x, int y) {
		this.getIconToPaint().paintIcon(c, g, x, y);
	}
}
